package com.elitesland.tw.tw5crm.server.tenant.dao;

import com.elitesland.tw.tw5crm.api.tenant.payload.TwTenantPayload;
import com.elitesland.tw.tw5crm.api.tenant.query.TwTenantQuery;
import com.elitesland.tw.tw5crm.api.tenant.vo.TwTenantVO;
import com.elitesland.tw.tw5crm.server.tenant.entity.TwTenantDO;
import com.elitesland.tw.tw5crm.server.tenant.entity.QTwTenantDO;
import com.elitesland.tw.tw5crm.server.tenant.repo.TwTenantRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * Tw租户管理
 *
 * @author duwh
 * @date 2023-04-06
 */
@Repository
@RequiredArgsConstructor
public class TwTenantDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final TwTenantRepo repo;
    private final QTwTenantDO qdo = QTwTenantDO.twTenantDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<TwTenantVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(TwTenantVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 租户编码
                qdo.tenantCode,
                // 租户名称
                qdo.tenantName,
                // 管理员的用户ID
                qdo.adminUserId,
                // 租户类型
                qdo.tenantType,
                // 是否初始化
                qdo.inited,
                // 是否启用
                qdo.enabled,
                // 联系人
                qdo.linkman,
                // 联系电话
                qdo.contactNumber,
                // 联系地址
                qdo.address,
                // 租户数据隔离
                qdo.tenantIsolation,
                // 数据库是否已初始化
                qdo.dbInitialized,
                // 租户数据隔离数据源
                qdo.databaseSourceId,
                // schema名称，如果租户数据隔离通过schema
                qdo.schemaName,
                // 域名地址
                qdo.tenantDomain,
                // 租户自定义域名
                qdo.customDomain,
                // 管理员登录账号
                qdo.adminAccount,
                // 基础数据同步记录ID
                qdo.baseDataSyncId,
                // 所属行业，[UDC]COM:INDUSTRY
                qdo.industry,
                // 所属客户，[UDC]SYS:CUSTOMER
                qdo.customer,
                // 平台租户主键
                qdo.sysTenantId
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<TwTenantVO> getJpaQueryWhere(TwTenantQuery query) {
        JPAQuery<TwTenantVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(TwTenantQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(TwTenantQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 租户编码 精确 */
        if (!ObjectUtils.isEmpty(query.getTenantCode())) {
            list.add(qdo.tenantCode.eq(query.getTenantCode()));
        }
        /** 租户名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getTenantName())) {
             list.add(qdo.tenantName.like(SqlUtil.toSqlLikeString(query.getTenantName())));
        }
        /** 管理员的用户ID 精确 */
        if (!ObjectUtils.isEmpty(query.getAdminUserId())) {
            list.add(qdo.adminUserId.eq(query.getAdminUserId()));
        }
        /** 租户类型 精确 */
        if (!ObjectUtils.isEmpty(query.getTenantType())) {
            list.add(qdo.tenantType.eq(query.getTenantType()));
        }
        /** 是否初始化 精确 */
        if (!ObjectUtils.isEmpty(query.getInited())) {
            list.add(qdo.inited.eq(query.getInited()));
        }
        /** 是否启用 精确 */
        if (!ObjectUtils.isEmpty(query.getEnabled())) {
            list.add(qdo.enabled.eq(query.getEnabled()));
        }
        /** 联系人 模糊 */
        if (!ObjectUtils.isEmpty(query.getLinkman())) {
             list.add(qdo.linkman.like(SqlUtil.toSqlLikeString(query.getLinkman())));
        }
        /** 联系电话 模糊 */
        if (!ObjectUtils.isEmpty(query.getContactNumber())) {
             list.add(qdo.contactNumber.like(SqlUtil.toSqlLikeString(query.getContactNumber())));
        }
        /** 联系地址 模糊 */
        if (!ObjectUtils.isEmpty(query.getAddress())) {
             list.add(qdo.address.like(SqlUtil.toSqlLikeString(query.getAddress())));
        }
        /** 租户数据隔离 精确 */
        if (!ObjectUtils.isEmpty(query.getTenantIsolation())) {
            list.add(qdo.tenantIsolation.eq(query.getTenantIsolation()));
        }
        /** 数据库是否已初始化 精确 */
        if (!ObjectUtils.isEmpty(query.getDbInitialized())) {
            list.add(qdo.dbInitialized.eq(query.getDbInitialized()));
        }
        /** 租户数据隔离数据源 精确 */
        if (!ObjectUtils.isEmpty(query.getDatabaseSourceId())) {
            list.add(qdo.databaseSourceId.eq(query.getDatabaseSourceId()));
        }
        /** schema名称，如果租户数据隔离通过schema 精确 */
        if (!ObjectUtils.isEmpty(query.getSchemaName())) {
            list.add(qdo.schemaName.eq(query.getSchemaName()));
        }
        /** 域名地址 模糊 */
        if (!ObjectUtils.isEmpty(query.getTenantDomain())) {
             list.add(qdo.tenantDomain.like(SqlUtil.toSqlLikeString(query.getTenantDomain())));
        }
        /** 租户自定义域名 模糊 */
        if (!ObjectUtils.isEmpty(query.getCustomDomain())) {
             list.add(qdo.customDomain.like(SqlUtil.toSqlLikeString(query.getCustomDomain())));
        }
        /** 管理员登录账号 精确 */
        if (!ObjectUtils.isEmpty(query.getAdminAccount())) {
            list.add(qdo.adminAccount.eq(query.getAdminAccount()));
        }
        /** 基础数据同步记录ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBaseDataSyncId())) {
            list.add(qdo.baseDataSyncId.eq(query.getBaseDataSyncId()));
        }
        /** 所属行业，[UDC]COM:INDUSTRY 精确 */
        if (!ObjectUtils.isEmpty(query.getIndustry())) {
            list.add(qdo.industry.eq(query.getIndustry()));
        }
        /** 所属客户，[UDC]SYS:CUSTOMER 精确 */
        if (!ObjectUtils.isEmpty(query.getCustomer())) {
            list.add(qdo.customer.eq(query.getCustomer()));
        }
        /** 平台租户主键 精确 */
        if (!ObjectUtils.isEmpty(query.getSysTenantId())) {
            list.add(qdo.sysTenantId.eq(query.getSysTenantId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public TwTenantVO queryByKey(Long id) {
        JPAQuery<TwTenantVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TwTenantVO> queryListDynamic(TwTenantQuery query) {
        JPAQuery<TwTenantVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<TwTenantVO> queryPaging(TwTenantQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<TwTenantVO> jpaQuery = getJpaQueryWhere(query);
        List<TwTenantVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<TwTenantVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public TwTenantDO save(TwTenantDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<TwTenantDO> saveAll(List<TwTenantDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(TwTenantPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 租户编码
        if (payload.getTenantCode() != null) {
            update.set(qdo.tenantCode, payload.getTenantCode());
        }
        // 租户名称
        if (payload.getTenantName() != null) {
            update.set(qdo.tenantName, payload.getTenantName());
        }
        // 管理员的用户ID
        if (payload.getAdminUserId() != null) {
            update.set(qdo.adminUserId, payload.getAdminUserId());
        }
        // 租户类型
        if (payload.getTenantType() != null) {
            update.set(qdo.tenantType, payload.getTenantType());
        }
        // 是否初始化
        if (payload.getInited() != null) {
            update.set(qdo.inited, payload.getInited());
        }
        // 是否启用
        if (payload.getEnabled() != null) {
            update.set(qdo.enabled, payload.getEnabled());
        }
        // 联系人
        if (payload.getLinkman() != null) {
            update.set(qdo.linkman, payload.getLinkman());
        }
        // 联系电话
        if (payload.getContactNumber() != null) {
            update.set(qdo.contactNumber, payload.getContactNumber());
        }
        // 联系地址
        if (payload.getAddress() != null) {
            update.set(qdo.address, payload.getAddress());
        }
        // 租户数据隔离
        if (payload.getTenantIsolation() != null) {
            update.set(qdo.tenantIsolation, payload.getTenantIsolation());
        }
        // 数据库是否已初始化
        if (payload.getDbInitialized() != null) {
            update.set(qdo.dbInitialized, payload.getDbInitialized());
        }
        // 租户数据隔离数据源
        if (payload.getDatabaseSourceId() != null) {
            update.set(qdo.databaseSourceId, payload.getDatabaseSourceId());
        }
        // schema名称，如果租户数据隔离通过schema
        if (payload.getSchemaName() != null) {
            update.set(qdo.schemaName, payload.getSchemaName());
        }
        // 域名地址
        if (payload.getTenantDomain() != null) {
            update.set(qdo.tenantDomain, payload.getTenantDomain());
        }
        // 租户自定义域名
        if (payload.getCustomDomain() != null) {
            update.set(qdo.customDomain, payload.getCustomDomain());
        }
        // 管理员登录账号
        if (payload.getAdminAccount() != null) {
            update.set(qdo.adminAccount, payload.getAdminAccount());
        }
        // 基础数据同步记录ID
        if (payload.getBaseDataSyncId() != null) {
            update.set(qdo.baseDataSyncId, payload.getBaseDataSyncId());
        }
        // 所属行业，[UDC]COM:INDUSTRY
        if (payload.getIndustry() != null) {
            update.set(qdo.industry, payload.getIndustry());
        }
        // 所属客户，[UDC]SYS:CUSTOMER
        if (payload.getCustomer() != null) {
            update.set(qdo.customer, payload.getCustomer());
        }
        // 平台租户主键
        if (payload.getSysTenantId() != null) {
            update.set(qdo.sysTenantId, payload.getSysTenantId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 租户编码
            if (nullFields.contains("tenantCode")) {
                update.setNull(qdo.tenantCode);
            }
            // 租户名称
            if (nullFields.contains("tenantName")) {
                update.setNull(qdo.tenantName);
            }
            // 管理员的用户ID
            if (nullFields.contains("adminUserId")) {
                update.setNull(qdo.adminUserId);
            }
            // 租户类型
            if (nullFields.contains("tenantType")) {
                update.setNull(qdo.tenantType);
            }
            // 是否初始化
            if (nullFields.contains("inited")) {
                update.setNull(qdo.inited);
            }
            // 是否启用
            if (nullFields.contains("enabled")) {
                update.setNull(qdo.enabled);
            }
            // 联系人
            if (nullFields.contains("linkman")) {
                update.setNull(qdo.linkman);
            }
            // 联系电话
            if (nullFields.contains("contactNumber")) {
                update.setNull(qdo.contactNumber);
            }
            // 联系地址
            if (nullFields.contains("address")) {
                update.setNull(qdo.address);
            }
            // 租户数据隔离
            if (nullFields.contains("tenantIsolation")) {
                update.setNull(qdo.tenantIsolation);
            }
            // 数据库是否已初始化
            if (nullFields.contains("dbInitialized")) {
                update.setNull(qdo.dbInitialized);
            }
            // 租户数据隔离数据源
            if (nullFields.contains("databaseSourceId")) {
                update.setNull(qdo.databaseSourceId);
            }
            // schema名称，如果租户数据隔离通过schema
            if (nullFields.contains("schemaName")) {
                update.setNull(qdo.schemaName);
            }
            // 域名地址
            if (nullFields.contains("tenantDomain")) {
                update.setNull(qdo.tenantDomain);
            }
            // 租户自定义域名
            if (nullFields.contains("customDomain")) {
                update.setNull(qdo.customDomain);
            }
            // 管理员登录账号
            if (nullFields.contains("adminAccount")) {
                update.setNull(qdo.adminAccount);
            }
            // 基础数据同步记录ID
            if (nullFields.contains("baseDataSyncId")) {
                update.setNull(qdo.baseDataSyncId);
            }
            // 所属行业，[UDC]COM:INDUSTRY
            if (nullFields.contains("industry")) {
                update.setNull(qdo.industry);
            }
            // 所属客户，[UDC]SYS:CUSTOMER
            if (nullFields.contains("customer")) {
                update.setNull(qdo.customer);
            }
            // 平台租户主键
            if (nullFields.contains("sysTenantId")) {
                update.setNull(qdo.sysTenantId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

