package com.elitesland.tw.tw5crm.server.visit.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmOperationPlanDetailVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.crm.entity.QCrmCustomerDO;
import com.elitesland.tw.tw5.server.prd.crm.entity.QCrmCustomerOperationDO;
import com.elitesland.tw.tw5.server.prd.crm.entity.QCrmOperationPlanDetailDO;
import com.elitesland.tw.tw5crm.api.visit.payload.VisitPlanPayload;
import com.elitesland.tw.tw5crm.api.visit.query.VisitPlanQuery;
import com.elitesland.tw.tw5crm.api.visit.vo.VisitPlanVO;
import com.elitesland.tw.tw5crm.server.common.constants.VisitTaskPlanStatusEnum;
import com.elitesland.tw.tw5crm.server.visit.entity.QVisitPlanDO;
import com.elitesland.tw.tw5crm.server.visit.entity.VisitPlanDO;
import com.elitesland.tw.tw5crm.server.visit.repo.VisitPlanRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 销售拜访计划
 *
 * @author liwenpeng
 * @date 2023-03-14
 */
@Repository
@RequiredArgsConstructor
public class VisitPlanDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final VisitPlanRepo repo;
    private final QVisitPlanDO qdo = QVisitPlanDO.visitPlanDO;
    private final QCrmCustomerDO customerDO = QCrmCustomerDO.crmCustomerDO;

    private final QCrmCustomerOperationDO customerOperationDO = QCrmCustomerOperationDO.crmCustomerOperationDO;
    private final QCrmOperationPlanDetailDO operationPlanDetailDO = QCrmOperationPlanDetailDO.crmOperationPlanDetailDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<VisitPlanVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(VisitPlanVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 计划类型
                qdo.visitPlanType,
                qdo.visitPlanStatus,
                // 计划名称
                qdo.visitPlanName,
                // 拜访客户id
                qdo.customId,
                customerDO.customerName,
                customerDO.customerName,
                // 客户地址
                qdo.customAdress,
                // 拜访时间
                qdo.visitDateFrom,
                // 拜访时间
                qdo.visitDateTo,
                // 拜访次数
                qdo.visitTimes,
                // 添加提醒
                qdo.remindTime,
                qdo.extString1,
                qdo.extString2,
                qdo.extString3,
                qdo.extString4,
                qdo.extString5,
                // 关联信息类型（线索、商机OPPO、经营计划、销售目标）
                qdo.objType,
                // 关联信息主键（关联主键)
                qdo.objId,
                // 关联信息名称（冗余）
                qdo.objName
        )).from(qdo).leftJoin(customerDO).on(customerDO.id.eq(qdo.customId));
    }



    /**
    * 拼装查询条件
    *
    * @param query 查询参数
    * @return jpaQuery对象
    */
    private JPAQuery<VisitPlanVO> getJpaQueryWhere(VisitPlanQuery query) {
        JPAQuery<VisitPlanVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(VisitPlanQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
    * 查询条件封装
    *
    * @param query 条件
    * @return {@link Predicate}
    */
    private Predicate where(VisitPlanQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 计划类型 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitPlanType())) {
            list.add(qdo.visitPlanType.eq(query.getVisitPlanType()));
        }
        /** 计划类型 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitPlanStatus())) {
            list.add(qdo.visitPlanStatus.eq(query.getVisitPlanStatus()));
        }
        /** 计划名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getVisitPlanName())) {
             list.add(qdo.visitPlanName.like(SqlUtil.toSqlLikeString(query.getVisitPlanName())));
        }
        /** 拜访客户id 精确 */
        if (!ObjectUtils.isEmpty(query.getCustomId())) {
            list.add(qdo.customId.eq(query.getCustomId()));
        }
        /** 客户地址 模糊 */
        if (!ObjectUtils.isEmpty(query.getCustomAdress())) {
             list.add(qdo.customAdress.like(SqlUtil.toSqlLikeString(query.getCustomAdress())));
        }
        /** 拜访次数 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitTimes())) {
            list.add(qdo.visitTimes.eq(query.getVisitTimes()));
        }
        /** 创建人 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            list.add(qdo.createUserId.eq(query.getCreateUserId()));
        }
        /** 拜访时间 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitDateFrom())) {
            list.add(qdo.visitDateFrom.goe(query.getVisitDateFrom()));
        }
        if (!ObjectUtils.isEmpty(query.getVisitDateTo())) {
            list.add(qdo.visitDateTo.loe(query.getVisitDateTo()));
        }
        /** 创建人 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            list.add(qdo.createUserId.eq(query.getCreateUserId()));
        }

        /** 拜访人 */
        if (!ObjectUtils.isEmpty(query.getVisitPersonNameStr())) {
            list.add(qdo.extString1.contains(query.getVisitPersonNameStr()));
        }

        if (!ObjectUtils.isEmpty(query.getAccompanyPersonNameStr())) {
            list.add(qdo.extString2.contains(query.getAccompanyPersonNameStr()));
        }

        /** 权限相关 数据权限：创建人、创建人的领导、计划的拜访/协访成员
         */
        if (query.getPermissionFlag()) {
            BooleanExpression jpaQueryOr = qdo.createUserId.eq(query.getLoginUserId());
            // 下属的数据
            if (!CollectionUtils.isEmpty(query.getSubordinatesIds())) {
                jpaQueryOr = jpaQueryOr.or(qdo.createUserId.in(query.getSubordinatesIds()));
            }
            //拜访成员和协防成员
            jpaQueryOr = jpaQueryOr.or(qdo.extString1.contains(query.getLoginUserId()+",")).or(qdo.extString2.contains(query.getLoginUserId()+","));

            // 数据权限新增 拜访成员的上级
            jpaQueryOr = jpaQueryOr.or(qdo.extString3.contains(query.getLoginUserId()+","));
            list.add(jpaQueryOr);
        }


        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public VisitPlanVO queryByKey(Long id) {
        JPAQuery<VisitPlanVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    public int verifyVisitPlanName(String visitPlanName) {
        JPAQuery<VisitPlanVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.visitPlanName.eq(visitPlanName));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch().size();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<VisitPlanVO> queryListDynamic(VisitPlanQuery query) {
        JPAQuery<VisitPlanVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    public VisitPlanVO findByKey(Long key){
        JPAQuery<VisitPlanVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQuerySelect.where(qdo.id.eq(key));
        jpaQuerySelect.where(qdo.deleteFlag.eq(0));
        return jpaQuerySelect.fetchFirst();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<VisitPlanVO> queryPaging(VisitPlanQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<VisitPlanVO> jpaQuery = getJpaQueryWhere(query);
        List<VisitPlanVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<VisitPlanVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public VisitPlanDO save(VisitPlanDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<VisitPlanDO> saveAll(List<VisitPlanDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    public long updateByKeyDynamic(VisitPlanPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
                // 计划类型
        if (payload.getVisitPlanType() != null) {
            update.set(qdo.visitPlanType, payload.getVisitPlanType());
        }
        // 计划状态
        if (payload.getVisitPlanStatus() != null) {
            update.set(qdo.visitPlanStatus, payload.getVisitPlanStatus());
        }
                // 计划名称
        if (payload.getVisitPlanName() != null) {
            update.set(qdo.visitPlanName, payload.getVisitPlanName());
        }
                // 拜访客户id
        if (payload.getCustomId() != null) {
            update.set(qdo.customId, payload.getCustomId());
        }
                // 客户地址
        if (payload.getCustomAdress() != null) {
            update.set(qdo.customAdress, payload.getCustomAdress());
        }
        if (payload.getVisitDateFrom() != null) {
            update.set(qdo.visitDateFrom, payload.getVisitDateFrom());
        }
        if (payload.getVisitDateTo() != null) {
            update.set(qdo.visitDateTo, payload.getVisitDateTo());
        }

                // 拜访次数
        if (payload.getVisitTimes() != null) {
            update.set(qdo.visitTimes, payload.getVisitTimes());
        }

        // 提醒
        if (payload.getRemindTime() != null) {
            update.set(qdo.remindTime, payload.getRemindTime());
        }
                // 关联信息类型（线索、商机OPPO、经营计划、销售目标）
        if (payload.getObjType() != null) {
            update.set(qdo.objType, payload.getObjType());
        }
                // 关联信息主键（关联主键)
        if (payload.getObjId() != null) {
            update.set(qdo.objId, payload.getObjId());
        }
                // 关联信息名称（冗余）
        if (payload.getObjName() != null) {
            update.set(qdo.objName, payload.getObjName());
        }
        if (payload.getExtString1() != null) {
            update.set(qdo.extString1, payload.getExtString1());
        }
        if (payload.getExtString2() != null) {
            update.set(qdo.extString2, payload.getExtString2());
        }

        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            if (nullFields.contains("remindTime")) {
                update.setNull(qdo.remindTime);
            }
            // 计划类型
            if (nullFields.contains("visitPlanType")) {
                update.setNull(qdo.visitPlanType);
            }
            // 计划名称
            if (nullFields.contains("visitPlanName")) {
                update.setNull(qdo.visitPlanName);
            }
            // 拜访客户id
            if (nullFields.contains("customId")) {
                update.setNull(qdo.customId);
            }
            // 客户地址
            if (nullFields.contains("customAdress")) {
                update.setNull(qdo.customAdress);
            }
            // 拜访次数
            if (nullFields.contains("visitTimes")) {
                update.setNull(qdo.visitTimes);
            }
            // 拜访次数
            if (nullFields.contains("visitDateFrom")) {
                update.setNull(qdo.visitDateFrom);
            }
            // 拜访次数
            if (nullFields.contains("visitDateTo")) {
                update.setNull(qdo.visitDateTo);
            }
            // 关联信息类型（线索、商机OPPO、经营计划、销售目标）
            if (nullFields.contains("objType")) {
                update.setNull(qdo.objType);
            }
            // 关联信息主键（关联主键)
            if (nullFields.contains("objId")) {
                update.setNull(qdo.objId);
            }
            // 关联信息名称（冗余）
            if (nullFields.contains("objName")) {
                update.setNull(qdo.objName);
            }
            // 关联信息名称（冗余）
            if (nullFields.contains("extString1")) {
                update.setNull(qdo.extString1);
            }
            if (nullFields.contains("extString2")) {
                update.setNull(qdo.extString2);
            }
        }

        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    public long completeVisitPlan(Long visitPlanId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(visitPlanId));

        update.set(qdo.visitPlanStatus, VisitTaskPlanStatusEnum.completed.getCode());
        SqlUtil.updateCommonJpaQuery(update, qdo._super);

        // 执行修改
        return update.execute();
    }
    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    private JPAQuery<CrmOperationPlanDetailVO> getOperationPlanJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CrmOperationPlanDetailVO.class,
                operationPlanDetailDO.id,
                operationPlanDetailDO.planName
        )).from(operationPlanDetailDO).leftJoin(customerOperationDO).on(customerOperationDO.id.eq(operationPlanDetailDO.operId));
    }

    public List<CrmOperationPlanDetailVO> getOperationPlanByCustomId(Long customId) {
        JPAQuery<CrmOperationPlanDetailVO> jpaQuery = getOperationPlanJpaQuerySelect();
        jpaQuery.where(operationPlanDetailDO.deleteFlag.eq(0));
        jpaQuery.where(customerOperationDO.deleteFlag.eq(0));
        jpaQuery.where(customerOperationDO.customerId.eq(customId));
        return jpaQuery.fetch();
    }

}

