package com.elitesland.tw.tw5crm.server.visit.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.visit.payload.VisitPlanDetailPayload;
import com.elitesland.tw.tw5crm.api.visit.query.VisitPlanDetailQuery;
import com.elitesland.tw.tw5crm.api.visit.vo.VisitPlanDetailVO;
import com.elitesland.tw.tw5crm.server.visit.entity.QVisitPlanDetailDO;
import com.elitesland.tw.tw5crm.server.visit.entity.VisitPlanDetailDO;
import com.elitesland.tw.tw5crm.server.visit.repo.VisitPlanDetailRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 销售拜访计划成员
 *
 * @author liwenpeng
 * @date 2023-03-14
 */
@Repository
@RequiredArgsConstructor
public class VisitPlanDetailDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final VisitPlanDetailRepo repo;
    private final QVisitPlanDetailDO qdo = QVisitPlanDetailDO.visitPlanDetailDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<VisitPlanDetailVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(VisitPlanDetailVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 拜访计划id
                qdo.visitPlanId,
                // 拜访人员id
                qdo.visitPersonId,
                // 拜访成员类型，0：拜访成员，1：协访成员
                qdo.visitPersonType,
                qdo.extString1
        )).from(qdo);
    }

    /**
    * 拼装查询条件
    *
    * @param query 查询参数
    * @return jpaQuery对象
    */
    private JPAQuery<VisitPlanDetailVO> getJpaQueryWhere(VisitPlanDetailQuery query) {
        JPAQuery<VisitPlanDetailVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(VisitPlanDetailQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
    * 查询条件封装
    *
    * @param query 条件
    * @return {@link Predicate}
    */
    private Predicate where(VisitPlanDetailQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 拜访计划id 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitPlanId())) {
            list.add(qdo.visitPlanId.eq(query.getVisitPlanId()));
        }
        /** 拜访人员id 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitPersonId())) {
            list.add(qdo.visitPersonId.eq(query.getVisitPersonId()));
        }
        /** 拜访成员类型，0：拜访成员，1：协访成员 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitPersonType())) {
            list.add(qdo.visitPersonType.eq(query.getVisitPersonType()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public VisitPlanDetailVO queryByKey(Long id) {
        JPAQuery<VisitPlanDetailVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<VisitPlanDetailVO> queryListDynamic(VisitPlanDetailQuery query) {
        JPAQuery<VisitPlanDetailVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<VisitPlanDetailVO> queryPaging(VisitPlanDetailQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<VisitPlanDetailVO> jpaQuery = getJpaQueryWhere(query);
        List<VisitPlanDetailVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<VisitPlanDetailVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public VisitPlanDetailDO save(VisitPlanDetailDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<VisitPlanDetailDO> saveAll(List<VisitPlanDetailDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(VisitPlanDetailPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
                // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
                // 拜访计划id
        if (payload.getVisitPlanId() != null) {
            update.set(qdo.visitPlanId, payload.getVisitPlanId());
        }
                // 拜访人员id
        if (payload.getVisitPersonId() != null) {
            update.set(qdo.visitPersonId, payload.getVisitPersonId());
        }
                // 拜访成员类型，0：拜访成员，1：协访成员
        if (payload.getVisitPersonType() != null) {
            update.set(qdo.visitPersonType, payload.getVisitPersonType());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 拜访计划id
            if (nullFields.contains("visitPlanId")) {
                update.setNull(qdo.visitPlanId);
            }
            // 拜访人员id
            if (nullFields.contains("visitPersonId")) {
                update.setNull(qdo.visitPersonId);
            }
            // 拜访成员类型，0：拜访成员，1：协访成员
            if (nullFields.contains("visitPersonType")) {
                update.setNull(qdo.visitPersonType);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public long deleteByVisitPlanId(Long visitPlanId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.visitPlanId.eq(visitPlanId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

