package com.elitesland.tw.tw5crm.server.visit.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.visit.payload.VisitTaskPayload;
import com.elitesland.tw.tw5crm.api.visit.query.VisitTaskQuery;
import com.elitesland.tw.tw5crm.api.visit.vo.VisitTaskVO;
import com.elitesland.tw.tw5crm.server.common.constants.VisitTaskStatusEnum;
import com.elitesland.tw.tw5crm.server.visit.entity.QVisitPlanDO;
import com.elitesland.tw.tw5crm.server.visit.entity.QVisitTaskDO;
import com.elitesland.tw.tw5crm.server.visit.entity.VisitTaskDO;
import com.elitesland.tw.tw5crm.server.visit.repo.VisitTaskRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 销售拜访任务管理
 *
 * @author duwh
 * @date 2023-03-13
 */
@Repository
@RequiredArgsConstructor
public class VisitTaskDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final VisitTaskRepo repo;
    private final QVisitTaskDO qdo = QVisitTaskDO.visitTaskDO;
    private final QVisitPlanDO qVisitPlanDO=QVisitPlanDO.visitPlanDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<VisitTaskVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(VisitTaskVO.class,
            qdo.id,
            //qdo.remark,
            qdo.createUserId,
            qdo.creator,
            //qdo.createTime,
            //qdo.modifyUserId,
            //qdo.updater,
            qdo.modifyTime,
            // 拜访计划主键
            qdo.planId,
            // 任务名称
            qdo.taskName,
            // 任务编号
            qdo.taskNo,
            // 拜访计划名称（冗余）
            qdo.planName,
            // 拜访计划类型（冗余）
            qdo.planType,
            // 拜访客户主键 crm_customer.id
            qdo.customerId,
            // 拜访客户名称（冗余）crm_customer.customer_name
            qdo.customerName,
            // 任务状态 udc[crm:visit:task_status](已完成OK、待提交READY、未开始NEW)
            qdo.status,
            // 拜访成员
            qdo.visitMemberId,
            // 拜访成员姓名
            qdo.visitMemberName,
            // 协防人员 多个英文逗号隔开
            qdo.teamMemberIds,
            // 拜访开始时间
            qdo.startTime,
            // 拜访结束时间
            qdo.endTime,
            qdo.submitTime,
            // 关联信息类型（线索、商机OPPO、经营计划、销售目标）
            qdo.objType,
            // 关联信息主键（关联主键)
            qdo.objId,
            // 关联信息名称（冗余）
            qdo.objName,
            // 计划创建人
            qdo.planCreateUserId,
            // 评价状态 NEW待评价 OK已评价
            qdo.evalStatus,
            // 领导评分
            qdo.leaderGradle,
            // 下次拜访行动部署
            qdo.nextPlan,
            // 客户评分
            qdo.customerGradle,
            // 客户建议
            qdo.customerAdvise,
            // 拓展字段1
            qdo.ext1,
            // 拓展字段2
            qdo.ext2,
            // 拓展字段3
            qdo.ext3,
            // 拓展字段4
            qdo.ext4,
            // 拓展字段5
            qdo.ext5,
            // 排序号
            qdo.sortNo
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<VisitTaskVO> getJpaQueryWhere(VisitTaskQuery query) {
        JPAQuery<VisitTaskVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(VisitTaskQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
                .leftJoin(qVisitPlanDO)
                .on(qVisitPlanDO.id.eq(qdo.planId))
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 统计任务数量 ：拜访计划下 任务状态为【已完成、待提交】的数据
     *
     * @param planId 计划id
     * @return long
     */
    public long countByPlanId(Long planId) {
        VisitTaskQuery taskQuery = new VisitTaskQuery();
        taskQuery.setPlanId(planId);
        List<String> statusList = new ArrayList<>();
        statusList.add(VisitTaskStatusEnum.OK.getCode());
        statusList.add(VisitTaskStatusEnum.READY.getCode());
        taskQuery.setStatusList(statusList);
        return count(taskQuery);
    }


    /**
     * 统计
     * @param query 条件
     * @return long
     */
    public long done(VisitTaskQuery query) {
        Long count = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .leftJoin(qVisitPlanDO)
                .on(qdo.planId.eq(qVisitPlanDO.id))
                .where(qVisitPlanDO.remindTime.loe(query.getRemindTime())
                        .and(qdo.deleteFlag.eq(0))
                        .and(qVisitPlanDO.deleteFlag.eq(0))
                )
                .fetchOne();
        return count;
    }

    /**
     * 更新短信标记
     * @return long
     */
    public long updateSendFlag(){
        JPAUpdateClause update=jpaQueryFactory.update(qdo);
        update.set(qdo.sendFlag,1);
        return update.execute();
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(VisitTaskQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 拜访计划主键 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanId())) {
            list.add(qdo.planId.eq(query.getPlanId()));
        }
        /** 任务名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getTaskName())) {
            list.add(qdo.taskName.like(SqlUtil.toSqlLikeString(query.getTaskName())));
        }
        /** 任务编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getTaskNo())) {
            list.add(qdo.taskNo.like(SqlUtil.toSqlLikeString(query.getTaskNo())));
        }
        /** 拜访计划名称（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getPlanName())) {
            list.add(qdo.planName.like(SqlUtil.toSqlLikeString(query.getPlanName())));
        }
        /** 拜访计划类型（冗余） 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanType())) {
            list.add(qdo.planType.eq(query.getPlanType()));
        }
        /** 拜访客户主键 crm_customer.id 精确 */
        if (!ObjectUtils.isEmpty(query.getCustomerId())) {
            list.add(qdo.customerId.eq(query.getCustomerId()));
        }
        /** 拜访客户名称（冗余）crm_customer.customer_name 模糊 */
        if (!ObjectUtils.isEmpty(query.getCustomerName())) {
            list.add(qdo.customerName.like(SqlUtil.toSqlLikeString(query.getCustomerName())));
        }
        /** 任务状态 udc[crm:visit:task_status](已完成OK、待提交READY、未开始NEW) 精确 */
        if (!ObjectUtils.isEmpty(query.getStatus())) {
            list.add(qdo.status.eq(query.getStatus()));
        }
        if (!ObjectUtils.isEmpty(query.getStatusList())) {
            list.add(qdo.status.in(query.getStatusList()));
        }
        /** 拜访成员 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitMemberId())) {
            list.add(qdo.visitMemberId.eq(query.getVisitMemberId()));
        }
        // 拜访成员 直属上级
        if (!ObjectUtils.isEmpty(query.getVisitMemberParentId())) {
            list.add(qdo.visitMemberParentId.eq(query.getVisitMemberParentId()));
        }
        /** 拜访成员姓名 模糊 */
        if (!ObjectUtils.isEmpty(query.getVisitMemberName())) {
            list.add(qdo.visitMemberName.like(SqlUtil.toSqlLikeString(query.getVisitMemberName())));
        }
        /** 协防人员 多个英文逗号隔开 模糊 */
        if (!ObjectUtils.isEmpty(query.getTeamMemberIds())) {
            list.add(qdo.teamMemberIds.like(SqlUtil.toSqlLikeString(query.getTeamMemberIds())));
        }
        /** 关联信息类型（线索、商机OPPO、经营计划、销售目标） 精确 */
        if (!ObjectUtils.isEmpty(query.getObjType())) {
            list.add(qdo.objType.eq(query.getObjType()));
        }
        /** 关联信息主键（关联主键) 精确 */
        if (!ObjectUtils.isEmpty(query.getObjId())) {
            list.add(qdo.objId.eq(query.getObjId()));
        }
        /** 关联信息名称（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getObjName())) {
            list.add(qdo.objName.like(SqlUtil.toSqlLikeString(query.getObjName())));
        }
        /** 计划创建人 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanCreateUserId())) {
            list.add(qdo.planCreateUserId.eq(query.getPlanCreateUserId()));
        }
        /** 评价状态 NEW待评价 OK已评价 精确 */
        if (!ObjectUtils.isEmpty(query.getEvalStatus())) {
            list.add(qdo.evalStatus.eq(query.getEvalStatus()));
        }
        /** 评价状态 查询为空 or  待评价数据 */
        if (!ObjectUtils.isEmpty(query.getEvalStatusPro())) {
            final BooleanExpression eq = qdo.evalStatus.eq(query.getEvalStatusPro()).or(qdo.evalStatus.isNull());
            list.add(eq);
        }
        /** 领导评分 精确 */
        if (!ObjectUtils.isEmpty(query.getLeaderGradle())) {
            list.add(qdo.leaderGradle.eq(query.getLeaderGradle()));
        }
        /** 下次拜访行动部署 模糊 */
        if (!ObjectUtils.isEmpty(query.getNextPlan())) {
            list.add(qdo.nextPlan.like(SqlUtil.toSqlLikeString(query.getNextPlan())));
        }
        /** 客户评分 精确 */
        if (!ObjectUtils.isEmpty(query.getCustomerGradle())) {
            list.add(qdo.customerGradle.eq(query.getCustomerGradle()));
        }
        /** 客户建议 模糊 */
        if (!ObjectUtils.isEmpty(query.getCustomerAdvise())) {
            list.add(qdo.customerAdvise.like(SqlUtil.toSqlLikeString(query.getCustomerAdvise())));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 拓展字段4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 拓展字段5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 拜访开始时间 大于 */
        if (!ObjectUtils.isEmpty(query.getStartTime())) {
            list.add(qdo.startTime.goe(query.getStartTime()));
        }
        /** 拜访结束时间 小于等于 */
        if (!ObjectUtils.isEmpty(query.getEndTime())) {
            list.add(qdo.endTime.loe(query.getEndTime()));
        }
        // 拜访时间 TODO 生成的sql缺少括号 但是查询结果正确
        if (!ObjectUtils.isEmpty(query.getVisitTime())) {
            BooleanExpression right = qdo.startTime.loe(query.getVisitTime()).and(qdo.endTime.goe(query.getVisitTime()).and(qdo.status.in(VisitTaskStatusEnum.NEW.getCode(),VisitTaskStatusEnum.READY.getCode())));
            BooleanExpression left = qdo.status.eq(VisitTaskStatusEnum.OK.getCode()).and(qdo.submitTime.goe(query.getVisitTime().atStartOfDay()).and(qdo.submitTime.lt(query.getVisitTime().plusDays(1).atStartOfDay())));
            Predicate or = ExpressionUtils.or(right, left);
            list.add(or);
        }
        if (query.getPermissionFlag()) {
            // 数据权限：计划创建人、创建人领导、拜访成员、拜访成员领导
            if (null == query.getCreateUserIdPro()) {
                throw TwException.error("", "createUserIdPro参数缺失");
            }
            BooleanExpression jpaQueryOr = qdo.createUserId.eq(query.getCreateUserIdPro())
                .or(qdo.planCreateUserId.eq(query.getCreateUserIdPro()))
                .or(qdo.visitMemberId.eq(query.getCreateUserIdPro()));
            // 下属的数据
            if (!CollectionUtils.isEmpty(query.getSubordinatesIds())) {
                jpaQueryOr = jpaQueryOr.or(qdo.createUserId.in(query.getSubordinatesIds()))
                    .or(qdo.visitMemberId.in(query.getSubordinatesIds()));

            }
            list.add(jpaQueryOr);
        }

        // 下属的数据 for 评价
        if (!CollectionUtils.isEmpty(query.getEvalSubordinatesIds())) {
            BooleanExpression jpaQueryOr = qdo.visitMemberId.in(query.getEvalSubordinatesIds());
            list.add(jpaQueryOr);
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public VisitTaskVO queryByKey(Long id) {
        JPAQuery<VisitTaskVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<VisitTaskVO> queryListDynamic(VisitTaskQuery query) {
        JPAQuery<VisitTaskVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<VisitTaskVO> queryPaging(VisitTaskQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<VisitTaskVO> jpaQuery = getJpaQueryWhere(query);
        List<VisitTaskVO> result = jpaQuery
            .offset(query.getPageRequest().getOffset())
            .limit(query.getPageRequest().getPageSize())
            .fetch();
        return PagingVO.<VisitTaskVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public VisitTaskDO save(VisitTaskDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<VisitTaskDO> saveAll(List<VisitTaskDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(VisitTaskPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 拜访计划主键
        if (payload.getPlanId() != null) {
            update.set(qdo.planId, payload.getPlanId());
        }
        // 任务名称
        if (payload.getTaskName() != null) {
            update.set(qdo.taskName, payload.getTaskName());
        }
        // 任务编号
        if (payload.getTaskNo() != null) {
            update.set(qdo.taskNo, payload.getTaskNo());
        }
        // 拜访计划名称（冗余）
        if (payload.getPlanName() != null) {
            update.set(qdo.planName, payload.getPlanName());
        }
        // 拜访计划类型（冗余）
        if (payload.getPlanType() != null) {
            update.set(qdo.planType, payload.getPlanType());
        }
        // 拜访客户主键 crm_customer.id
        if (payload.getCustomerId() != null) {
            update.set(qdo.customerId, payload.getCustomerId());
        }
        // 拜访客户名称（冗余）crm_customer.customer_name
        if (payload.getCustomerName() != null) {
            update.set(qdo.customerName, payload.getCustomerName());
        }
        // 任务状态 udc[crm:visit:task_status](已完成OK、待提交READY、未开始NEW)
        if (payload.getStatus() != null) {
            update.set(qdo.status, payload.getStatus());
        }
        // 拜访成员
        if (payload.getVisitMemberId() != null) {
            update.set(qdo.visitMemberId, payload.getVisitMemberId());
        }
        // 拜访成员直属上级
        if (payload.getVisitMemberParentId() != null) {
            update.set(qdo.visitMemberParentId, payload.getVisitMemberParentId());
        }
        // 拜访成员姓名
        if (payload.getVisitMemberName() != null) {
            update.set(qdo.visitMemberName, payload.getVisitMemberName());
        }
        // 协防人员 多个英文逗号隔开
        if (payload.getTeamMemberIds() != null) {
            update.set(qdo.teamMemberIds, payload.getTeamMemberIds());
        }
        // 关联信息类型（线索、商机OPPO、经营计划、销售目标）
        if (payload.getObjType() != null) {
            update.set(qdo.objType, payload.getObjType());
        }
        // 关联信息主键（关联主键)
        if (payload.getObjId() != null) {
            update.set(qdo.objId, payload.getObjId());
        }
        // 关联信息名称（冗余）
        if (payload.getObjName() != null) {
            update.set(qdo.objName, payload.getObjName());
        }
        // 计划创建人
        if (payload.getPlanCreateUserId() != null) {
            update.set(qdo.planCreateUserId, payload.getPlanCreateUserId());
        }
        // 评价状态 NEW待评价 OK已评价
        if (payload.getEvalStatus() != null) {
            update.set(qdo.evalStatus, payload.getEvalStatus());
        }
        // 领导评分
        if (payload.getLeaderGradle() != null) {
            update.set(qdo.leaderGradle, payload.getLeaderGradle());
        }
        // 下次拜访行动部署
        if (payload.getNextPlan() != null) {
            update.set(qdo.nextPlan, payload.getNextPlan());
        }
        // 客户评分
        if (payload.getCustomerGradle() != null) {
            update.set(qdo.customerGradle, payload.getCustomerGradle());
        }
        // 客户建议
        if (payload.getCustomerAdvise() != null) {
            update.set(qdo.customerAdvise, payload.getCustomerAdvise());
        }
        // 拓展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 拓展字段4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 拓展字段5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 拜访计划主键
            if (nullFields.contains("planId")) {
                update.setNull(qdo.planId);
            }
            // 任务名称
            if (nullFields.contains("taskName")) {
                update.setNull(qdo.taskName);
            }
            // 任务编号
            if (nullFields.contains("taskNo")) {
                update.setNull(qdo.taskNo);
            }
            // 拜访计划名称（冗余）
            if (nullFields.contains("planName")) {
                update.setNull(qdo.planName);
            }
            // 拜访计划类型（冗余）
            if (nullFields.contains("planType")) {
                update.setNull(qdo.planType);
            }
            // 拜访客户主键 crm_customer.id
            if (nullFields.contains("customerId")) {
                update.setNull(qdo.customerId);
            }
            // 拜访客户名称（冗余）crm_customer.customer_name
            if (nullFields.contains("customerName")) {
                update.setNull(qdo.customerName);
            }
            // 任务状态 udc[crm:visit:task_status](已完成OK、待提交READY、未开始NEW)
            if (nullFields.contains("status")) {
                update.setNull(qdo.status);
            }
            // 拜访成员
            if (nullFields.contains("visitMemberId")) {
                update.setNull(qdo.visitMemberId);
            }
            // 拜访成员姓名
            if (nullFields.contains("visitMemberName")) {
                update.setNull(qdo.visitMemberName);
            }
            // 协防人员 多个英文逗号隔开
            if (nullFields.contains("teamMemberIds")) {
                update.setNull(qdo.teamMemberIds);
            }
            // 关联信息类型（线索、商机OPPO、经营计划、销售目标）
            if (nullFields.contains("objType")) {
                update.setNull(qdo.objType);
            }
            // 关联信息主键（关联主键)
            if (nullFields.contains("objId")) {
                update.setNull(qdo.objId);
            }
            // 关联信息名称（冗余）
            if (nullFields.contains("objName")) {
                update.setNull(qdo.objName);
            }
            // 计划创建人
            if (nullFields.contains("planCreateUserId")) {
                update.setNull(qdo.planCreateUserId);
            }
            // 评价状态 NEW待评价 OK已评价
            if (nullFields.contains("evalStatus")) {
                update.setNull(qdo.evalStatus);
            }
            // 领导评分
            if (nullFields.contains("leaderGradle")) {
                update.setNull(qdo.leaderGradle);
            }
            // 下次拜访行动部署
            if (nullFields.contains("nextPlan")) {
                update.setNull(qdo.nextPlan);
            }
            // 客户评分
            if (nullFields.contains("customerGradle")) {
                update.setNull(qdo.customerGradle);
            }
            // 客户建议
            if (nullFields.contains("customerAdvise")) {
                update.setNull(qdo.customerAdvise);
            }
            // 拓展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 拓展字段4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 拓展字段5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据计划删除软id
     *
     * @param planId 计划id
     * @return long
     */
    public long deleteSoftByPlanId(Long planId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.planId.eq(planId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

