package com.elitesland.tw.tw5crm.server.visit.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.api.prd.ab.query.PrdAbContactsQuery;
import com.elitesland.tw.tw5.api.prd.ab.service.PrdAbContactsService;
import com.elitesland.tw.tw5.api.prd.ab.vo.PrdAbContactsVO;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmCustomerService;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmLeadsService;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmOpportunityService;
import com.elitesland.tw.tw5.api.prd.crm.vo.*;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgEmployeeService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgDataRefVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeRefVO;
import com.elitesland.tw.tw5.api.prd.system.service.PrdMessageConfigService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdMessageConfigVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdMessageConfigConvert;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdMessageConfigDO;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdMessageConfigRepo;
import com.elitesland.tw.tw5crm.api.visit.payload.VisitPlanPayload;
import com.elitesland.tw.tw5crm.api.visit.query.VisitPlanQuery;
import com.elitesland.tw.tw5crm.api.visit.service.VisitPlanService;
import com.elitesland.tw.tw5crm.api.visit.service.VisitTaskService;
import com.elitesland.tw.tw5crm.api.visit.vo.VisitPlanVO;
import com.elitesland.tw.tw5crm.server.common.constants.VisitTaskPlanObjTypeEnum;
import com.elitesland.tw.tw5crm.server.common.constants.VisitTaskPlanPersonTypeEnum;
import com.elitesland.tw.tw5crm.server.common.constants.VisitTaskPlanStatusEnum;
import com.elitesland.tw.tw5crm.server.common.constants.VisitTaskPlanTypeEnum;
import com.elitesland.tw.tw5crm.server.visit.convert.VisitPlanConvert;
import com.elitesland.tw.tw5crm.server.visit.dao.VisitPlanDAO;
import com.elitesland.tw.tw5crm.server.visit.dao.VisitPlanDetailDAO;
import com.elitesland.tw.tw5crm.server.visit.entity.VisitPlanDO;
import com.elitesland.tw.tw5crm.server.visit.entity.VisitPlanDetailDO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 销售拜访计划
 *
 * @author liwenpeng
 * @date 2023-03-14
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class VisitPlanServiceImpl extends BaseServiceImpl implements VisitPlanService {

    private final VisitPlanDAO visitPlanDAO;
    private final VisitPlanDetailDAO visitPlanDetailDAO;
    private final VisitTaskService visitTaskService;
    private final CacheUtil cacheUtil;
    private final PrdOrgEmployeeService prdOrgEmployeeService;
    private final CrmCustomerService customerService;
    private final PrdAbContactsService prdAbContactsService;

    private final CrmLeadsService leadsService;

    private final CrmOpportunityService opportunityService;

    private final PrdMessageConfigService messageConfigService;
    private final PrdMessageConfigRepo repoMessage;


    @Override
    public PagingVO<VisitPlanVO> queryPaging(VisitPlanQuery query){
        //数据权限相关
        dataPermissionFlag(query);
        //拜访人员处理
        if (query.getVisitPerson()!=null){
            query.setVisitPersonNameStr(query.getVisitPerson()+",");
        }
        if (query.getAccompanyPerson()!=null){
            query.setAccompanyPersonNameStr(query.getAccompanyPerson()+",");
        }

        PagingVO<VisitPlanVO> visitPlanVOPagingVO = visitPlanDAO.queryPaging(query);

        //完善信息
        List<VisitPlanVO> records = visitPlanVOPagingVO.getRecords();
        records.forEach(this::tranVisitPlanVo);

        return visitPlanVOPagingVO;
    }

    private void dataPermissionFlag(VisitPlanQuery query){
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode()));
        query.setPermissionFlag(!rolePermission);
        if (!rolePermission) {
            //需要处理权限
            Long userId = GlobalUtil.getLoginUserId();
            if (userId == null){
                throw TwException.error("1","获取当前登陆用户失败");
            }
            query.setLoginUserId(userId);
            // 查找下级
            final Set<Long> subordinatesIdsByUserId = prdOrgEmployeeService.queryOneSubordinatesIdsByUserId(userId);
            query.setSubordinatesIds(subordinatesIdsByUserId);
        }
    }

    @Override
    public VisitPlanVO queryByKey(Long key) {
        VisitPlanVO vo = visitPlanDAO.findByKey(key);
        //完善信息
        tranVisitPlanVo(vo);
        return vo;
    }

    private void tranVisitPlanVo(VisitPlanVO vo){
        if (StringUtils.hasText(vo.getVisitPlanType())){
            vo.setVisitPlanTypeDesc(cacheUtil.transferSystemSelection("crm:visit:task_plan:type",vo.getVisitPlanType()));
            if (VisitTaskPlanTypeEnum.general_plan.getCode().equals(vo.getVisitPlanType())){
                vo.setVisitDate(vo.getVisitDateFrom() == null?"":vo.getVisitDateFrom().toString());
            }else {
                String dateFrom = vo.getVisitDateFrom() == null ? "" : vo.getVisitDateFrom().toString();
                String dateTo = vo.getVisitDateTo() == null ? "" : vo.getVisitDateTo().toString();
                vo.setVisitDate(dateFrom+"~"+dateTo);
            }
        }
        if (StringUtils.hasText(vo.getVisitPlanStatus())){
            vo.setVisitPlanStatus(cacheUtil.transferSystemSelection("crm:visit:task_plan:status",vo.getVisitPlanStatus()));
        }

        if (StringUtils.hasText(vo.getExtString1())){
            String visitPersonStr = vo.getExtString1().substring(0,vo.getExtString1().length()-1);
            String visitPersonNameStr = getPersonNameStr(visitPersonStr);
            List<Long> visitPersonList = Stream.of(visitPersonStr.split(",")).map(Long::valueOf).toList();
            vo.setVisitPersonNameStr(visitPersonNameStr);
            vo.setVisitPersonList(visitPersonList);
        }

        if (StringUtils.hasText(vo.getExtString2())){
            String accompanyPersonStr = vo.getExtString2().substring(0,vo.getExtString2().length()-1);
            String accompanyPersonNameStr = getPersonNameStr(accompanyPersonStr);
            List<Long> accompanyPersonList = Stream.of(accompanyPersonStr.split(",")).map(Long::valueOf).toList();
            vo.setAccompanyPersonNameStr(accompanyPersonNameStr);
            vo.setAccompanyPersonList(accompanyPersonList);
        }
        if (StringUtils.hasText(vo.getObjType())){
            vo.setObjTypeDesc(cacheUtil.transferSystemSelection("crm:visit:obj_type",vo.getObjType()));
        }

        // 客户信息
        if(null == vo.getCustomId()){
            log.warn("销售拜访计划未关联客户；销售拜访计划：{}", vo);
            return;
        }
        final CrmCustomerSimpleVO simpleVO = customerService.querySimpleByKey(vo.getCustomId());
        if(null != simpleVO){
            final Long bookId = simpleVO.getBookId();
            vo.setBookId(bookId);
            vo.setCompanyAddress(simpleVO.getCompanyAddress());
            vo.setLongitudeLatitude(simpleVO.getLongitudeLatitude());

            if(null == bookId){
                log.warn("客户数据异常；找不到bookId;  customId:{}", vo.getCustomId());
                return;
            }
            // 联系人信息
            PrdAbContactsQuery prdAbContactsQuery = new PrdAbContactsQuery();
            prdAbContactsQuery.setBookId(bookId);
            final PagingVO<PrdAbContactsVO> paging = prdAbContactsService.paging(prdAbContactsQuery);
            if (paging.isNotEmpty()) {
                final PrdAbContactsVO prdAbContactsVO = paging.getRecords().get(0);
                vo.setContactsName(prdAbContactsVO.getContactsName());
                vo.setContactsPhone(prdAbContactsVO.getContactsPhone());
            }
        }
    }

    private String getPersonNameStr(String visitPersonStr){

        return Stream.of(visitPersonStr.split(",")).map(e -> {
            Long visitPersonId = Long.valueOf(e);
            String userName = cacheUtil.getUserName(visitPersonId);
            return userName == null ? "" : userName;
        }).collect(Collectors.joining(","));

    }

    @Override
    @Transactional(rollbackFor = RuntimeException.class)
    public VisitPlanVO insert(VisitPlanPayload payload) {
        int countNum = verifyVisitPlanName(payload.getVisitPlanName());
        if (countNum > 0) {
            throw TwException.error("", "计划名称不可重复");
        }

        //拜访人员和协访人员id排序后存入扩展字段中
        saveVisitPersonStr(payload);

        VisitPlanDO entityDo = VisitPlanConvert.INSTANCE.toDo(payload);
        //新增默认未完成
        entityDo.setVisitPlanStatus(VisitTaskPlanStatusEnum.incomplete.getCode());
        //存储计划
        VisitPlanDO save = visitPlanDAO.save(entityDo);
        //存储人员明细
        saveVisitPlanDetail(payload.getVisitPersonList(),save.getId(), VisitTaskPlanPersonTypeEnum.visit_person.getCode());
        saveVisitPlanDetail(payload.getAccompanyPersonList(),save.getId(), VisitTaskPlanPersonTypeEnum.accompany_person.getCode());

        //调用任务接口生产任务
        visitTaskService.generateTaskByPlanId(save.getId());

        // 消息通知
        notice(payload,save);


        return VisitPlanConvert.INSTANCE.toVo(save);
    }

    private void notice(VisitPlanPayload payload, VisitPlanDO save){
        String loginUserName = GlobalUtil.getLoginUserName();
        Long loginUserId = GlobalUtil.getLoginUserId();
        List<Long> noticePersonList = new ArrayList();
        PrdOrgEmployeeRefVO prdOrgEmployeeRefVO = prdOrgEmployeeService.queryEmployeeRef(loginUserId);
        // 创建人上级
        if (prdOrgEmployeeRefVO != null && prdOrgEmployeeRefVO.getParentId()!=null){
            noticePersonList.add(prdOrgEmployeeRefVO.getParentId());
        }
        // 协访成员
        if (payload.getAccompanyPersonList() != null){
            noticePersonList.addAll(payload.getAccompanyPersonList());
        }
        if (noticePersonList.size() > 0){
            List<PrdMessageConfigDO> messageConfigDOS = new ArrayList<>();
            noticePersonList.forEach(userId -> {
                String code = generateSeqNum("MESSAGE_CONFIG_NO");
                Long visitPlanId = save.getId();
                String content = loginUserName + " 新增了一条拜访计划 <a href=\"/customerVisit/visitPlanDetail?id="+visitPlanId+"\">" +
                        save.getVisitPlanName() + "</a> 。";
                PrdMessageConfigDO ado = new PrdMessageConfigDO();
                ado.setMessageCode(code);
                ado.setMessageTitle("新增拜访计划");
                ado.setMessageContent(content);
                ado.setCreateUserId(loginUserId);
                ado.setContentBigType("businessMessage");
                ado.setContentType("systemMessage");
                ado.setCreateSource("新增拜访计划");
                ado.setIsEnable(0);
                ado.setMessageType(2);
                ado.setNoticeScope("appoint_people");
                ado.setNoticeSource(userId + "");
                ado.setNoticeWay("instation");
                ado.setTenantId(save.getTenantId());
                ado.setReleaseSource("profileMessage");
                ado.setReleaseStatus(3);
                ado.setTriggerTime(LocalDateTime.now());
                messageConfigDOS.add(ado);
            });
            //保存消息
            List<PrdMessageConfigDO> messageConfigDOS1 = repoMessage.saveAll(messageConfigDOS);
            //发送消息
            List<PrdMessageConfigVO> messageConfigVOS = messageConfigDOS1.stream().map(PrdMessageConfigConvert.INSTANCE::toVo).collect(Collectors.toList());
            messageConfigService.releaseMessage(messageConfigVOS);
        }
    }

    private void saveVisitPersonStr(VisitPlanPayload payload){
        List<Long> visitPersonList = payload.getVisitPersonList();
        if (visitPersonList == null || visitPersonList.size() < 1){
            throw new RuntimeException("新建计划 拜访成员必填");
        }
        String visitPersonStr = visitPersonList.stream().sorted().map(Object::toString).collect(Collectors.joining(",", "", ","));
        payload.setExtString1(visitPersonStr);
        List<Long> accompanyPersonList = payload.getAccompanyPersonList();
        if(accompanyPersonList!=null && accompanyPersonList.size()>0){
            String accompanyPersonStr = accompanyPersonList.stream().sorted().map(Object::toString).collect(Collectors.joining(",", "", ","));
            payload.setExtString2(accompanyPersonStr);
        }
        //拜访成员的上级 存起来 数据权限用
        String visitPersonParentStr = visitPersonList.stream().map(visitPersonId -> {
            PrdOrgDataRefVO defaultOrgInfoByUserId = cacheUtil.getDefaultOrgInfoByUserId(visitPersonId);
            if(defaultOrgInfoByUserId == null){
                return -1L;
            }else {
                return defaultOrgInfoByUserId.getParentId() == null ? -1L : defaultOrgInfoByUserId.getParentId();
            }
        }).map(Object::toString).distinct().collect(Collectors.joining(",", "", ","));
        payload.setExtString3(visitPersonParentStr);
    }

    private void saveVisitPlanDetail(List<Long> visitPersonList,Long visitPlanId,String visitTaskPlanPersonType){
        if(visitPersonList!=null &&visitPersonList.size()>0){
            //存储人员明细
            for (Long visitPersonId : visitPersonList){
                VisitPlanDetailDO ado = new VisitPlanDetailDO();
                ado.setVisitPlanId(visitPlanId);
                ado.setVisitPersonId(visitPersonId);
                ado.setVisitPersonType(visitTaskPlanPersonType);
                visitPlanDetailDAO.save(ado);
            }
        }

    }

    @Override
    @Transactional(rollbackFor = RuntimeException.class)
    public long update(VisitPlanPayload payload) {
        if (payload.getId()==null || payload.getId() < 0){
            throw TwException.error("1","拜访计划id有误");
        }
        //只有创建人可以修改计划 只有计划下的任务都是未开始状态才可以修改计划
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode()));
        if (!rolePermission){
            Long userId = GlobalUtil.getLoginUserId();
            VisitPlanVO vo = visitPlanDAO.findByKey(payload.getId());
            if (!userId.equals(vo.getCreateUserId())) {
                throw TwException.error("1", "只有创建人可以修改计划，当前登陆用户不是创建人");
            }
        }

        long count = visitTaskService.countByPlanId(payload.getId());
        if (count > 0){
            throw TwException.error("1","该计划下的任务已经开始执行，不能修改计划");
        }
        //拜访人员和协访人员id排序后存入扩展字段中
        saveVisitPersonStr(payload);
        long l = visitPlanDAO.updateByKeyDynamic(payload);

        //更新明细信息
        visitPlanDetailDAO.deleteByVisitPlanId(payload.getId());
        //存储人员明细
        saveVisitPlanDetail(payload.getVisitPersonList(),payload.getId(), VisitTaskPlanPersonTypeEnum.visit_person.getCode());
        saveVisitPlanDetail(payload.getAccompanyPersonList(),payload.getId(), VisitTaskPlanPersonTypeEnum.accompany_person.getCode());
        //同步修改任务
        visitTaskService.generateTaskByPlanId(payload.getId());
        return l;
    }

    @Override
    @Transactional(rollbackFor = RuntimeException.class)
    public long completeVisitPlan(Long visitPlanId) {
        if (visitPlanId == null){
            throw TwException.error("1","拜访计划id不能为空");
        }
        return visitPlanDAO.completeVisitPlan(visitPlanId);
    }

    @Override
    @Transactional(rollbackFor = RuntimeException.class)
    public void deleteSoft(List<Long> keys) {
        //只有创建人可以删除计划 前端控制

        // 校验任务状态
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                final VisitPlanVO visitPlanVO = visitPlanDAO.queryByKey(id);
                if (!visitPlanVO.getCreateUserId().equals(GlobalUtil.getLoginUserId())) {
                    throw TwException.error("1", "只有创建人可以删除计划，当前登陆用户不是创建人");
                }
                long count = visitTaskService.countByPlanId(id);
                if (count > 0) {
//                    throw TwException.error("1", "该计划【" + visitPlanVO.getVisitPlanName() + "】下的任务已开始/已完成，不能删除");
                    throw TwException.error("1", "拜访任务已开始/已完成，不能删除拜访计划");
                }
            });
        }
        if (!keys.isEmpty()) {
            visitPlanDAO.deleteSoft(keys);
            // 删除拜访任务
            keys.stream().forEach(id -> {
                visitTaskService.deleteSoftByPlanId(id);
            });

        }
    }

    @Override
    @Transactional
    public int verifyVisitPlanName(String visitPlanName) {
        //验证重名数据 返回重名数据的数量
        if (StringUtils.hasText(visitPlanName)){
            return visitPlanDAO.verifyVisitPlanName(visitPlanName);
        }
        return 0;
    }

    @Override
    public Map<String, Integer> visitPlanCount() {
        Map<String, Integer> result = new HashMap<>();

        //数据权限相关
        VisitPlanQuery query = new VisitPlanQuery();
        dataPermissionFlag(query);

        //总拜访计划
        List<VisitPlanVO> visitPlans = visitPlanDAO.queryListDynamic(query);
        result.put("visitPlan_count",visitPlans.size());

        //待完成计划
        long count = visitPlans.stream()
                .filter(e -> e.getVisitPlanStatus().equals(VisitTaskPlanStatusEnum.incomplete.getCode()))
                .count();
        result.put("incomplete_VisitPlan_count",(int)count);


        //常规、周期、销访计划
        Map<String, Long> collect = visitPlans.stream()
                .collect(Collectors.groupingBy(VisitPlanVO::getVisitPlanType,Collectors.counting()));
       // collect.forEach((k,v)->result.put(k+"_count",v.intValue()));
        //循环枚举类 返回常规、周期、销访计划
        Arrays.stream(VisitTaskPlanTypeEnum.values()).forEach(key->{
            if(collect.containsKey(key.getCode())){
                result.put(key.getCode()+"_count",collect.get(key.getCode()).intValue());
            }else{
                result.put(key.getCode()+"_count",0);

            }
        });
        return result;
    }

    @Override
    public String getVisitCustomAddress(Long customId) {
        return customerService.queryCompanyAddress(customId);
    }

    @Override
    public List<Map<String, Object>> visitCustomRelationInfo(Long customId, String objType) {
        // 线索
        if (VisitTaskPlanObjTypeEnum.LEAD.getCode().equals(objType)){
            CrmCustomerVO crmCustomerVO = customerService.queryDetail(customId);
            String customerName = crmCustomerVO.getCustomerName();
            List<CrmLeadsListVO> crmLeadsListVOS = leadsService.queryListByCustomName(customerName);
            List<Map<String, Object>> list = crmLeadsListVOS.stream().map(e -> {
                Map<String, Object> result = new HashMap<>();
                result.put("id", e.getId());
                result.put("name", e.getLeadsName());
                return result;
            }).toList();
            return list;
        }else if(VisitTaskPlanObjTypeEnum.OPPO.getCode().equals(objType)){
            List<CrmActProjectVO> crmActProjectVOS = opportunityService.queryListByCustomId(customId);
            List<Map<String, Object>> list = crmActProjectVOS.stream().map(e -> {
                Map<String, Object> result = new HashMap<>();
                result.put("id", e.getId());
                result.put("name", e.getProjectName());
                return result;
            }).toList();
            return list;
        }else {
            List<CrmOperationPlanDetailVO> operationPlanByCustomId = visitPlanDAO.getOperationPlanByCustomId(customId);
            List<Map<String, Object>> list = operationPlanByCustomId.stream().map(e -> {
                Map<String, Object> result = new HashMap<>();
                result.put("id", e.getId());
                result.put("name", e.getPlanName());
                return result;
            }).toList();
            return list;
        }
    }
}
