package com.elitesland.tw.tw5crm.server.visit.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5crm.api.visit.payload.VisitSignRecordPayload;
import com.elitesland.tw.tw5crm.api.visit.payload.VisitTaskPayload;
import com.elitesland.tw.tw5crm.api.visit.query.VisitSignRecordQuery;
import com.elitesland.tw.tw5crm.api.visit.service.VisitSignRecordService;
import com.elitesland.tw.tw5crm.api.visit.vo.VisitSignRecordVO;
import com.elitesland.tw.tw5crm.api.visit.vo.VisitTaskVO;
import com.elitesland.tw.tw5crm.server.common.constants.VisitSignTypeEnum;
import com.elitesland.tw.tw5crm.server.visit.convert.VisitSignRecordConvert;
import com.elitesland.tw.tw5crm.server.visit.dao.VisitSignRecordDAO;
import com.elitesland.tw.tw5crm.server.visit.dao.VisitTaskDAO;
import com.elitesland.tw.tw5crm.server.visit.entity.VisitSignRecordDO;
import com.elitesland.tw.tw5crm.server.visit.repo.VisitSignRecordRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

/**
 * 销售拜访任务执行情况签到记录
 *
 * @author duwh
 * @date 2023-03-13
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class VisitSignRecordServiceImpl extends BaseServiceImpl implements VisitSignRecordService {

    private final VisitSignRecordRepo visitSignRecordRepo;
    private final VisitSignRecordDAO visitSignRecordDAO;
    private final VisitTaskDAO visitTaskDAO;
    private final FileUtil fileUtil;

    @Override
    public PagingVO<VisitSignRecordVO> paging(VisitSignRecordQuery query) {
        Page<VisitSignRecordDO> page = visitSignRecordRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page.map(VisitSignRecordConvert.INSTANCE::toVo));
    }

    @Override
    public PagingVO<VisitSignRecordVO> queryPaging(VisitSignRecordQuery query) {
        return visitSignRecordDAO.queryPaging(query);
    }

    @Override
    public List<VisitSignRecordVO> queryList(VisitSignRecordQuery query) {
        return VisitSignRecordConvert.INSTANCE.toVoList(
            visitSignRecordRepo.findAll(
                (root, criteriaQuery, criteriaBuilder)
                    -> QueryHelp.getPredicate(root, query, criteriaBuilder)
                , query.getPageRequest().getSort()
            )
        );
    }

    @Override
    public List<VisitSignRecordVO> queryListDynamic(VisitSignRecordQuery query) {
        final List<VisitSignRecordVO> visitSignRecordVOS = visitSignRecordDAO.queryListDynamic(query);
        return visitSignRecordVOS;
    }

    @Override
    public VisitSignRecordVO queryByKey(Long key) {
        VisitSignRecordDO entity = visitSignRecordRepo.findById(key).orElseGet(VisitSignRecordDO::new);
        Assert.notNull(entity.getId(), "不存在");
        VisitSignRecordVO vo = VisitSignRecordConvert.INSTANCE.toVo(entity);
        final Object fileDatas = fileUtil.getFileDatas(vo.getFileCodes());
        vo.setFileDatas(fileDatas);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public VisitSignRecordVO insert(VisitSignRecordPayload payload) {
        // 数据校验
        check(payload);
        // 数据处理
        dataProcess(payload);
        VisitSignRecordDO entityDo = VisitSignRecordConvert.INSTANCE.toDo(payload);
        final VisitSignRecordDO save = visitSignRecordRepo.save(entityDo);
        // 更新任务执行签到信息
        taskRunDataProcess(payload, save);

        return VisitSignRecordConvert.INSTANCE.toVo(save);
    }

    /**
     * 更新任务执行签到信息
     *
     * @param payload 有效载荷
     * @param save    保存
     */
    private void taskRunDataProcess(VisitSignRecordPayload payload, VisitSignRecordDO save) {
        if (VisitSignTypeEnum.IN.getCode().equals(save.getType())) {
            VisitTaskPayload taskRunPayload = new VisitTaskPayload();
            taskRunPayload.setId(save.getTaskId());
            taskRunPayload.setSignLocations(payload.getLocations());
            taskRunPayload.setSignAddress(payload.getAddress());
            taskRunPayload.setSignTime(LocalDateTime.now());
            taskRunPayload.setSignFileCodes(payload.getFileCodes());
            visitTaskDAO.updateByKeyDynamic(taskRunPayload);
        } else if (VisitSignTypeEnum.OUT.getCode().equals(save.getType())) {
            VisitTaskPayload taskRunPayload = new VisitTaskPayload();
            taskRunPayload.setId(save.getTaskId());
            taskRunPayload.setSignOutLocations(payload.getLocations());
            taskRunPayload.setSignOutAddress(payload.getAddress());
            taskRunPayload.setSignOutTime(LocalDateTime.now());
            taskRunPayload.setSignOutFileCodes(payload.getFileCodes());
            visitTaskDAO.updateByKeyDynamic(taskRunPayload);
        }
    }

    /**
     * 检查
     *
     * @param payload 有效载荷
     */
    private void check(VisitSignRecordPayload payload) {
        if (null == payload.getTaskId()) {
            throw TwException.error("", "taskRunId不能为空");
        }
        if (ObjectUtils.isEmpty(payload.getType())) {
            throw TwException.error("", "type不能为空");
        }
        if (ObjectUtils.isEmpty(payload.getLocations())) {
            throw TwException.error("", "locations不能为空");
        }
        if (ObjectUtils.isEmpty(payload.getAddress())) {
            throw TwException.error("", "address不能为空");
        }

    }

    /**
     * 数据处理
     *
     * @param payload 有效载荷
     */
    private void dataProcess(VisitSignRecordPayload payload) {
        payload.setSignTime(LocalDateTime.now());
        final VisitTaskVO visitTaskVO = visitTaskDAO.queryByKey(payload.getTaskId());
        if (null == payload.getPlanId()) {
            payload.setPlanId(visitTaskVO.getPlanId());
        }
        if (ObjectUtils.isEmpty(payload.getTaskName())) {
            payload.setTaskName(visitTaskVO.getTaskName());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<VisitSignRecordDO> optional = visitSignRecordRepo.findById(id);
                if (!optional.isEmpty()) {
                    VisitSignRecordDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    visitSignRecordRepo.save(entity);
                }
            });
        }
    }

}
