package com.elitesland.tw.tw5crm.server.visit.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitescloud.coord.messenger.sender.provider.SmsRpcService;
import com.elitescloud.coord.messenger.sender.provider.param.TemplateSmsDTO;
import com.elitesland.tw.tw5.api.prd.ab.query.PrdAbContactsQuery;
import com.elitesland.tw.tw5.api.prd.ab.service.PrdAbContactsService;
import com.elitesland.tw.tw5.api.prd.ab.vo.PrdAbContactsVO;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmCustomerService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerSimpleVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerVO;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgEmployeeService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgDataRefVO;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.crm.dao.CrmCustomerDAO;
import com.elitesland.tw.tw5crm.api.open.payload.VisitTaskOpenPayload;
import com.elitesland.tw.tw5crm.api.visit.payload.VisitTaskPayload;
import com.elitesland.tw.tw5crm.api.visit.query.VisitPlanDetailQuery;
import com.elitesland.tw.tw5crm.api.visit.query.VisitTaskQuery;
import com.elitesland.tw.tw5crm.api.visit.service.VisitTaskService;
import com.elitesland.tw.tw5crm.api.visit.vo.VisitPlanDetailVO;
import com.elitesland.tw.tw5crm.api.visit.vo.VisitPlanVO;
import com.elitesland.tw.tw5crm.api.visit.vo.VisitTaskVO;
import com.elitesland.tw.tw5crm.server.common.constants.*;
import com.elitesland.tw.tw5crm.server.visit.convert.VisitTaskConvert;
import com.elitesland.tw.tw5crm.server.visit.dao.VisitPlanDAO;
import com.elitesland.tw.tw5crm.server.visit.dao.VisitPlanDetailDAO;
import com.elitesland.tw.tw5crm.server.visit.dao.VisitSignRecordDAO;
import com.elitesland.tw.tw5crm.server.visit.dao.VisitTaskDAO;
import com.elitesland.tw.tw5crm.server.visit.entity.VisitSignRecordDO;
import com.elitesland.tw.tw5crm.server.visit.entity.VisitTaskDO;
import com.elitesland.tw.tw5crm.server.visit.repo.VisitTaskRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 销售拜访任务管理
 *
 * @author duwh
 * @date 2023-03-13
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class VisitTaskServiceImpl extends BaseServiceImpl implements VisitTaskService {

    private final VisitTaskRepo visitTaskRepo;
    private final VisitTaskDAO visitTaskDAO;
    private final CacheUtil cacheUtil;
    private final PrdOrgEmployeeService prdOrgEmployeeService;
    private final VisitPlanDAO visitPlanDAO;
    private final VisitPlanDetailDAO visitPlanDetailDAO;
    private final CrmCustomerDAO crmCustomerDAO;
    private final VisitSignRecordDAO visitSignRecordDAO;
    private final FileUtil fileUtil;
    private final SmsRpcService smsRpcService;
    @Autowired
    private CrmCustomerService customerService;
    private final PrdAbContactsService prdAbContactsService;

    @Override
    public PagingVO<VisitTaskVO> paging(VisitTaskQuery query) {
        Page<VisitTaskDO> page = visitTaskRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page.map(VisitTaskConvert.INSTANCE::toVo));
    }

    @Override
    public PagingVO<VisitTaskVO> queryPaging(VisitTaskQuery query) {
        // 数据权限处理
        dataPermissionFlag(query);
        // 常规计划的任务按拜访时间归在具体日期下
        final PagingVO<VisitTaskVO> pagingVO = visitTaskDAO.queryPaging(query);
        pagingVO.getRecords().forEach(visitTaskVO -> {
            translation(visitTaskVO, false);
        });
        return pagingVO;
    }

    /**
     * 权限处理
     *
     * @param query
     */
    void dataPermissionFlag(VisitTaskQuery query) {
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode()));
        query.setPermissionFlag(!rolePermission);
        if (!rolePermission) {
            //需要处理权限
            Long userId = GlobalUtil.getLoginUserId();
            query.setCreateUserIdPro(userId);
            // 查找下级
            final Set<Long> subordinatesIdsByUserId = prdOrgEmployeeService.queryOneSubordinatesIdsByUserId(userId);
            query.setSubordinatesIds(subordinatesIdsByUserId);
        }
    }

    @Override
    public List<VisitTaskVO> queryList(VisitTaskQuery query) {
        return VisitTaskConvert.INSTANCE.toVoList(
            visitTaskRepo.findAll(
                (root, criteriaQuery, criteriaBuilder)
                    -> QueryHelp.getPredicate(root, query, criteriaBuilder)
                , query.getPageRequest().getSort()
            )
        );
    }

    @Override
    public List<VisitTaskVO> queryListDynamic(VisitTaskQuery query) {
        // 数据权限处理
        dataPermissionFlag(query);
        return visitTaskDAO.queryListDynamic(query);
    }

    @Override
    public VisitTaskVO queryByKey(Long key) {
        VisitTaskDO entity = visitTaskRepo.findById(key).orElseGet(VisitTaskDO::new);
        Assert.notNull(entity.getId(), "不存在");
        VisitTaskVO vo = VisitTaskConvert.INSTANCE.toVo(entity);

        // 翻译
        translation(vo, true);
        return vo;
    }

    /**
     * 翻译
     *
     * @param vo                  实体类
     * @param translationFileFlag 翻译附件标志
     */
    private void translation(VisitTaskVO vo, boolean translationFileFlag) {
        final String teamMemberIds = vo.getTeamMemberIds();
        if (StringUtils.hasText(teamMemberIds)) {
            final String[] teamMemberIdArr = teamMemberIds.split(",");
            List<String> teamMemberNameList = new ArrayList<>();
            for (int i = 0; i < teamMemberIdArr.length; i++) {
                final String userName = cacheUtil.getUserName(Long.valueOf(teamMemberIdArr[i]));
                teamMemberNameList.add(userName);
            }
            final String teamMemberNames = teamMemberNameList.stream().collect(Collectors.joining(","));
            vo.setTeamMemberNames(teamMemberNames);
        }

        final PrdOrgDataRefVO defaultOrgInfo = cacheUtil.getDefaultOrgInfoByUserId(vo.getVisitMemberId());
        // 拜访成员上级
        vo.setVisitMemberParentIdPro(defaultOrgInfo != null ? defaultOrgInfo.getParentId() : null);

        if (translationFileFlag) {
            vo.setFileDatas(fileUtil.getFileDatas(vo.getFileCodes()));
            vo.setSignFileDatas(fileUtil.getFileDatas(vo.getSignFileCodes()));
            vo.setSignOutFileDatas(fileUtil.getFileDatas(vo.getSignOutFileCodes()));

            // 客户信息
            final Long customerId = vo.getCustomerId();
            if(null == customerId){
                log.warn("销售拜访计划未关联客户；销售拜访计划：{}", vo);
                return;
            }
            final CrmCustomerSimpleVO simpleVO = customerService.querySimpleByKey(customerId);
            if(null != simpleVO){
                final Long bookId = simpleVO.getBookId();
                vo.setBookId(bookId);
                vo.setCompanyAddress(simpleVO.getCompanyAddress());
                vo.setLongitudeLatitude(simpleVO.getLongitudeLatitude());

                if(null == bookId){
                    log.warn("客户数据异常；找不到bookId;  customId:{}", customerId);
                    return;
                }
                // 联系人信息
                PrdAbContactsQuery prdAbContactsQuery = new PrdAbContactsQuery();
                prdAbContactsQuery.setBookId(bookId);
                final PagingVO<PrdAbContactsVO> paging = prdAbContactsService.paging(prdAbContactsQuery);
                if (paging.isNotEmpty()) {
                    final PrdAbContactsVO prdAbContactsVO = paging.getRecords().get(0);
                    vo.setContactsName(prdAbContactsVO.getContactsName());
                    vo.setContactsPhone(prdAbContactsVO.getContactsPhone());
                }
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public VisitTaskVO insert(VisitTaskPayload payload) {
        // 检查
        check(payload);
        // 数据处理
        dataProcess(payload);

        VisitTaskDO entityDo = VisitTaskConvert.INSTANCE.toDo(payload);
        final VisitTaskDO save = visitTaskRepo.save(entityDo);

        // 处理签到信息历史记录
        signDataProcess(payload, save);
        return VisitTaskConvert.INSTANCE.toVo(save);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public VisitTaskVO update(VisitTaskPayload payload) {
        final Long loginUserId = GlobalUtil.getLoginUserId();
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode()));
        if (ObjectUtils.isEmpty(payload.getId())) {
            throw TwException.error("", "id不能为空");
        }
        VisitTaskDO entity = visitTaskRepo.findById(payload.getId()).orElseGet(VisitTaskDO::new);
        Assert.notNull(entity.getId(), "拜访任务不存在");

        // 校验
        checkUpdate(payload, entity);

        if (ObjectUtils.isEmpty(payload.getPlanId())) {
            payload.setPlanId(entity.getPlanId());
        }
        dataProcess(payload);

        if (!ObjectUtils.isEmpty(payload.getStatus()) && payload.getStatus().equals(VisitTaskStatusEnum.OK.getCode())) {
            // 校验操作权限
            //拜访成员可执行、提交、保存任务；
            if (!(loginUserId.equals(entity.getVisitMemberId()) || rolePermission)) {
                throw TwException.error("", "权限不足");
            }

            payload.setSubmitTime(LocalDateTime.now());
            payload.setEvalStatus(VisitTaskEvalStatusEnum.NEW.getCode());

            // 判断销售拜访计划 类型为 常规、周期；
            if (payload.getPlanType().equals(VisitTaskPlanTypeEnum.general_plan)||payload.getPlanType().equals(VisitTaskPlanTypeEnum.periodic_plan)){
                // 先校验是否发过短信
                if (entity.getSendFlag()==null){
                    TemplateSmsDTO smsDTO  = new TemplateSmsDTO();

                    Set<String> mobile =new HashSet<>();
                    mobile.add("13849191916");

                    Map<String,String> map=new HashMap<>();
                    map.put("customer","Zach");

                    smsDTO.setTemplateCode("SMS_114300083");
                    smsDTO.setSignName("埃林哲");
                    smsDTO.setMobiles(mobile);

                    smsDTO.setSubject("Elitesland");
                    smsDTO.setBusinessType("CRM-TESTSEND");
                    smsDTO.setTemplateParams(map);
                    // 根据客户id 查询 手机号 和名称;
                    //取出手机号后校验
                    isValidPhoneNumber("13849191916");
                    //发送短信
                     smsRpcService.sendTemplateMsg(smsDTO);
                }
            }
            // 发送成功以后，标记此任务 sendFlag=1
            visitTaskDAO.updateSendFlag();
        }

        if (StringUtils.hasText(payload.getCustomerDesc())
            || StringUtils.hasText(payload.getResult())
            || null != payload.getNextVisitTime()
            || StringUtils.hasText(payload.getSignLocations())
            || StringUtils.hasText(payload.getSignOutLocations())
        ) {
            // 校验操作权限
            //拜访成员可执行、提交、保存任务；
            if (!(loginUserId.equals(entity.getVisitMemberId()) || rolePermission)) {
                throw TwException.error("", "权限不足");
            }
            if (ObjectUtils.isEmpty(payload.getStatus())) {
                payload.setStatus(VisitTaskStatusEnum.READY.getCode());
            }
        }

        VisitTaskDO entityDo = VisitTaskConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);

        final VisitTaskDO save = visitTaskRepo.save(entity);
        final Long planId = save.getPlanId();

        // 处理签到信息历史记录
        signDataProcess(payload, save);

        // 校验任务状态  某个计划下所有任务都提交后，计划状态修改为已完成
        // 更新拜访计划状态
        updatePlanStatus(planId);

        return VisitTaskConvert.INSTANCE.toVo(save);
    }
    //手机号校验
    public static boolean isValidPhoneNumber(String phoneNumber) {
        if ((phoneNumber != null) && (!phoneNumber.isEmpty())) {
            return Pattern.matches("/^1(3[0-9]|4[01456879]|5[0-35-9]|6[2567]|7[0-8]|8[0-9]|9[0-35-9])\\d{8}$/", phoneNumber);
        }
        return false;
    }

    /**
     * 处理签到信息历史记录
     *
     * @param payload 有效载荷
     * @param save    保存
     */
    private void signDataProcess(VisitTaskPayload payload, VisitTaskDO save) {
        final LocalDateTime now = LocalDateTime.now();
        if (!ObjectUtils.isEmpty(payload.getSignLocations())) {
            VisitSignRecordDO signRecordDO = new VisitSignRecordDO();
            signRecordDO.setTaskId(save.getId());
            signRecordDO.setPlanId(save.getPlanId());
            signRecordDO.setType(VisitSignTypeEnum.IN.getCode());
            signRecordDO.setTaskName(save.getTaskName());
            signRecordDO.setFileCodes(payload.getSignFileCodes());
            signRecordDO.setSignTime(now);
            signRecordDO.setLocations(payload.getSignLocations());
            signRecordDO.setAddress(payload.getSignAddress());
            visitSignRecordDAO.save(signRecordDO);
        }
        if (!ObjectUtils.isEmpty(payload.getSignOutLocations())) {
            VisitSignRecordDO signRecordDO = new VisitSignRecordDO();
            signRecordDO.setTaskId(save.getId());
            signRecordDO.setPlanId(save.getPlanId());
            signRecordDO.setType(VisitSignTypeEnum.OUT.getCode());
            signRecordDO.setTaskName(save.getTaskName());
            signRecordDO.setFileCodes(payload.getSignOutFileCodes());
            signRecordDO.setSignTime(now);
            signRecordDO.setLocations(payload.getSignOutLocations());
            signRecordDO.setAddress(payload.getSignOutAddress());
            visitSignRecordDAO.save(signRecordDO);
        }
    }

    /**
     * 根据拜访计划生成任务
     *
     * @param planId 计划id
     * @return {@link List}<{@link VisitTaskVO}>
     */
    @Override
    public List<VisitTaskVO> generateTaskByPlanId(Long planId) {
        List<VisitTaskDO> taskDOList = new ArrayList<>();
        if (null != planId) {
            final VisitPlanVO visitPlanVO = visitPlanDAO.queryByKey(planId);
            Assert.notNull(visitPlanVO, "拜访计划不存在");
            // 计划类型
            final String visitPlanType = visitPlanVO.getVisitPlanType();
            int visitTimesInt = 1;
            // 常规计划 只生成一次任务  其他类型判断 次数
            if (!VisitTaskPlanTypeEnum.general_plan.getCode().equals(visitPlanType)) {
                final String visitTimes = visitPlanVO.getVisitTimes();
                if (ObjectUtils.isEmpty(visitTimes)) {
                    throw TwException.error("", "拜访次数不能为空");
                }
                visitTimesInt = Integer.valueOf(visitTimes);
            }
            final Long customId = visitPlanVO.getCustomId();
            String customerName = "";
            if (!ObjectUtils.isEmpty(customId)) {
                // 查询客户名称
                final CrmCustomerVO crmCustomerVO = crmCustomerDAO.queryDetail(customId);
                Assert.notNull(crmCustomerVO, "客户不存在");
                customerName = crmCustomerVO.getCustomerName();
            }
            // 先清空 ，再插入
            deleteSoftByPlanId(planId);

            VisitPlanDetailQuery planDetailQuery = new VisitPlanDetailQuery();
            planDetailQuery.setVisitPlanId(planId);
            List<VisitPlanDetailVO> planDetailVOList = visitPlanDetailDAO.queryListDynamic(planDetailQuery);

            if (!CollectionUtils.isEmpty(planDetailVOList)) {
                // 查出协访人员
                //final String teamMemberIds = planDetailVOList.stream()
                //    .filter(visitPlanDetailVO -> visitPlanDetailVO.getVisitPersonType().equals(VisitTaskPlanPersonType.accompany_person.getCode()))
                //    .distinct()
                //    .map(visitPlanDetailVO -> visitPlanDetailVO.getVisitPersonId() + "")
                //    .collect(Collectors.joining(","));

                // 只取拜访成员
                planDetailVOList = planDetailVOList.stream()
                    .filter(visitPlanDetailVO -> visitPlanDetailVO.getVisitPersonType().equals(VisitTaskPlanPersonTypeEnum.visit_person.getCode()))
                    .distinct()
                    .collect(Collectors.toList());
                int i = 1;
                // 根据计划类型 判断生成任务数据  计划名称-拜访成员-任务编号
                for (VisitPlanDetailVO planDetailVO : planDetailVOList) {
                    // 拜访次数
                    for (int j = 0; j < visitTimesInt; j++) {
                        VisitTaskDO taskDO = new VisitTaskDO();
                        final String visitPlanName = visitPlanVO.getVisitPlanName();
                        taskDO.setPlanId(planId);
                        taskDO.setPlanName(visitPlanName);
                        taskDO.setPlanType(visitPlanType);
                        taskDO.setObjId(visitPlanVO.getObjId());
                        taskDO.setObjName(visitPlanVO.getObjName());
                        taskDO.setObjType(visitPlanVO.getObjType());
                        taskDO.setStartTime(visitPlanVO.getVisitDateFrom());
                        taskDO.setEndTime(visitPlanVO.getVisitDateTo());
                        taskDO.setPlanCreateUserId(visitPlanVO.getCreateUserId());
                        taskDO.setStatus(VisitTaskStatusEnum.NEW.getCode());
                        taskDO.setCustomerId(customId);
                        taskDO.setCustomerName(customerName);

                        taskDO.setTeamMemberIds(visitPlanVO.getExtString2());

                        final Long visitPersonId = planDetailVO.getVisitPersonId();
                        taskDO.setVisitMemberId(visitPersonId);
                        final String userName = cacheUtil.getUserName(visitPersonId);
                        taskDO.setVisitMemberName(userName);

                        final PrdOrgDataRefVO defaultOrgInfo = cacheUtil.getDefaultOrgInfoByUserId(visitPersonId);
                        // 直属上级
                        taskDO.setVisitMemberParentId(defaultOrgInfo != null ? defaultOrgInfo.getParentId() : null);

                        // 计划名称-拜访成员-任务编号
                        taskDO.setTaskName(visitPlanName + "-" + userName + "-" + i);
                        i++;

                        taskDOList.add(taskDO);
                    }
                }
            }

        }
        if (!CollectionUtils.isEmpty(taskDOList)) {
            visitTaskDAO.saveAll(taskDOList);
        }
        return VisitTaskConvert.INSTANCE.toVoList(taskDOList);
    }

    /**
     * 更新拜访计划状态
     *
     * @param planId 计划id
     */
    private void updatePlanStatus(Long planId) {
        VisitTaskQuery visitTaskQuery = new VisitTaskQuery();
        visitTaskQuery.setPlanId(planId);
        visitTaskQuery.setStatus(VisitTaskStatusEnum.OK.getCode());
        final long count = visitTaskDAO.count(visitTaskQuery);

        VisitTaskQuery visitTaskQueryAll = new VisitTaskQuery();
        visitTaskQueryAll.setPlanId(planId);
        final long countAll = visitTaskDAO.count(visitTaskQueryAll);

        if (count == countAll) {
            // 更新计划状态
            visitPlanDAO.completeVisitPlan(planId);
        }
    }

    /**
     * 数据处理
     *
     * @param payload 有效载荷
     */
    private void dataProcess(VisitTaskPayload payload) {
        final VisitPlanVO visitPlanVO = visitPlanDAO.queryByKey(payload.getPlanId());
        Assert.notNull(visitPlanVO, "拜访计划不存在");

        payload.setPlanName(visitPlanVO.getVisitPlanName());
        payload.setPlanType(visitPlanVO.getVisitPlanType());
        payload.setObjId(visitPlanVO.getObjId());
        payload.setObjName(visitPlanVO.getObjName());
        payload.setObjType(visitPlanVO.getObjType());
        payload.setStartTime(visitPlanVO.getVisitDateFrom());
        payload.setEndTime(visitPlanVO.getVisitDateTo());
        payload.setPlanCreateUserId(visitPlanVO.getCreateUserId());
        // 访销计划 计划里没有客户信息 拜访任务里维护客户信息
        Long customId = payload.getCustomerId();
        if (ObjectUtils.isEmpty(customId)) {
            customId = visitPlanVO.getCustomId();
            payload.setCustomerId(customId);
        }
        if (!ObjectUtils.isEmpty(customId)) {
            // 查询客户名称
            final CrmCustomerVO crmCustomerVO = crmCustomerDAO.queryDetail(customId);
            Assert.notNull(crmCustomerVO, "客户不存在");
            payload.setCustomerName(crmCustomerVO.getCustomerName());
        }

        // 新建状态
        if (ObjectUtils.isEmpty(payload.getStatus()) && null == payload.getId()) {
            payload.setStatus(VisitTaskStatusEnum.NEW.getCode());
        }
        if (!ObjectUtils.isEmpty(payload.getSignLocations())) {
            if (null == payload.getSignTime()) {
                payload.setSignTime(LocalDateTime.now());
            }
        }
        if (!ObjectUtils.isEmpty(payload.getSignOutLocations())) {
            if (null == payload.getSignOutTime()) {
                payload.setSignOutTime(LocalDateTime.now());
            }
        }
    }

    /**
     * 检查
     *
     * @param payload 有效载荷
     */
    private void check(VisitTaskPayload payload) {
        if (ObjectUtils.isEmpty(payload.getPlanId())) {
            throw TwException.error("", "planId不能为空");
        }
    }

    /**
     * 检查更新
     *
     * @param payload 有效载荷
     * @param entity  实体
     */
    private void checkUpdate(VisitTaskPayload payload, VisitTaskDO entity) {
        // 已提交的任务不可变更
        if (entity.getStatus().equals(VisitTaskStatusEnum.OK.getCode())) {
            throw TwException.error("", "已提交的任务不可变更");
        }
        // 未开始/待提交的任务可变更执行信息-客户情况说明、拜访进展与结果、下次拜访时间

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        final Long loginUserId = GlobalUtil.getLoginUserId();
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode()));
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<VisitTaskDO> optional = visitTaskRepo.findById(id);
                if (!optional.isEmpty()) {
                    VisitTaskDO entity = optional.get();
                    //计划创建人、拜访成员可删除任务，拜访成员可执行、提交、保存任务；
                    if (entity.getPlanCreateUserId().equals(loginUserId) || entity.getVisitMemberId().equals(loginUserId) || rolePermission) {
                        // 校验状态
                        if (!entity.getStatus().equals(VisitTaskStatusEnum.NEW.getCode())) {
                            throw TwException.error("", "[" + entity.getTaskName() + "]当前状态不允许删除");
                        }

                        entity.setDeleteFlag(1);
                        visitTaskRepo.save(entity);
                    } else {
                        throw TwException.error("", "[" + entity.getTaskName() + "]无权限删除");
                    }
                }
            });
        }
    }

    /**
     * 根据计划删除软id
     *
     * @param planId 计划id
     */
    @Override
    public void deleteSoftByPlanId(Long planId) {
        if (null != planId) {
            final long count = countByPlanId(planId);
            if (count > 0) {
                throw TwException.error("", "任务状态不允许删除");
            }
            visitTaskDAO.deleteSoftByPlanId(planId);
        }
    }

    /**
     * 统计任务数量 ：拜访计划下 任务状态为【已完成、待提交】的数据
     *
     * @param planId 计划id
     * @return long
     */
    @Override
    public long countByPlanId(Long planId) {
        VisitTaskQuery taskQuery = new VisitTaskQuery();
        taskQuery.setPlanId(planId);
        List<String> statusList = new ArrayList<>();
        statusList.add(VisitTaskStatusEnum.OK.getCode());
        statusList.add(VisitTaskStatusEnum.READY.getCode());
        taskQuery.setStatusList(statusList);
        return visitTaskDAO.count(taskQuery);
    }

    /**
     * 客户评价
     *
     * @param openPayload 开放有效载荷
     * @return {@link VisitTaskOpenPayload}
     */
    @Override
    public VisitTaskOpenPayload customerEval(VisitTaskOpenPayload openPayload) {
        VisitTaskDO entity = visitTaskRepo.findById(openPayload.getId()).orElseGet(VisitTaskDO::new);
        Assert.notNull(entity.getId(), "拜访不存在");

        VisitTaskPayload payload = new VisitTaskPayload();
        payload.setId(openPayload.getId());
        payload.setCustomerGradle(openPayload.getCustomerGradle());
        payload.setCustomerAdvise(openPayload.getCustomerAdvise());
        VisitTaskDO entityDo = VisitTaskConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        visitTaskRepo.save(entity);
        return openPayload;
    }

    /**
     * 领导评价
     *
     * @param payload 有效载荷
     * @return {@link VisitTaskVO}
     */
    @Override
    public VisitTaskVO leaderEval(VisitTaskPayload payload) {
        VisitTaskDO entity = visitTaskRepo.findById(payload.getId()).orElseGet(VisitTaskDO::new);
        Assert.notNull(entity.getId(), "拜访不存在");
        VisitTaskPayload update = new VisitTaskPayload();
        update.setId(payload.getId());
        update.setLeaderGradle(payload.getLeaderGradle());
        update.setEvalStatus(VisitTaskEvalStatusEnum.OK.getCode());
        update.setNextPlan(payload.getNextPlan());
        VisitTaskDO entityDo = VisitTaskConvert.INSTANCE.toDo(update);
        entity.copy(entityDo);
        final VisitTaskDO save = visitTaskRepo.save(entity);
        return VisitTaskConvert.INSTANCE.toVo(save);
    }

    /**
     * 统计
     *
     * @param query 查询
     * @return long
     */
    @Override
    public long count(VisitTaskQuery query){
        return visitTaskDAO.count(query);
    }

}
