package com.elitesland.tw.tw5pms.server.project.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5pms.api.project.payload.PmsProjectPlanPayload;
import com.elitesland.tw.tw5pms.api.project.query.PmsProjectPlanQuery;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectPlanSimpleVO;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectPlanVO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsProjectPlanDO;
import com.elitesland.tw.tw5pms.server.project.entity.QPmsProjectPlanDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsProjectPlanRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 项目计划
 *
 * @author carl
 * @date 2023-04-17
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectPlanDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectPlanRepo repo;
    private final QPmsProjectPlanDO qdo = QPmsProjectPlanDO.pmsProjectPlanDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectPlanVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectPlanVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                // 父主键
                qdo.parentId,
                // 计划编码
                qdo.planCode,
                // 父计划编码
                qdo.parentPlanCode,
                qdo.nodeCode,
                // 计划类型（STAGE,ACT,TASK）
                qdo.planType,
                // 计划描述（名称）
                qdo.planName,
                qdo.planProgress,
                // 所属资源id（活动，任务）
                qdo.sourceId,
                // 开始时间
                qdo.startDate,
                // 结束时间
                qdo.endDate,
                // 持续时间（天）
                qdo.durationDay,
                // 负责人
                qdo.managerUserId,
                // 状态
                qdo.sourceStatus,
                // 前置依赖资源ids
                qdo.relySourceIds,
                // 前置依赖计划名称ids
                qdo.relyPlanIds,
                // 前置依赖计划名称
                qdo.relyPlanNames,
                // 前置类型（FS）
                qdo.relyType,
                // 延迟天数
                qdo.relyDay,
                //是否是关键路径
                qdo.isKeyPath
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectPlanVO> getJpaQueryWhere(PmsProjectPlanQuery query) {
        JPAQuery<PmsProjectPlanVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectPlanQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectPlanQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** 父主键 精确 */
        if (!ObjectUtils.isEmpty(query.getParentId())) {
            list.add(qdo.parentId.eq(query.getParentId()));
        }

        /** 父计划编码 模糊 */
        if (!ObjectUtils.isEmpty(query.getParentPlanCode())) {
            list.add(qdo.parentPlanCode.like(SqlUtil.toSqlLikeString(query.getParentPlanCode())));
        }
        if (!ObjectUtils.isEmpty(query.getPlanTypes())) {
            List<String> planTypes = Arrays.asList(query.getPlanTypes());
            list.add(qdo.planType.in(planTypes));
        } else {
            /** 计划类型（STAGE,ACT,TASK） 精确 */
            if (!ObjectUtils.isEmpty(query.getPlanType())) {
                list.add(qdo.planType.eq(query.getPlanType()));
            }
        }

        /** 计划描述（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getPlanName())) {
            list.add(qdo.planName.like(SqlUtil.toSqlLikeString(query.getPlanName())));
        }
        /** 父计划编码 模糊 */
        if (!ObjectUtils.isEmpty(query.getPlanSearch())) {
            list.add(qdo.planName.like(SqlUtil.toSqlLikeString(query.getPlanSearch())).or(qdo.nodeCode.like(SqlUtil.toSqlLikeString(query.getPlanSearch()))));
        }
        /** 所属资源id（活动，任务） 精确 */
        if (!ObjectUtils.isEmpty(query.getSourceId())) {
            list.add(qdo.sourceId.eq(query.getSourceId()));
        }
        /** 持续时间（天） 精确 */
        if (!ObjectUtils.isEmpty(query.getDurationDay())) {
            list.add(qdo.durationDay.eq(query.getDurationDay()));
        }
        /** 负责人 精确 */
        if (!ObjectUtils.isEmpty(query.getManagerUserId())) {
            list.add(qdo.managerUserId.eq(query.getManagerUserId()));
        }
        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getSourceStatus())) {
            list.add(qdo.sourceStatus.eq(query.getSourceStatus()));
        }

        /** 前置依赖计划名称ids 模糊 */
        if (!ObjectUtils.isEmpty(query.getRelyPlanIds())) {
            list.add(qdo.relyPlanIds.like(SqlUtil.toSqlLikeString(query.getRelyPlanIds())));
        }
        /** 前置依赖计划名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getRelyPlanNames())) {
            list.add(qdo.relyPlanNames.like(SqlUtil.toSqlLikeString(query.getRelyPlanNames())));
        }
        /** 前置类型（FS） 精确 */
        if (!ObjectUtils.isEmpty(query.getRelyType())) {
            list.add(qdo.relyType.eq(query.getRelyType()));
        }
        /** 延迟天数 精确 */
        if (!ObjectUtils.isEmpty(query.getRelyDay())) {
            list.add(qdo.relyDay.eq(query.getRelyDay()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectPlanVO queryByKey(Long id) {
        JPAQuery<PmsProjectPlanVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据父主键及类型查询
     *
     * @param parentIds 父主键
     * @param type      计划类型
     * @return 结果
     */
    public List<PmsProjectPlanVO> queryByParentIdAndType(List<Long> parentIds, String type) {
        JPAQuery<PmsProjectPlanVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.parentId.in(parentIds));
        jpaQuery.where(qdo.planType.eq(type));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据project主键查询
     *
     * @param projectId project主键
     * @return 结果
     */
    public List<PmsProjectPlanVO> queryByProjectId(Long projectId) {
        JPAQuery<PmsProjectPlanVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据project主键查询
     *
     * @param projectId project主键
     * @return 结果
     */
    public List<PmsProjectPlanSimpleVO> querySimpleByProjectId(Long projectId) {
        JPAQuery<PmsProjectPlanSimpleVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PmsProjectPlanSimpleVO.class,
                qdo.id,
                // 计划描述（名称）
                qdo.planName,
                // 开始时间
                qdo.startDate,
                // 结束时间
                qdo.endDate,
                // 持续时间（天）
                qdo.durationDay
        )).from(qdo);
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }


    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectPlanVO> queryListDynamic(PmsProjectPlanQuery query) {
        JPAQuery<PmsProjectPlanVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectPlanVO> queryPaging(PmsProjectPlanQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectPlanVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectPlanVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectPlanVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectPlanDO save(PmsProjectPlanDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectPlanDO> saveAll(List<PmsProjectPlanDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 修改关键路径状态
     *
     * @param ids
     * @param isKeyPath
     * @return
     */
    public long updateKeyPathByIds(List<Long> ids, Boolean isKeyPath) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.in(ids)).set(qdo.isKeyPath, isKeyPath);

        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectPlanPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 项目主键
        if (payload.getProjectId() != null) {
            update.set(qdo.projectId, payload.getProjectId());
        }
        // 父主键
        if (payload.getParentId() != null) {
            update.set(qdo.parentId, payload.getParentId());
        }
        // 计划编码
        if (payload.getPlanCode() != null) {
            update.set(qdo.planCode, payload.getPlanCode());
        }
        // 父计划编码
        if (payload.getParentPlanCode() != null) {
            update.set(qdo.parentPlanCode, payload.getParentPlanCode());
        }
        // 计划类型（STAGE,ACT,TASK）
        if (payload.getPlanType() != null) {
            update.set(qdo.planType, payload.getPlanType());
        }
        // 计划描述（名称）
        if (payload.getPlanName() != null) {
            update.set(qdo.planName, payload.getPlanName());
        }
        // 所属资源id（活动，任务）
        if (payload.getSourceId() != null) {
            update.set(qdo.sourceId, payload.getSourceId());
        }
        // 持续时间（天）
        if (payload.getDurationDay() != null) {
            update.set(qdo.durationDay, payload.getDurationDay());
        }
        // 负责人
        if (payload.getManagerUserId() != null) {
            update.set(qdo.managerUserId, payload.getManagerUserId());
        }
        // 状态
        if (payload.getSourceStatus() != null) {
            update.set(qdo.sourceStatus, payload.getSourceStatus());
        }

        // 前置依赖计划名称ids
        if (payload.getRelyPlanIds() != null) {
            update.set(qdo.relyPlanIds, payload.getRelyPlanIds());
        }
        // 前置依赖计划名称
        if (payload.getRelyPlanNames() != null) {
            update.set(qdo.relyPlanNames, payload.getRelyPlanNames());
        }
        // 前置类型（FS）
        if (payload.getRelyType() != null) {
            update.set(qdo.relyType, payload.getRelyType());
        }
        // 延迟天数
        if (payload.getRelyDay() != null) {
            update.set(qdo.relyDay, payload.getRelyDay());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 项目主键
            if (nullFields.contains("projectId")) {
                update.setNull(qdo.projectId);
            }
            // 父主键
            if (nullFields.contains("parentId")) {
                update.setNull(qdo.parentId);
            }
            // 计划编码
            if (nullFields.contains("planCode")) {
                update.setNull(qdo.planCode);
            }
            // 父计划编码
            if (nullFields.contains("parentPlanCode")) {
                update.setNull(qdo.parentPlanCode);
            }
            // 计划类型（STAGE,ACT,TASK）
            if (nullFields.contains("planType")) {
                update.setNull(qdo.planType);
            }
            // 计划描述（名称）
            if (nullFields.contains("planName")) {
                update.setNull(qdo.planName);
            }
            // 所属资源id（活动，任务）
            if (nullFields.contains("sourceId")) {
                update.setNull(qdo.sourceId);
            }
            // 持续时间（天）
            if (nullFields.contains("durationDay")) {
                update.setNull(qdo.durationDay);
            }
            // 负责人
            if (nullFields.contains("managerUserId")) {
                update.setNull(qdo.managerUserId);
            }
            // 状态
            if (nullFields.contains("sourceStatus")) {
                update.setNull(qdo.sourceStatus);
            }

            // 前置依赖计划名称ids
            if (nullFields.contains("relyPlanIds")) {
                update.setNull(qdo.relyPlanIds);
            }
            // 前置依赖计划名称
            if (nullFields.contains("relyPlanNames")) {
                update.setNull(qdo.relyPlanNames);
            }
            // 前置类型（FS）
            if (nullFields.contains("relyType")) {
                update.setNull(qdo.relyType);
            }
            // 延迟天数
            if (nullFields.contains("relyDay")) {
                update.setNull(qdo.relyDay);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

