package com.elitesland.tw.tw5pms.server.project.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5pms.api.project.payload.PmsProjectPlanSnapshotPayload;
import com.elitesland.tw.tw5pms.api.project.query.PmsProjectPlanSnapshotQuery;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectPlanSnapshotVO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsProjectPlanSnapshotDO;
import com.elitesland.tw.tw5pms.server.project.entity.QPmsProjectPlanSnapshotDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsProjectPlanSnapshotRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目计划快照
 *
 * @author carl
 * @date 2023-04-17
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectPlanSnapshotDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectPlanSnapshotRepo repo;
    private final QPmsProjectPlanSnapshotDO qdo = QPmsProjectPlanSnapshotDO.pmsProjectPlanSnapshotDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectPlanSnapshotVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectPlanSnapshotVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                // 快照（名称）
                qdo.snapName,
                // 快照内容
                qdo.snapContent
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectPlanSnapshotVO> getJpaQueryWhere(PmsProjectPlanSnapshotQuery query) {
        JPAQuery<PmsProjectPlanSnapshotVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectPlanSnapshotQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectPlanSnapshotQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** 快照（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getSnapName())) {
            list.add(qdo.snapName.like(SqlUtil.toSqlLikeString(query.getSnapName())));
        }
        /** 快照内容 模糊 */
        if (!ObjectUtils.isEmpty(query.getSnapContent())) {
            list.add(qdo.snapContent.like(SqlUtil.toSqlLikeString(query.getSnapContent())));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectPlanSnapshotVO queryByKey(Long id) {
        JPAQuery<PmsProjectPlanSnapshotVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectPlanSnapshotVO> queryListDynamic(PmsProjectPlanSnapshotQuery query) {
        JPAQuery<PmsProjectPlanSnapshotVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectPlanSnapshotVO> queryPaging(PmsProjectPlanSnapshotQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectPlanSnapshotVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectPlanSnapshotVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectPlanSnapshotVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectPlanSnapshotDO save(PmsProjectPlanSnapshotDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectPlanSnapshotDO> saveAll(List<PmsProjectPlanSnapshotDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectPlanSnapshotPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 项目主键
        if (payload.getProjectId() != null) {
            update.set(qdo.projectId, payload.getProjectId());
        }
        // 快照（名称）
        if (payload.getSnapName() != null) {
            update.set(qdo.snapName, payload.getSnapName());
        }
        // 快照内容
        if (payload.getSnapContent() != null) {
            update.set(qdo.snapContent, payload.getSnapContent());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 项目主键
            if (nullFields.contains("projectId")) {
                update.setNull(qdo.projectId);
            }
            // 快照（名称）
            if (nullFields.contains("snapName")) {
                update.setNull(qdo.snapName);
            }
            // 快照内容
            if (nullFields.contains("snapContent")) {
                update.setNull(qdo.snapContent);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

