package com.elitesland.tw.tw5pms.server.project.dao;

import com.elitesland.tw.tw5pms.api.project.payload.PmsProjectTeamBasePayload;
import com.elitesland.tw.tw5pms.api.project.query.PmsProjectTeamBaseQuery;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectTeamBaseVO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsProjectTeamBaseDO;
import com.elitesland.tw.tw5pms.server.project.entity.QPmsProjectTeamBaseDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsProjectTeamBaseRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目成员基本信息
 *
 * @author Echo
 * @date 2023-04-11
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectTeamBaseDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectTeamBaseRepo repo;
    private final QPmsProjectTeamBaseDO qdo = QPmsProjectTeamBaseDO.pmsProjectTeamBaseDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectTeamBaseVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectTeamBaseVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                // 开始周
                qdo.startDate,
                // 持续周
                qdo.continueWeek,
                // 结束周
                qdo.stopDate
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectTeamBaseVO> getJpaQueryWhere(PmsProjectTeamBaseQuery query) {
        JPAQuery<PmsProjectTeamBaseVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectTeamBaseQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectTeamBaseQuery query){
        List<Predicate> list = new ArrayList<>();
        /** id 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** 开始周 精确 */
        if (!ObjectUtils.isEmpty(query.getStartDate())) {
            list.add(qdo.startDate.eq(query.getStartDate()));
        }
        /** 持续周 精确 */
        if (!ObjectUtils.isEmpty(query.getContinueWeek())) {
            list.add(qdo.continueWeek.eq(query.getContinueWeek()));
        }
        /** 结束周 精确 */
        if (!ObjectUtils.isEmpty(query.getStopDate())) {
            list.add(qdo.stopDate.eq(query.getStopDate()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectTeamBaseVO queryByKey(Long id) {
        JPAQuery<PmsProjectTeamBaseVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectTeamBaseVO> queryListDynamic(PmsProjectTeamBaseQuery query) {
        JPAQuery<PmsProjectTeamBaseVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectTeamBaseVO> queryPaging(PmsProjectTeamBaseQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PmsProjectTeamBaseVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectTeamBaseVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectTeamBaseVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectTeamBaseDO save(PmsProjectTeamBaseDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectTeamBaseDO> saveAll(List<PmsProjectTeamBaseDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectTeamBasePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // id
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 项目主键
        if (payload.getProjectId() != null) {
            update.set(qdo.projectId, payload.getProjectId());
        }
        // 开始周
        if (payload.getStartDate() != null) {
            update.set(qdo.startDate, payload.getStartDate());
        }
        // 持续周
        if (payload.getContinueWeek() != null) {
            update.set(qdo.continueWeek, payload.getContinueWeek());
        }
        // 结束周
        if (payload.getStopDate() != null) {
            update.set(qdo.stopDate, payload.getStopDate());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // id
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 项目主键
            if (nullFields.contains("projectId")) {
                update.setNull(qdo.projectId);
            }
            // 开始周
            if (nullFields.contains("startDate")) {
                update.setNull(qdo.startDate);
            }
            // 持续周
            if (nullFields.contains("continueWeek")) {
                update.setNull(qdo.continueWeek);
            }
            // 结束周
            if (nullFields.contains("stopDate")) {
                update.setNull(qdo.stopDate);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

