package com.elitesland.tw.tw5pms.server.project.dao;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5pms.api.project.payload.PmsProjectTeamMemberPayload;
import com.elitesland.tw.tw5pms.api.project.query.PmsProjectTeamMemberQuery;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectTeamMemberVO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsProjectTeamMemberDO;
import com.elitesland.tw.tw5pms.server.project.entity.QPmsProjectTeamMemberDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsProjectTeamMemberRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目成员资质信息
 *
 * @author Echo
 * @date 2023-04-11
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectTeamMemberDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectTeamMemberRepo repo;
    private final QPmsProjectTeamMemberDO qdo = QPmsProjectTeamMemberDO.pmsProjectTeamMemberDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectTeamMemberVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectTeamMemberVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 基本信息主键
                qdo.baseId,
                // 工种
                qdo.workType,
                // 资源
                qdo.userId,
                // 专业级别
                qdo.professionalLevel,
                // 预计开始日期
                qdo.startDate,
                // 预计结束日期
                qdo.stopDate,
                // 总人天
                qdo.totalDays,
                // week_data
                qdo.weekData
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectTeamMemberVO> getJpaQueryWhere(PmsProjectTeamMemberQuery query) {
        JPAQuery<PmsProjectTeamMemberVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectTeamMemberQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectTeamMemberQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** id 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 基本信息主键 精确 */
        if (!ObjectUtils.isEmpty(query.getBaseId())) {
            list.add(qdo.baseId.eq(query.getBaseId()));
        }
        /** 工种 精确 */
        if (!ObjectUtils.isEmpty(query.getWorkType())) {
            list.add(qdo.workType.eq(query.getWorkType()));
        }
        /** 资源 精确 */
        if (!ObjectUtils.isEmpty(query.getUserId())) {
            list.add(qdo.userId.eq(query.getUserId()));
        }
        /** 专业级别 精确 */
        if (!ObjectUtils.isEmpty(query.getProfessionalLevel())) {
            list.add(qdo.professionalLevel.eq(query.getProfessionalLevel()));
        }
        /** 预计开始日期 精确 */
        if (!ObjectUtils.isEmpty(query.getStartDate())) {
            list.add(qdo.startDate.eq(query.getStartDate()));
        }
        /** 预计结束日期 精确 */
        if (!ObjectUtils.isEmpty(query.getStopDate())) {
            list.add(qdo.stopDate.eq(query.getStopDate()));
        }
        /** 总人天 精确 */
        if (!ObjectUtils.isEmpty(query.getTotalDays())) {
            list.add(qdo.totalDays.eq(query.getTotalDays()));
        }
        /** week_data 精确 */
        if (!ObjectUtils.isEmpty(query.getWeekData())) {
            list.add(qdo.weekData.eq(query.getWeekData()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectTeamMemberVO queryByKey(Long id) {
        JPAQuery<PmsProjectTeamMemberVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectTeamMemberVO> queryListDynamic(PmsProjectTeamMemberQuery query) {
        JPAQuery<PmsProjectTeamMemberVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectTeamMemberVO> queryPaging(PmsProjectTeamMemberQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectTeamMemberVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectTeamMemberVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectTeamMemberVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectTeamMemberDO save(PmsProjectTeamMemberDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectTeamMemberDO> saveAll(List<PmsProjectTeamMemberDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectTeamMemberPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // id
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 基本信息主键
        if (payload.getBaseId() != null) {
            update.set(qdo.baseId, payload.getBaseId());
        }
        // 工种
        if (payload.getWorkType() != null) {
            update.set(qdo.workType, payload.getWorkType());
        }
        // 资源
        if (payload.getUserId() != null) {
            update.set(qdo.userId, payload.getUserId());
        }
        // 专业级别
        if (payload.getProfessionalLevel() != null) {
            update.set(qdo.professionalLevel, payload.getProfessionalLevel());
        }
        // 预计开始日期
        if (payload.getStartDate() != null) {
            update.set(qdo.startDate, payload.getStartDate());
        }
        // 预计结束日期
        if (payload.getStopDate() != null) {
            update.set(qdo.stopDate, payload.getStopDate());
        }
        // 总人天
        if (payload.getTotalDays() != null) {
            update.set(qdo.totalDays, payload.getTotalDays());
        }
        // weeks
        if (payload.getWeeks() != null) {
            //list -> Json
            JSONArray jsonArray = JSON.parseArray(JSON.toJSONString(payload.getWeeks()));
            update.set(qdo.weekData, jsonArray.toString());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // id
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 基本信息主键
            if (nullFields.contains("baseId")) {
                update.setNull(qdo.baseId);
            }
            // 工种
            if (nullFields.contains("workType")) {
                update.setNull(qdo.workType);
            }
            // 资源
            if (nullFields.contains("userId")) {
                update.setNull(qdo.userId);
            }
            // 专业级别
            if (nullFields.contains("professionalLevel")) {
                update.setNull(qdo.professionalLevel);
            }
            // 预计开始日期
            if (nullFields.contains("startDate")) {
                update.setNull(qdo.startDate);
            }
            // 预计结束日期
            if (nullFields.contains("stopDate")) {
                update.setNull(qdo.stopDate);
            }
            // 总人天
            if (nullFields.contains("totalDays")) {
                update.setNull(qdo.totalDays);
            }
            // week_data
            if (nullFields.contains("weekData")) {
                update.setNull(qdo.weekData);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据用户id（资源）和基本信息表id 查找
     *
     * @param payload 单据数据
     */
    public PmsProjectTeamMemberVO queryByUserIdAndBaseId(PmsProjectTeamMemberPayload payload) {
        JPAQuery<PmsProjectTeamMemberVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.userId.eq(payload.getUserId()));
        jpaQuery.where(qdo.baseId.eq(payload.getBaseId()));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据项目主键查询所有项目成员
     *
     * @param projectId 项目主键
     */
    public List<PmsProjectTeamMemberVO> queryByProjectId(Long projectId) {
        JPAQuery<PmsProjectTeamMemberVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId).or(qdo.baseId.eq(projectId)));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }


}

