package com.elitesland.tw.tw5pms.server.project.dao;

import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5pms.api.project.query.PmsProjectWbsQuery;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectWbsVO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsProjectWbsDO;
import com.elitesland.tw.tw5pms.server.project.entity.QPmsProjectWbsDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsProjectWbsRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目wbs表
 *
 * @author carl
 * @date 2023-04-06
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectWbsDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectWbsRepo repo;
    private final QPmsProjectWbsDO qdo = QPmsProjectWbsDO.pmsProjectWbsDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectWbsVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectWbsVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                // 父主键
                qdo.parentId,
                // wbs描述（名称）
                qdo.wbsName,
                // wbs类型(WBS,NET,ACT,MS)
                qdo.wbsType,
                // wbs编码
                qdo.wbsCode,
                // node码
                qdo.nodeCode,
                // 父节点Wbs码
                qdo.parentWbsCode,
                // 成本计划，勾选为1
                qdo.costPlan,
                // 科目分配，勾选为1
                qdo.subjectDist,
                // 开票属性，勾选为1
                qdo.invoiceAttr,
                // 持续时间（天）
                qdo.durationDay,
                // 活动关联任务id
                qdo.actTaskId,
                // 活动成本类型
                qdo.actCostType
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectWbsVO> getJpaQueryWhere(PmsProjectWbsQuery query) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }


    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectWbsQuery query) {
        List<Predicate> list = new ArrayList<>();

        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** wbs描述（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsSearch())) {
            list.add(qdo.wbsName.like(SqlUtil.toSqlLikeString(query.getWbsSearch())).or(qdo.nodeCode.like(SqlUtil.toSqlLikeString(query.getWbsSearch()))));
        }
        /** wbs描述（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsName())) {
            list.add(qdo.wbsName.like(SqlUtil.toSqlLikeString(query.getWbsName())));
        }
        /** wbs类型(WBS,NET,ACT,MS) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsType())) {
            list.add(qdo.wbsType.eq(query.getWbsType()));
        }
        /** wbs编码 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsCode())) {
            list.add(qdo.wbsCode.like(SqlUtil.toSqlLikeString(query.getWbsCode())));
        }
        /** node码 精确 */
        if (!ObjectUtils.isEmpty(query.getNodeCode())) {
            list.add(qdo.nodeCode.eq(query.getNodeCode()));
        }
        /** 父节点Wbs码 精确 */
        if (!ObjectUtils.isEmpty(query.getParentWbsCode())) {
            list.add(qdo.parentWbsCode.eq(query.getParentWbsCode()));
        }
        /** 成本计划，勾选为1 精确 */
        if (!ObjectUtils.isEmpty(query.getCostPlan())) {
            list.add(qdo.costPlan.eq(query.getCostPlan()));
        }
        /** 科目分配，勾选为1 精确 */
        if (!ObjectUtils.isEmpty(query.getSubjectDist())) {
            list.add(qdo.subjectDist.eq(query.getSubjectDist()));
        }
        /** 开票属性，勾选为1 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceAttr())) {
            list.add(qdo.invoiceAttr.eq(query.getInvoiceAttr()));
        }
        /** 持续时间（天） 精确 */
        if (!ObjectUtils.isEmpty(query.getDurationDay())) {
            list.add(qdo.durationDay.eq(query.getDurationDay()));
        }
        /** 活动关联任务id 精确 */
        if (!ObjectUtils.isEmpty(query.getActTaskId())) {
            list.add(qdo.actTaskId.eq(query.getActTaskId()));
        }
        /** 活动成本类型 精确 */
        if (!ObjectUtils.isEmpty(query.getActCostType())) {
            list.add(qdo.actCostType.eq(query.getActCostType()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectWbsVO queryByKey(Long id) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据project主键查询
     *
     * @param projectId project主键
     * @return 结果
     */
    public List<PmsProjectWbsVO> queryByProjectId(Long projectId) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectWbsVO> queryListDynamic(PmsProjectWbsQuery query) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }


    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectWbsDO save(PmsProjectWbsDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectWbsDO> saveAll(List<PmsProjectWbsDO> dos) {
        return repo.saveAll(dos);
    }


    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

