package com.elitesland.tw.tw5pms.server.project.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5pms.api.project.payload.PmsWbsTemplatePayload;
import com.elitesland.tw.tw5pms.api.project.query.PmsWbsTemplateQuery;
import com.elitesland.tw.tw5pms.api.project.vo.PmsWbsTemplateVO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsWbsTemplateDO;
import com.elitesland.tw.tw5pms.server.project.entity.QPmsWbsTemplateDO;
import com.elitesland.tw.tw5pms.server.project.entity.QPmsWbsTemplateNodeDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsWbsTemplateRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * wbs模板
 *
 * @author Echo
 * @date 2023-03-29
 */
@Repository
@RequiredArgsConstructor
public class PmsWbsTemplateDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsWbsTemplateRepo repo;
    private final QPmsWbsTemplateDO qdo = QPmsWbsTemplateDO.pmsWbsTemplateDO;
    private final QPmsWbsTemplateNodeDO qndo = QPmsWbsTemplateNodeDO.pmsWbsTemplateNodeDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsTemplateVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsWbsTemplateVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 模板名称
                qdo.templateName,
                // 适用项目类型
                qdo.suitProjectType,
                // 模板状态，0：激活，1：暂挂
                qdo.templateStatus
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsTemplateVO> getJpaQueryWhere(PmsWbsTemplateQuery query) {
        JPAQuery<PmsWbsTemplateVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsWbsTemplateQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsWbsTemplateQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 模板名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getTemplateName())) {
            list.add(qdo.templateName.like(SqlUtil.toSqlLikeString(query.getTemplateName())));
        }
        /** 适用项目类型 精确 */
        if (!ObjectUtils.isEmpty(query.getSuitProjectType())) {
            list.add(qdo.suitProjectType.like(SqlUtil.toSqlLikeString("," + query.getSuitProjectType() + ",")));
        }
        /** 模板状态，0：激活，1：暂挂 精确 */
        if (!ObjectUtils.isEmpty(query.getTemplateStatus())) {
            list.add(qdo.templateStatus.eq(query.getTemplateStatus()));
        }
        /** 创建日期范围 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateTimeStart()) && !ObjectUtils.isEmpty(query.getCreateTimeEnd())) {
            list.add(qdo.createTime.between(query.getCreateTimeStart(), query.getCreateTimeEnd()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsWbsTemplateVO queryByKey(Long id) {
        JPAQuery<PmsWbsTemplateVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsWbsTemplateVO> queryListDynamic(PmsWbsTemplateQuery query) {
        JPAQuery<PmsWbsTemplateVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsWbsTemplateVO> queryPaging(PmsWbsTemplateQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsWbsTemplateVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsWbsTemplateVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsWbsTemplateVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsWbsTemplateDO save(PmsWbsTemplateDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsWbsTemplateDO> saveAll(List<PmsWbsTemplateDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsWbsTemplatePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 模板名称
        if (payload.getTemplateName() != null) {
            update.set(qdo.templateName, payload.getTemplateName());
        }
        // 适用项目类型
        if (payload.getSuitProjectType() != null) {
            update.set(qdo.suitProjectType, payload.getSuitProjectType());
        }
        // 模板状态，0：激活，1：暂挂
        if (payload.getTemplateStatus() != null) {
            update.set(qdo.templateStatus, payload.getTemplateStatus());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 模板名称
            if (nullFields.contains("templateName")) {
                update.setNull(qdo.templateName);
            }
            // 适用项目类型
            if (nullFields.contains("suitProjectType")) {
                update.setNull(qdo.suitProjectType);
            }
            // 模板状态，0：激活，1：暂挂
            if (nullFields.contains("templateStatus")) {
                update.setNull(qdo.templateStatus);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 多主键查询
     *
     * @param ids 主键集合
     * @return vo集合
     */
    public List<PmsWbsTemplateVO> queryByKeys(List<Long> ids) {
        JPAQuery<PmsWbsTemplateVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 修改状态
     *
     * @param keys   主键集合
     * @param status 状态
     * @return 受影响的行数
     */
    public Long updateStatus(List<Long> keys, String status) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.templateStatus, status)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据名称查数据
     *
     * @param payload 单据数据
     * @return 是否存在
     */
    public boolean queryByName(PmsWbsTemplatePayload payload){
        JPAQuery<PmsWbsTemplateVO> jpaQuerySelect = getJpaQuerySelect();
        JPAQuery<PmsWbsTemplateVO> jpaQuery = jpaQuerySelect.where(qdo.templateName.eq(payload.getTemplateName()));
        PmsWbsTemplateVO pmsWbsTemplateVO = jpaQuery.fetchOne();
        if (pmsWbsTemplateVO==null){
            return false;
        } else {
            return true;
        }
    }
}

