package com.elitesland.tw.tw5pms.server.project.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.api.common.change.payload.ComChangePayload;
import com.elitesland.tw.tw5.api.common.change.service.ComChangeService;
import com.elitesland.tw.tw5.api.common.log.payload.ComLogPayload;
import com.elitesland.tw.tw5.api.common.log.service.ComLogService;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.common.workFlow.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5pms.api.project.payload.PmsProjectPayload;
import com.elitesland.tw.tw5pms.api.project.query.PmsProjectQuery;
import com.elitesland.tw.tw5pms.api.project.service.PmsProjectService;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectTemplateVO;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectVO;
import com.elitesland.tw.tw5pms.server.common.constans.GenerateSeqNumConstants;
import com.elitesland.tw.tw5pms.server.common.functionEnum.ComChangeTypeEnum;
import com.elitesland.tw.tw5pms.server.common.functionEnum.ComLogTypeEnum;
import com.elitesland.tw.tw5pms.server.common.functionEnum.ProjectStatusEnum;
import com.elitesland.tw.tw5pms.server.project.convert.PmsProjectConvert;
import com.elitesland.tw.tw5pms.server.project.dao.PmsProjectDAO;
import com.elitesland.tw.tw5pms.server.project.dao.PmsProjectTemplateDAO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsProjectDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsProjectRepo;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.elitesland.workflow.payload.StartProcessPayload;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

/**
 * 项目
 *
 * @author carl
 * @date 2023-03-22
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsProjectServiceImpl extends BaseServiceImpl implements PmsProjectService {

    private final PmsProjectRepo pmsProjectRepo;
    private final PmsProjectDAO pmsProjectDAO;
    private final CacheUtil cacheUtil;
    private final ComLogService logService;
    private final ComChangeService changeService;
    private final PmsProjectTemplateDAO pmsProjectTemplateDAO;
    private final WorkflowUtil workflowUtil;
    private final TransactionUtilService transactionUtilService;
    private final FileUtil fileUtil;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsProjectVO insert(PmsProjectPayload payload) {
        //数据校验
        checkData(payload);
        PmsProjectTemplateVO pmsProjectTemplateVO = pmsProjectTemplateDAO.queryByKey(payload.getTemplateId());
        PmsProjectDO entityDo = PmsProjectConvert.INSTANCE.toDo(payload);
        String code = generateSeqNum(GenerateSeqNumConstants.PMS_PROJECT);
        entityDo.setProjectCode(code);
        entityDo.setHideFields(pmsProjectTemplateVO.getHideFields());
        entityDo.setProjectStatus(ProjectStatusEnum.CREATE.getCode());
        entityDo = pmsProjectDAO.save(entityDo);

        ComLogPayload logPayload = new ComLogPayload();
        logPayload.setObjectId(entityDo.getId() + "");
        logPayload.setLogType(ComLogTypeEnum.pms_project.getCode());
        String content = "新增项目：【" + entityDo.getProjectName() + "】";
        logPayload.setLogContent(content);
        logService.insert(logPayload);
        //数据提交,执行工作流
        if (payload.getSubmit()) {
            payload.setId(entityDo.getId());
            submitProc(payload);
        }

        return PmsProjectConvert.INSTANCE.toVo(entityDo);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsProjectVO update(PmsProjectPayload payload) {
        PmsProjectVO pmsProjectVO = pmsProjectDAO.queryByKey(payload.getId());
        if (!ObjectUtils.isEmpty(pmsProjectVO)) {
            //数据校验
            checkData(payload);
            if (pmsProjectVO.getProjectStatus().equals(ProjectStatusEnum.CREATE.getCode())) {
                updateProject(payload);
                //数据提交,执行工作流
                if (payload.getSubmit()) {
                    submitProc(payload);
                }
            } else if (pmsProjectVO.getProjectStatus().equals(ProjectStatusEnum.APPROVED.getCode())) {
                //变更流程
                transferData(pmsProjectVO);
                PmsProjectVO pmsProjectVO1 = PmsProjectConvert.INSTANCE.toVO(payload);
                transferData(pmsProjectVO1);
                Long saveId = changeService.save(ComChangeTypeEnum.PMS_PROJECT_CHANGE.getCode(), PmsProjectConvert.INSTANCE.toPayload(pmsProjectVO), PmsProjectConvert.INSTANCE.toPayload(pmsProjectVO1), payload.getId() + "");
                ProcessInfo processInfo = startChangeWorkFlow(pmsProjectVO, saveId);

                PmsProjectPayload payload0 = new PmsProjectPayload();
                payload0.setId(payload.getId());
                payload0.setSubmitTime(LocalDateTime.now());
                payload0.setProjectStatus(ProjectStatusEnum.APPROVING.getCode());

                ComChangePayload changePayload = new ComChangePayload();
                changePayload.setId(saveId);
                changePayload.setApprProcInstId(processInfo.getProcInstId());
                changePayload.setApprStatus(ProcInstStatus.APPROVING.name());
                changePayload.setChangeStatus(WorkFlowStatusEnum.CHANGING_WORK.getCode());
                //开启事务执行修改，主要是修改审批状态
                transactionUtilService.executeWithRunnable(() -> {
                    pmsProjectDAO.updateWorkFlow(payload0);
                    changeService.updateWorkFlow(changePayload);
                });
            } else {
                throw TwException.error("", "仅新建和激活状态支持变更，请核验！");
            }

        } else {
            throw TwException.error("", "变更数据不存在，请核验！");
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateWorkflowProject(PmsProjectPayload payload) {
        PmsProjectVO pmsProjectVO = pmsProjectDAO.queryByKey(payload.getId());
        if (pmsProjectVO != null) {
            if (pmsProjectVO.getProjectStatus().equals(ProjectStatusEnum.APPROVING_EJECTED.getCode())) {
                PmsProjectVO pmsProjectVO1 = PmsProjectConvert.INSTANCE.toVO(payload);
                transferData(pmsProjectVO1);
                changeService.update(ComChangeTypeEnum.PMS_PROJECT_CHANGE.getCode(), payload.getId() + "", PmsProjectConvert.INSTANCE.toPayload(pmsProjectVO1));

            } else {
                throw TwException.error("", "仅支持流程被驳回的变更，请核验！");
            }
        } else {
            throw TwException.error("", "变更数据不存在，请核验！");
        }

    }

    /**
     * 新建变更工作流
     *
     * @param pmsProjectVO
     * @return
     */
    private ProcessInfo startChangeWorkFlow(PmsProjectVO pmsProjectVO, Long saveId) {
        HashMap<String, Object> batchMap = new HashMap<>();
        batchMap.put("Activity_1kq74go", Lists.newArrayList(pmsProjectVO.getPmoManagerUserId().toString()));
        batchMap.put("Activity_0suh44b", Lists.newArrayList(pmsProjectVO.getManagerUserId().toString()));

        ProcessInfo processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.PRODUCT_PRICE_CHANGE.name(),
                pmsProjectVO.getProjectName() + "-项目变更审批流程",
                saveId + "",
                batchMap)
        );
        return processInfo;
    }

    /**
     * 创建工作流
     *
     * @param payload
     */
    void submitProc(PmsProjectPayload payload) {

        HashMap<String, Object> batchMap = new HashMap<>();
        batchMap.put("Activity_1kq74go", Lists.newArrayList(payload.getPmoManagerUserId().toString()));
        batchMap.put("Activity_0suh44b", Lists.newArrayList(payload.getManagerUserId().toString()));
        //发起流程审批
        ProcessInfo processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.PMS_PROJECT.name(),
                payload.getProjectName() + "-项目审批流程",
                payload.getId() + "",
                batchMap)
        );
        PmsProjectPayload payload0 = new PmsProjectPayload();
        payload0.setId(payload.getId());
        payload0.setProcInstId(processInfo.getProcInstId());
        payload0.setProcInstStatus(processInfo.getProcInstStatus());
        payload0.setSubmitTime(LocalDateTime.now());
        payload0.setProjectStatus(ProjectStatusEnum.APPROVING.getCode());
        //开启事务执行修改，主要是修改审批状态
        transactionUtilService.executeWithRunnable(() -> {
            pmsProjectDAO.updateWorkFlow(payload0);
        });
    }

    @Override
    public void updateProject(PmsProjectPayload payload) {
        PmsProjectDO entityDo = PmsProjectConvert.INSTANCE.toDo(payload);
        pmsProjectDAO.save(entityDo);
        ComLogPayload logPayload = new ComLogPayload();
        logPayload.setObjectId(entityDo.getId() + "");
        logPayload.setLogType(ComLogTypeEnum.pms_project.getCode());
        String content = "编辑项目：【" + entityDo.getProjectName() + "】";
        logPayload.setLogContent(content);
        logService.insert(logPayload);
    }


    @Override
    public PagingVO<PmsProjectVO> queryPaging(PmsProjectQuery query) {
        //权限处理
        operPermissionFlag(query);
        PagingVO<PmsProjectVO> projectVOPagingVO = pmsProjectDAO.queryPaging(query);
        projectVOPagingVO.getRecords().forEach(vo -> transferData(vo));
        return projectVOPagingVO;
    }

    /**
     * 翻译数据
     *
     * @param vo
     */
    void transferData(PmsProjectVO vo) {
        vo.setContractFilesDatas(fileUtil.getFileDatas(vo.getContractFiles()));
        vo.setSowFilesDatas(fileUtil.getFileDatas(vo.getSowFiles()));
        vo.setBudgetFilesDatas(fileUtil.getFileDatas(vo.getBudgetFiles()));

        vo.setCreator(cacheUtil.getUserName(vo.getCreateUserId()));
        vo.setCustIndustryName(cacheUtil.transferSystemSelection("crm:leads_customer_industry", vo.getCustIndustry()));
        vo.setProjectTypeName(cacheUtil.transferSystemSelection("PMS:PROJECT:TYPE", vo.getProjectType()));
        vo.setTimesheetPeriodName(cacheUtil.transferSystemSelection("PMS:PROJECT:TIMESHEET", vo.getTimesheetPeriod()));
        vo.setProjectImportanceName(cacheUtil.transferSystemSelection("PMS:PROJECT:IMPORTANCE", vo.getProjectImportance()));
        vo.setProjectDifficultyName(cacheUtil.transferSystemSelection("PMS:PROJECT;DIFFICULTY", vo.getProjectDifficulty()));
        vo.setProjectStatusName(cacheUtil.transferSystemSelection("PMS:PROJECT:STATUS", vo.getProjectStatus()));
        vo.setManagerUserName(cacheUtil.getUserName(vo.getManagerUserId()));
        vo.setPmoManagerUserName(cacheUtil.getUserName(vo.getPmoManagerUserId()));
        vo.setSaleManagerUserName(cacheUtil.getUserName(vo.getSaleManagerUserId()));
        vo.setPayManagerUserName(cacheUtil.getUserName(vo.getPayManagerUserId()));
        vo.setCreator(cacheUtil.getUserName(vo.getCreateUserId()));

    }

    @Override
    public List<PmsProjectVO> queryList(PmsProjectQuery query) {
        return PmsProjectConvert.INSTANCE.toVoList(
                pmsProjectRepo.findAll(
                        (root, criteriaQuery, criteriaBuilder)
                                -> QueryHelp.getPredicate(root, query, criteriaBuilder)
                        , query.getPageRequest().getSort()
                )
        );
    }

    @Override
    public List<PmsProjectVO> queryListDynamic(PmsProjectQuery query) {
        return pmsProjectDAO.queryListDynamic(query);
    }

    @Override
    public PmsProjectVO queryByKey(Long key) {
        PmsProjectVO pmsProjectVO = pmsProjectDAO.queryByKey(key);
        transferData(pmsProjectVO);
        return pmsProjectVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void submitStartProject(Long key) {
        PmsProjectVO pmsProjectVO = pmsProjectDAO.queryByKey(key);
        if (ObjectUtils.isEmpty(pmsProjectVO)) {
            throw TwException.error("", "提交数据不存在，请核验！");
        }
        PmsProjectPayload pmsProjectPayload = PmsProjectConvert.INSTANCE.toPayload(pmsProjectVO);
        submitProc(pmsProjectPayload);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            List<PmsProjectVO> pmsProjectVOS = pmsProjectDAO.queryByKeys(keys);
            String content = "删除项目：【";
            for (PmsProjectVO pmsProjectVO : pmsProjectVOS) {
                String projectStatus = pmsProjectVO.getProjectStatus();
                if (!projectStatus.equals(ProjectStatusEnum.CREATE.getCode()) && !projectStatus.equals(ProjectStatusEnum.PENDING.getCode())) {
                    throw TwException.error("", "仅支持新建和暂挂状态下的删除，请核验！");
                }
                content += pmsProjectVO.getProjectName() + "】,【";
            }
            //删除项目
            pmsProjectDAO.deleteSoft(keys);
            ComLogPayload logPayload = new ComLogPayload();
            logPayload.setObjectId("");
            logPayload.setLogType(ComLogTypeEnum.pms_project.getCode());
            content = content.substring(0, content.length() - 2);
            logPayload.setLogContent(content);
            logService.insert(logPayload);
        }
    }


    /**
     * 数据校验
     *
     * @param payload
     */
    void checkData(PmsProjectPayload payload) {
        if (ObjectUtils.isEmpty(payload.getCustId())) {
            throw TwException.error("", "关联客户不存在，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getTemplateId())) {
            throw TwException.error("", "未选择模板，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getManagerUserId())) {
            throw TwException.error("", "项目经理不存在，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getPmoManagerUserId())) {
            throw TwException.error("", "pmo负责人不存在，请核验！");
        }
    }

    /**
     * 权限处理
     *
     * @param query
     */
    void operPermissionFlag(PmsProjectQuery query) {
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.PROJECT_ADMIN.getCode()));
        query.setPermissionFlag(!rolePermission);
        if (!rolePermission) {
            //需要处理权限
            query.setLoginUserId(GlobalUtil.getLoginUserId());
        }
    }

}
