package com.elitesland.tw.tw5pms.server.project.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5pms.api.project.payload.PmsProjectTeamBasePayload;
import com.elitesland.tw.tw5pms.api.project.query.PmsProjectTeamBaseQuery;
import com.elitesland.tw.tw5pms.api.project.service.PmsProjectTeamBaseService;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectTeamBaseVO;
import com.elitesland.tw.tw5pms.server.project.convert.PmsProjectTeamBaseConvert;
import com.elitesland.tw.tw5pms.server.project.dao.PmsProjectTeamBaseDAO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsProjectTeamBaseDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsProjectTeamBaseRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * 项目成员基本信息
 *
 * @author Echo
 * @date 2023-04-11
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsProjectTeamBaseServiceImpl extends BaseServiceImpl implements PmsProjectTeamBaseService {

    private final PmsProjectTeamBaseRepo pmsProjectTeamBaseRepo;
    private final PmsProjectTeamBaseDAO pmsProjectTeamBaseDAO;

    /**
     * 新增
     *
     * @param payload 单据数据
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsProjectTeamBaseVO insert(PmsProjectTeamBasePayload payload) {
        //校验开始周必须为周一
        checkStartWeek(payload);

        PmsProjectTeamBaseDO entityDo = PmsProjectTeamBaseConvert.INSTANCE.toDo(payload);

        //根据传入的开始周和持续时间，生成结束周
        entityDo.setStopDate(entityDo.getStartDate().plusWeeks(entityDo.getContinueWeek()));
        return PmsProjectTeamBaseConvert.INSTANCE.toVo(pmsProjectTeamBaseRepo.save(entityDo));
    }

    /**
     * 校验开始周必须为周一
     *
     * @param payload 单据数据
     */
    private void checkStartWeek(PmsProjectTeamBasePayload payload) {
        LocalDate startDate = payload.getStartDate();
        //LocalDate -> Date
        Date date = Date.from(startDate.atStartOfDay(ZoneId.systemDefault()).toInstant());
        //判断该周是这周的第几天，不是第一天则报错
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        //由于Calendar类中：第一天是周日，所以需要用到第几天等类似方法的时候需要-1
        if (calendar.get(Calendar.DAY_OF_WEEK)-1 != 1){
            throw TwException.error("","开始周必须为周一！");
        }
    }

    /**
     * 更新
     *
     * @param payload 单据数据
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsProjectTeamBaseVO update(PmsProjectTeamBasePayload payload) {
        PmsProjectTeamBaseDO entity = pmsProjectTeamBaseRepo.findById(payload.getId()).orElseGet(PmsProjectTeamBaseDO::new);
        Assert.notNull(entity.getId(), "不存在");
        checkStartWeek(payload);
        PmsProjectTeamBaseDO entityDo = PmsProjectTeamBaseConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        entity.setStopDate(entity.getStartDate().plusWeeks(entity.getContinueWeek()));
        return PmsProjectTeamBaseConvert.INSTANCE.toVo(pmsProjectTeamBaseRepo.save(entity));
    }

    /**
     * 根据项目id查询
     *
     * @param projectId 主键集合
     * @return
     */
    @Override
    public PmsProjectTeamBaseVO queryByProjectId(Long projectId) {
        PmsProjectTeamBaseDO entity = pmsProjectTeamBaseRepo.findByProjectId(projectId).orElseGet(PmsProjectTeamBaseDO::new);
//        Assert.notNull(entity.getId(), "项目成员不存在，请新建！");
        PmsProjectTeamBaseVO vo = PmsProjectTeamBaseConvert.INSTANCE.toVo(entity);
        return vo;
    }

    /**
     * 逻辑删除
     *
     * @param keys 主键集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            pmsProjectTeamBaseDAO.deleteSoft(keys);
        }
    }

    @Override
    public PmsProjectTeamBaseVO queryByKey(Long key) {
        PmsProjectTeamBaseDO entity = pmsProjectTeamBaseRepo.findById(key).orElseGet(PmsProjectTeamBaseDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PmsProjectTeamBaseVO vo = PmsProjectTeamBaseConvert.INSTANCE.toVo(entity);
        return vo;
    }


    @Override
    public PagingVO<PmsProjectTeamBaseVO> paging(PmsProjectTeamBaseQuery query){
        Page<PmsProjectTeamBaseDO> page = pmsProjectTeamBaseRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root,query,criteriaBuilder),query.getPageRequest());
        return PageUtil.toPageVo(page.map(PmsProjectTeamBaseConvert.INSTANCE::toVo));
    }

    @Override
    public PagingVO<PmsProjectTeamBaseVO> queryPaging(PmsProjectTeamBaseQuery query){
        return pmsProjectTeamBaseDAO.queryPaging(query);
    }


    @Override
    public List<PmsProjectTeamBaseVO> queryList(PmsProjectTeamBaseQuery query){
        return PmsProjectTeamBaseConvert.INSTANCE.toVoList(
                pmsProjectTeamBaseRepo.findAll(
                        (root, criteriaQuery, criteriaBuilder)
                                -> QueryHelp.getPredicate(root, query, criteriaBuilder)
                        , query.getPageRequest().getSort()
                )
        );
    }

    @Override
    public List<PmsProjectTeamBaseVO> queryListDynamic(PmsProjectTeamBaseQuery query){
        return pmsProjectTeamBaseDAO.queryListDynamic(query);
    }
}

