package com.elitesland.tw.tw5pms.server.project.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5pms.api.project.payload.PmsProjectTeamMemberPayload;
import com.elitesland.tw.tw5pms.api.project.payload.PmsProjectTeamMemberWeekPayload;
import com.elitesland.tw.tw5pms.api.project.query.PmsProjectTeamMemberQuery;
import com.elitesland.tw.tw5pms.api.project.service.PmsProjectTeamMemberService;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectTeamMemberVO;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectTeamMemberWeekVO;
import com.elitesland.tw.tw5pms.server.project.convert.PmsProjectTeamMemberConvert;
import com.elitesland.tw.tw5pms.server.project.dao.PmsProjectTeamMemberDAO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsProjectTeamMemberDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsProjectTeamMemberRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 项目成员资质信息
 *
 * @author Echo
 * @date 2023-04-11
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsProjectTeamMemberServiceImpl extends BaseServiceImpl implements PmsProjectTeamMemberService {

    private final PmsProjectTeamMemberRepo pmsProjectTeamMemberRepo;
    private final PmsProjectTeamMemberDAO pmsProjectTeamMemberDAO;
    private final CacheUtil cacheUtil;


    /**
     * 分页查询
     *
     * @param query 条件
     * @return vo
     */
    @Override
    public PagingVO<PmsProjectTeamMemberVO> queryPaging(PmsProjectTeamMemberQuery query) {
        PagingVO<PmsProjectTeamMemberVO> pagingVO = pmsProjectTeamMemberDAO.queryPaging(query);
        //翻译
        pagingVO.getRecords().forEach(this::transferData);
        //json转换list
        pagingVO.getRecords().forEach(this::setWeekData);
        return pagingVO;
    }

    /**
     * 翻译
     *
     * @param vo 返回前端数据
     */
    void transferData(PmsProjectTeamMemberVO vo) {
        vo.setWorkType(cacheUtil.transferSystemSelection("org:employee:worktype", vo.getWorkType()));
        vo.setUserName(cacheUtil.getUserName(vo.getUserId()));
//        vo.setProfessionalLevel((cacheUtil.transferSystemSelection("",vo.getProfessionalLevel())));
    }

    /**
     * 将查询出的json转换为list<PmsProjectTeamMemberWeekVo>
     *
     * @param vo 返回前端数据
     */
    void setWeekData(PmsProjectTeamMemberVO vo) {
        //String -> list
        List<PmsProjectTeamMemberWeekVO> list = JSONArray.parseArray(vo.getWeekData(), PmsProjectTeamMemberWeekVO.class);
        //设置list
        vo.setWeeks(list);
    }

    /**
     * 批量插入
     *
     * @param payloads 单据数据
     * @return vo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsProjectTeamMemberVO insert(List<PmsProjectTeamMemberPayload> payloads) {
        if (payloads.isEmpty()) {
            throw TwException.error("", "添加资源不存在，请核验！");
        }
        List<Long> collect = payloads.stream().map(PmsProjectTeamMemberPayload::getUserId).distinct().collect(Collectors.toList());
        if (collect.size() < payloads.size()) {
            throw TwException.error("", "资源不可重复添加，请核验！");
        }
        //TODO 现在传输的是baseid，后期改成projectId
        List<PmsProjectTeamMemberVO> vos = pmsProjectTeamMemberDAO.queryByProjectId(payloads.get(0).getBaseId());
        payloads.forEach(payload -> {
            //1. 检查字段非空
            checkData(payload);
            //2. 检查预计开始和结束日期是否符合逻辑
            checkDate(payload);
            //3. 检查周的开始日期是否符合逻辑
            checkDays(payload);

            if (!ObjectUtils.isEmpty(vos)) {
                Optional<PmsProjectTeamMemberVO> first = vos.stream().filter(vo -> vo.getUserId().equals(payload.getUserId())).findFirst();
                if (first.isPresent()) {
                    /**
                     * 两条数据主键不等说明不是同一条数据但是资源重复了
                     * 1.资源改掉了
                     * 2.资源重复了
                     */
                    PmsProjectTeamMemberVO pmsProjectTeamMemberVO = first.get();
                    if (!pmsProjectTeamMemberVO.getId().equals(payload.getId())) {
                        throw TwException.error("", "资源不可重复添加，请核验！");
//                        Optional<PmsProjectTeamMemberPayload> first1 = payloads.stream().filter(payload0 -> payload0.getId().equals(pmsProjectTeamMemberVO.getId())).findFirst();
//                        if (!first1.isPresent()) {
//                            throw TwException.error("", "，资源不可重复添加，请核验！");
//                        }
                    }
                }
            }

            String jsonString = JSON.toJSONString(payload.getWeeks());
            payload.setWeekData(jsonString);
        });


        //4.2. 将json赋给weekDate
        List<PmsProjectTeamMemberDO> entityDos = PmsProjectTeamMemberConvert.INSTANCE.toDos(payloads);
        pmsProjectTeamMemberRepo.saveAll(entityDos);
        return null;
    }

    /**
     * 检查逻辑需要的字段非空
     *
     * @param payload 单据数据
     */
    private void checkData(PmsProjectTeamMemberPayload payload) {
        if (ObjectUtils.isEmpty(payload.getStartDate())) {
            throw TwException.error("", "预计开始日期不存在，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getStopDate())) {
            throw TwException.error("", "预计结束日期不存在，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getWeeks())) {
            throw TwException.error("", "WEEK天数不存在，请核验！");
        }
    }

    /**
     * 检查预计开始和结束日期
     *
     * @param payload 单据数据
     */
    private void checkDate(PmsProjectTeamMemberPayload payload) {
        //判断 预计开始日期 不得早于 开始周；
        if (payload.getStartDate().isBefore(payload.getPmsProjectTeamBasePayload().getStartDate())) {
            throw TwException.error("", "预计开始日期不得早于开始周！");
        }
        //判断 预计结束日期 不得晚于 结束周
        if (payload.getStopDate().isAfter(payload.getPmsProjectTeamBasePayload().getStopDate())) {
            throw TwException.error("", "预计结束日期不得晚于结束周！");
        }
        if (payload.getStartDate().isAfter(payload.getStopDate())) {
            throw TwException.error("", "预计结束日期不得在预计开始日期之前，请核验！");
        }
    }

    /**
     * 检查周开始日期
     *
     * @param payload 单据数据
     */
    private void checkDays(PmsProjectTeamMemberPayload payload) {
        //拿到week集合
        List<PmsProjectTeamMemberWeekPayload> pmsProjectTeamMemberWeekPayloads = payload.getWeeks();
        //判断非空
        if (pmsProjectTeamMemberWeekPayloads.size() > 0) {
            //遍历
            for (PmsProjectTeamMemberWeekPayload pmsProjectTeamMemberWeekPayload : pmsProjectTeamMemberWeekPayloads) {
                //判断人天（每周工作天数）不得大于7
                if (pmsProjectTeamMemberWeekPayload.getDays() != null && pmsProjectTeamMemberWeekPayload.getDays() > 7) {
                    throw TwException.error("", "人天不得大于6");
                }
            }
            //判断第一周的week_start不得早于预计开始日期
//            PmsProjectTeamMemberWeekPayload startWeek = pmsProjectTeamMemberWeekPayloads.get(0);
//            if (startWeek.getWeek_start().isBefore(payload.getStartDate())) {
//                throw TwException.error("", "开始周不得早于预计开始日期！");
//            }

            //判断最后一周的week_start不得晚于预计结束日期
//            PmsProjectTeamMemberWeekPayload stopWeek = pmsProjectTeamMemberWeekPayloads.get(pmsProjectTeamMemberWeekPayloads.size() - 1);
//            if (stopWeek.getWeek_start().isAfter(payload.getStopDate())){
//                throw TwException.error("", "结束周不得晚于预计结束日期！");
//            }
        }
    }

    /**
     * 修改
     *
     * @param payload 单据数据
     * @return vo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsProjectTeamMemberVO update(PmsProjectTeamMemberPayload payload) {
        //根据Id查出数据库中的entity
        PmsProjectTeamMemberDO entity = pmsProjectTeamMemberRepo.findById(payload.getId()).orElseGet(PmsProjectTeamMemberDO::new);
        Assert.notNull(entity.getId(), "不存在");

        //检查资源是否已存在
        if (pmsProjectTeamMemberDAO.queryByUserIdAndBaseId(payload) != null) {
            //排除自己
            if (!Objects.equals(pmsProjectTeamMemberDAO.queryByKey(payload.getId()).getId(), payload.getId())) {
                throw TwException.error("", "该资源已存在！");
            }
        }


        //1. 检查字段非空
        checkData(payload);
        //2. 检查预计开始和结束日期是否符合逻辑
        checkDate(payload);
        //3. 检查周的开始日期是否符合逻辑
        checkDays(payload);

        //将payload中的List<PmsProjectTeamMemberWeekPayload -> Json
        JSONArray jsonArray = JSON.parseArray(JSON.toJSONString(payload.getWeeks()));
        //将payload -> entityDo
        PmsProjectTeamMemberDO entityDo = PmsProjectTeamMemberConvert.INSTANCE.toDo(payload);
        //将Json赋值给entityDo
        entityDo.setWeekData(jsonArray.toString());

        entity.copy(entityDo);
        return PmsProjectTeamMemberConvert.INSTANCE.toVo(pmsProjectTeamMemberRepo.save(entity));
    }

    /**
     * 逻辑删除
     *
     * @param keys 主键集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            pmsProjectTeamMemberDAO.deleteSoft(keys);
        }
    }

    @Override
    public PmsProjectTeamMemberVO queryByKey(Long key) {
        PmsProjectTeamMemberDO entity = pmsProjectTeamMemberRepo.findById(key).orElseGet(PmsProjectTeamMemberDO::new);
        Assert.notNull(entity.getId(), "不存在");
        return PmsProjectTeamMemberConvert.INSTANCE.toVo(entity);
    }

    @Override
    public PagingVO<PmsProjectTeamMemberVO> paging(PmsProjectTeamMemberQuery query) {
        Page<PmsProjectTeamMemberDO> page = pmsProjectTeamMemberRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page.map(PmsProjectTeamMemberConvert.INSTANCE::toVo));
    }

    @Override
    public List<PmsProjectTeamMemberVO> queryList(PmsProjectTeamMemberQuery query) {
        return PmsProjectTeamMemberConvert.INSTANCE.toVoList(
                pmsProjectTeamMemberRepo.findAll(
                        (root, criteriaQuery, criteriaBuilder)
                                -> QueryHelp.getPredicate(root, query, criteriaBuilder)
                        , query.getPageRequest().getSort()
                )
        );
    }

    @Override
    public List<PmsProjectTeamMemberVO> queryListDynamic(PmsProjectTeamMemberQuery query) {
        return pmsProjectTeamMemberDAO.queryListDynamic(query);
    }
}
