package com.elitesland.tw.tw5pms.server.project.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.api.common.log.payload.ComLogPayload;
import com.elitesland.tw.tw5.api.common.log.service.ComLogService;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5pms.api.project.payload.PmsProjectTemplatePayload;
import com.elitesland.tw.tw5pms.api.project.query.PmsProjectTemplateQuery;
import com.elitesland.tw.tw5pms.api.project.service.PmsProjectTemplateService;
import com.elitesland.tw.tw5pms.api.project.vo.PmsProjectTemplateVO;
import com.elitesland.tw.tw5pms.server.common.functionEnum.ComLogTypeEnum;
import com.elitesland.tw.tw5pms.server.project.convert.PmsProjectTemplateConvert;
import com.elitesland.tw.tw5pms.server.project.dao.PmsProjectTemplateDAO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsProjectTemplateDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsProjectTemplateRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 项目模板
 *
 * @author carl
 * @date 2023-03-22
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsProjectTemplateServiceImpl extends BaseServiceImpl implements PmsProjectTemplateService {

    private final PmsProjectTemplateRepo pmsProjectTemplateRepo;
    private final PmsProjectTemplateDAO pmsProjectTemplateDAO;
    private final CacheUtil cacheUtil;
    private final ComLogService logService;

    @Override
    public PagingVO<PmsProjectTemplateVO> paging(PmsProjectTemplateQuery query) {
        query.setTemplateType("tem");
        PagingVO<PmsProjectTemplateVO> paging = pmsProjectTemplateDAO.paging(query);
        if (!ObjectUtils.isEmpty(paging.getRecords())) {
            paging.getRecords().forEach(record -> transferData(record));
        }
        return paging;
    }

    @Override
    public PagingVO<PmsProjectTemplateVO> queryPaging(PmsProjectTemplateQuery query) {
        query.setTemplateType("pre_tem");
        return pmsProjectTemplateDAO.queryPaging(query);
    }

    void transferData(PmsProjectTemplateVO vo) {
        vo.setTemplateStatusName(cacheUtil.transferSystemSelection("SYSTEM_BASIC:STATUS", vo.getTemplateStatus()));
        vo.setCreator(cacheUtil.getUserName(vo.getCreateUserId()));
    }

    @Override
    public List<PmsProjectTemplateVO> queryList(PmsProjectTemplateQuery query) {
        query.setTemplateType("tem");
        return PmsProjectTemplateConvert.INSTANCE.toVoList(
                pmsProjectTemplateRepo.findAll(
                        (root, criteriaQuery, criteriaBuilder)
                                -> QueryHelp.getPredicate(root, query, criteriaBuilder)
                        , query.getPageRequest().getSort()
                )
        );
    }

    @Override
    public List<PmsProjectTemplateVO> queryListDynamic(PmsProjectTemplateQuery query) {
        query.setTemplateType("tem");
        return pmsProjectTemplateDAO.queryListDynamic(query);
    }

    @Override
    public PmsProjectTemplateVO queryByKey(Long key) {
        PmsProjectTemplateVO pmsProjectTemplateVO = pmsProjectTemplateDAO.queryByKey(key);
        transferData(pmsProjectTemplateVO);
        return pmsProjectTemplateVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsProjectTemplateVO insert(PmsProjectTemplatePayload payload) {

        PmsProjectTemplateVO pmsProjectTemplateVO = pmsProjectTemplateDAO.queryByName(payload.getTemplateName());
        if (!ObjectUtils.isEmpty(pmsProjectTemplateVO)) {
            throw TwException.error("", "名称可不重复，请核验！");
        }
        PmsProjectTemplateDO entityDo = PmsProjectTemplateConvert.INSTANCE.toDo(payload);
        entityDo.setTemplateStatus(WorkFlowStatusEnum.APPROVED_WORK.getCode());
        entityDo.setTemplateType("tem");

        PmsProjectTemplateDO save = pmsProjectTemplateRepo.save(entityDo);
        ComLogPayload logPayload = new ComLogPayload();
        logPayload.setObjectId(save.getId() + "");
        logPayload.setLogType(ComLogTypeEnum.pms_project_template.getCode());
        String content = "新增模板：【" + save.getTemplateName() + "】";
        logPayload.setLogContent(content);
        logService.insert(logPayload);
        return PmsProjectTemplateConvert.INSTANCE.toVo(save);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsProjectTemplateVO update(PmsProjectTemplatePayload payload) {

        PmsProjectTemplateDO entity = pmsProjectTemplateRepo.findById(payload.getId()).orElseGet(PmsProjectTemplateDO::new);
        if (!entity.getTemplateName().equals(payload.getTemplateName())) {
            PmsProjectTemplateVO pmsProjectTemplateVO = pmsProjectTemplateDAO.queryByName(payload.getTemplateName());
            if (!ObjectUtils.isEmpty(pmsProjectTemplateVO)) {
                throw TwException.error("", "名称可不重复，请核验！");
            }
        }
        Assert.notNull(entity.getId(), "不存在");
        PmsProjectTemplateDO entityDo = PmsProjectTemplateConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        //记录日志
        ComLogPayload logPayload = new ComLogPayload();
        logPayload.setObjectId(entity.getId() + "");
        logPayload.setLogType(ComLogTypeEnum.pms_project_template.getCode());
        String content = "编辑模板：【" + entity.getTemplateName() + "】";
        logPayload.setLogContent(content);
        logService.insert(logPayload);
        return PmsProjectTemplateConvert.INSTANCE.toVo(pmsProjectTemplateRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            String content = "删除模板：【";
            List<PmsProjectTemplateVO> templateVOS = pmsProjectTemplateDAO.queryByKeys(keys);
            for (PmsProjectTemplateVO templateVO : templateVOS) {
                String templateStatus = templateVO.getTemplateStatus();
                if (!templateStatus.equals(WorkFlowStatusEnum.PENDING_WORK.getCode())) {
                    throw TwException.error("", "仅支持暂挂状态下的删除，请核验！");
                }
                content += templateVO.getTemplateName() + "】,【";
            }
            //删除项目模板
            pmsProjectTemplateDAO.deleteSoft(keys);
            //记录日志
            ComLogPayload logPayload = new ComLogPayload();
            logPayload.setObjectId("");
            logPayload.setLogType(ComLogTypeEnum.pms_project_template.getCode());
            content = content.substring(0, content.length() - 2);
            logPayload.setLogContent(content);
            logService.insert(logPayload);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long updateStatus(List<Long> keys) {
        //判断不为空
        if (!keys.isEmpty()) {
            //根据keys查询得到vos
            List<PmsProjectTemplateVO> templateVOS = pmsProjectTemplateDAO.queryByKeys(keys);
            //判断非空
            if (ObjectUtils.isEmpty(templateVOS)) {
                throw TwException.error("", "操作数据不存在，请核验！");
            }

            List<String> collect = templateVOS.stream().map(PmsProjectTemplateVO::getTemplateStatus).distinct().collect(Collectors.toList());
            if (collect.size() > 1) {
                throw TwException.error("", "相同数据状态可进行该操作，请核验！");
            }
            String templateStatus = collect.get(0);
            String templateStatusNew = templateStatus.equals(WorkFlowStatusEnum.PENDING_WORK.getCode()) ? WorkFlowStatusEnum.APPROVED_WORK.getCode() : WorkFlowStatusEnum.PENDING_WORK.getCode();
            return pmsProjectTemplateDAO.updateStatus(keys, templateStatusNew);
        } else {
            throw TwException.error("", "操作数据不可为空，请核验！");
        }
    }

}
