package com.elitesland.tw.tw5pms.server.project.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5pms.api.project.payload.PmsWbsTemplateNodePayload;
import com.elitesland.tw.tw5pms.api.project.payload.PmsWbsTemplatePayload;
import com.elitesland.tw.tw5pms.api.project.query.PmsWbsTemplateQuery;
import com.elitesland.tw.tw5pms.api.project.service.PmsWbsTemplateService;
import com.elitesland.tw.tw5pms.api.project.vo.PmsWbsTemplateNodeVO;
import com.elitesland.tw.tw5pms.api.project.vo.PmsWbsTemplateVO;
import com.elitesland.tw.tw5pms.server.project.convert.PmsWbsTemplateConvert;
import com.elitesland.tw.tw5pms.server.project.convert.PmsWbsTemplateNodeConvert;
import com.elitesland.tw.tw5pms.server.project.dao.PmsWbsTemplateDAO;
import com.elitesland.tw.tw5pms.server.project.dao.PmsWbsTemplateNodeDAO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsWbsTemplateDO;
import com.elitesland.tw.tw5pms.server.project.entity.PmsWbsTemplateNodeDO;
import com.elitesland.tw.tw5pms.server.project.repo.PmsWbsTemplateRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * wbs模板
 *
 * @author Echo
 * @date 2023-03-29
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsWbsTemplateServiceImpl extends BaseServiceImpl implements PmsWbsTemplateService {

    private final PmsWbsTemplateRepo pmsWbsTemplateRepo;
    private final PmsWbsTemplateDAO pmsWbsTemplateDAO;
    private final PmsWbsTemplateNodeDAO pmsWbsTemplateNodeDAO;
    private final CacheUtil cacheUtil;

    /**
     * 分页
     *
     * @param query 条件
     * @return pagingVo
     */
    @Override
    public PagingVO<PmsWbsTemplateVO> paging(PmsWbsTemplateQuery query) {
        Page<PmsWbsTemplateDO> page = pmsWbsTemplateRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page.map(PmsWbsTemplateConvert.INSTANCE::toVo));
    }

    void transferData(PmsWbsTemplateVO vo) {
        //翻译模板状态
        //vo.setTemplateStatusName(cacheUtil.transferSystemSelection("PMS:WBS:STATUS", vo.getTemplateStatus()));
        //去除拼装的“，”
        vo.setSuitProjectType(vo.getSuitProjectType().substring(1, vo.getSuitProjectType().length() - 1));
        //翻译适用项目类型
        String[] split = vo.getSuitProjectType().split(",");
        //可变字符串
        StringBuilder temp = new StringBuilder();
        for (String s : split) {
            //字符串拼接
            temp.append(cacheUtil.transferSystemSelection("PMS:PROJECT:TYPE", s)).append(",");
        }
        //去除最后多余的","
        temp = new StringBuilder(temp.substring(0, temp.length() - 1));
        vo.setSuitProjectTypeNames(temp.toString());

        //设置nodes
        Long key = vo.getId();
        List<PmsWbsTemplateNodeVO> pmsWbsTemplateNodeVOS = pmsWbsTemplateNodeDAO.queryByTemplateIds(List.of(key));
        vo.setNodes(pmsWbsTemplateNodeVOS);
    }

    /**
     * 分页查询
     *
     * @param query 条件
     * @return pagingVo
     */
    @Override
    public PagingVO<PmsWbsTemplateVO> queryPaging(PmsWbsTemplateQuery query) {
        PagingVO<PmsWbsTemplateVO> pagingVO = pmsWbsTemplateDAO.queryPaging(query);
        //翻译
        pagingVO.getRecords().forEach(this::transferData);
        return pagingVO;
    }

    /**
     * 查询列表
     *
     * @param query 条件参数
     * @return vo
     */
    @Override
    public List<PmsWbsTemplateVO> queryList(PmsWbsTemplateQuery query) {
        return PmsWbsTemplateConvert.INSTANCE.toVoList(
                pmsWbsTemplateRepo.findAll(
                        (root, criteriaQuery, criteriaBuilder)
                                -> QueryHelp.getPredicate(root, query, criteriaBuilder)
                        , query.getPageRequest().getSort()
                )
        );
    }

    /**
     * 动态查询列表
     *
     * @param query 条件参数
     * @return vo
     */
    @Override
    public List<PmsWbsTemplateVO> queryListDynamic(PmsWbsTemplateQuery query) {
        return pmsWbsTemplateDAO.queryListDynamic(query);
    }

    /**
     * 通过主键查询
     *
     * @param key 主键
     * @return vo
     */
    @Override
    public PmsWbsTemplateVO queryByKey(Long key) {
        // 根据主键查询
        PmsWbsTemplateDO entity = pmsWbsTemplateRepo.findById(key).orElseGet(PmsWbsTemplateDO::new);
        Assert.notNull(entity.getId(), "不存在");
        // do -> vo
        PmsWbsTemplateVO vo = PmsWbsTemplateConvert.INSTANCE.toVo(entity);
        //查询子表（templateNode）
        List<PmsWbsTemplateNodeVO> pmsWbsTemplateNodeVOS = pmsWbsTemplateNodeDAO.queryByTemplateIds(List.of(key));
        vo.setNodes(pmsWbsTemplateNodeVOS);
        //翻译数据
        transferData(vo);
        //排序
        vo.getNodes().sort(Comparator.comparing(PmsWbsTemplateNodeVO::getWbsCode));
        return vo;
    }

    /**
     * 插入
     *
     * @param payload 单据数据
     * @return vo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsWbsTemplateVO insert(PmsWbsTemplatePayload payload) {
        //数据校验(必选不得为null)
        checkData(payload);
        //校验不得重名
        if (pmsWbsTemplateDAO.queryByName(payload)) {
            throw TwException.error("","已存在！");
        }
        //payload -> do
        payload.setSuitProjectType("," + payload.getSuitProjectType() + ",");
        PmsWbsTemplateDO entityDo = PmsWbsTemplateConvert.INSTANCE.toDo(payload);

        pmsWbsTemplateDAO.save(entityDo);
        Long id = entityDo.getId();
        //修改节点编码
        setNodeCode(payload, entityDo ,id);
        //do -> vo 并返回
        return PmsWbsTemplateConvert.INSTANCE.toVo(entityDo);
    }

    /**
     * 修改节点编码
     *
     * @param payload 单据数据
     * @param id      wbs模板id
     */
    private void setNodeCode(PmsWbsTemplatePayload payload, PmsWbsTemplateDO entity,Long id) {
        //payload获取节点集合
        List<PmsWbsTemplateNodePayload> nodes = payload.getNodes();
        List<PmsWbsTemplateNodeDO> dNodes = new ArrayList<>();
        //存放nodeName的集合
//        ArrayList<Object> names = new ArrayList<>();
        //遍历节点集合(该数据为必填，无需判断是否为空）
        nodes.forEach(node -> {
            //绑定模板id
            node.setTemplateId(id);
            //拼接nodeCode字符串并设置
            node.setNodeCode("TEMP-[" + id + "]-" + node.getWbsType() + "-" + node.getWbsCode());
            dNodes.add(PmsWbsTemplateNodeConvert.INSTANCE.toDo(node));
            //将名字加入集合中
//            names.add(node.getWbsName());
        });
        //将list转换为set 如果长度不等则代表有重复，则抛 出同名异常
//        HashSet<Object> set = new HashSet<>(names);
//        if (set.size() != names.size()){
//            throw TwException.error("","节点名不得重复！");
//        }
        checkTemplateNode(payload.getNodes(),PmsWbsTemplateConvert.INSTANCE.toVo(entity).getNodes());
        pmsWbsTemplateNodeDAO.saveAll(dNodes);
    }


    /**
     * 校验数据，必要字段不能为空
     *
     * @param payload 单据数据
     */
    void checkData(PmsWbsTemplatePayload payload) {
        if (ObjectUtils.isEmpty(payload.getTemplateName()) ||
                ObjectUtils.isEmpty(payload.getSuitProjectType()) ||
                ObjectUtils.isEmpty(payload.getTemplateStatus())) {
            throw TwException.error("", "请完善标 * 的相关信息");
        }
    }

    /**
     * 更新
     *
     * @param payload 单据数据
     * @return vo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsWbsTemplateVO update(PmsWbsTemplatePayload payload) {
        //检查数据
        checkData(payload);
        //根据id获取数据库中的数据
        PmsWbsTemplateDO entity = pmsWbsTemplateRepo.findById(payload.getId()).orElseGet(PmsWbsTemplateDO::new);
        //断言notNull
        Assert.notNull(entity.getId(), "不存在");
        //payload -> do
        payload.setSuitProjectType("," + payload.getSuitProjectType() + ",");
        PmsWbsTemplateDO entityDo = PmsWbsTemplateConvert.INSTANCE.toDo(payload);
        //获取主键id
        Long id = entityDo.getId();
        //根据模板id删除节点
        pmsWbsTemplateNodeDAO.deleteSoftByTemplateId(id);
        //修改节点编码
        setNodeCode(payload,entity,id);
        //将do赋给数据库
        entity.copy(entityDo);
        return PmsWbsTemplateConvert.INSTANCE.toVo(pmsWbsTemplateRepo.save(entity));
    }

    /**
     * 逻辑删除
     *
     * @param keys 主键集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        //判断不为空
        if (!keys.isEmpty()) {
            //通过主键查询，得到vo
            List<PmsWbsTemplateVO> pmsWbsTemplateVOS = pmsWbsTemplateDAO.queryByKeys(keys);
            for (PmsWbsTemplateVO pmsWbsTemplateVO : pmsWbsTemplateVOS) {
                //得到模板状态
                String templateStatus = pmsWbsTemplateVO.getTemplateStatus();
                //判断： 激活状态的模版不可删除
                if (templateStatus.equals(WorkFlowStatusEnum.APPROVED_WORK.getCode())) {
                    throw TwException.error("", "激活状态的模版不可删除");
                }
            }
            //执行逻辑删除
            pmsWbsTemplateDAO.deleteSoft(keys);

            //删除模板节点
            pmsWbsTemplateNodeDAO.deleteSoftByTemplateIds(keys);
        }
    }

    /**
     * 修改状态
     *
     * @param keys 单据数据
     * @return 受影响的行数
     */
    @Override
    @Transactional
    public Long updateStatus(List<Long> keys) {
        //判断不为空
        if (!keys.isEmpty()) {
            //根据keys查询得到vos
            List<PmsWbsTemplateVO> pmsWbsTemplateVOS = pmsWbsTemplateDAO.queryByKeys(keys);
            //判断非空
            if (ObjectUtils.isEmpty(pmsWbsTemplateVOS)) {
                throw TwException.error("", "操作数据不存在，请核验！");
            }

            List<String> collect = pmsWbsTemplateVOS.stream().map(PmsWbsTemplateVO::getTemplateStatus).distinct().collect(Collectors.toList());
            if (collect.size() > 1) {
                throw TwException.error("", "相同数据状态可进行该操作，请核验！");
            }
            String templateStatus = collect.get(0);
            String templateStatusNew = templateStatus.equals("0") ? "1" : "0";
            return pmsWbsTemplateDAO.updateStatus(keys, templateStatusNew);
        } else {
            throw TwException.error("", "操作数据不可为空，请核验！");
        }
    }


    /**
     * 核验节点数据
     *
     * @param templateNodePayload
     * @param templateNodeVOS
     */
    void checkTemplateNode(List<PmsWbsTemplateNodePayload> templateNodePayload, List<PmsWbsTemplateNodeVO> templateNodeVOS) {

        List<String> collect = templateNodePayload.stream().map(PmsWbsTemplateNodePayload::getWbsName).distinct().collect(Collectors.toList());
        //获取WbsName判断长度是否相等，如果不等则说明有重复元素，抛异常
        if (templateNodePayload.size() != collect.size()) {
            throw TwException.error("", "节点描述不可重复，请核验！");
        }
        List<String> collect0 = templateNodePayload.stream().map(PmsWbsTemplateNodePayload::getWbsCode).distinct().collect(Collectors.toList());
        //WbsCode判断长度是否相等，如果不等则说明有重复元素，抛异常
        if (templateNodePayload.size() != collect0.size()) {
            throw TwException.error("", "wbs编码不可重复，请核验！");
        }
        //遍历模板节点集合
        templateNodePayload.forEach(entityDo -> {
            //判断wbs编码是否为空
            if (ObjectUtils.isEmpty(entityDo.getWbsCode())) {
                throw TwException.error("", "wbs编码不存在，请核验！");
            }
            //判断wbs类型是否为空
            if (ObjectUtils.isEmpty(entityDo.getWbsType())) {
                throw TwException.error("", "wbs类型不存在，请核验！");
            }
            //判断父节点编码是否为空
            if (ObjectUtils.isEmpty(entityDo.getParentWbsCode())) {
                //如果为空，说明是一级节点，判断一级节点只能是WBS和网络
                if (!entityDo.getWbsType().equals("WBS") && !entityDo.getWbsType().equals("NET")) {
                    throw TwException.error("", "一级节点只可新建“WBS元素”和“网络”，请核验！");
                }
                //如果父节点不为空，则说明是子节点
            } else {
                Optional<PmsWbsTemplateNodePayload> first = templateNodePayload.stream().filter(wbsPayload -> wbsPayload.getWbsCode().equals(entityDo.getParentWbsCode())).findFirst();
                if (first.isPresent()) {
                    //获得父节点的编码
                    PmsWbsTemplateNodePayload parentWbsPayload = first.get();
                    if (entityDo.getWbsType().equals("WBS")) {
                        if (!parentWbsPayload.getWbsType().equals("WBS")) {
                            throw TwException.error("", "WBS元素上级只能是WBS元素，请核验！");
                        }
                    }
                    if (entityDo.getWbsType().equals("NET")) {
                        if (!parentWbsPayload.getWbsType().equals("WBS")) {
                            throw TwException.error("", "网络上级只能是WBS元素，请核验！");
                        }
                    }
                    if (entityDo.getWbsType().equals("ACT")) {
                        if (!parentWbsPayload.getWbsType().equals("WBS") && !parentWbsPayload.getWbsType().equals("NET")) {
                            throw TwException.error("", "活动上级只能是WBS元素或网络，请核验！");
                        }
                    }
                    if (entityDo.getWbsType().equals("MS")) {
                        if (!parentWbsPayload.getWbsType().equals("WBS") && !parentWbsPayload.getWbsType().equals("NET")) {
                            throw TwException.error("", "里程碑上级只能是WBS元素或网络，请核验！");
                        }
                    }
                } else {
                    throw TwException.error("", entityDo.getParentWbsCode() + "编码数据不存在，请核验！");
                }
            }
            if (!ObjectUtils.isEmpty(templateNodeVOS)) {

                PmsWbsTemplateNodeVO first = null;
                PmsWbsTemplateNodeVO first0 = null;
                for (PmsWbsTemplateNodeVO wbsVO : templateNodeVOS) {
                    Optional<PmsWbsTemplateNodePayload> firstPayload = templateNodePayload.stream().filter(wbsPayload -> wbsPayload.getId().equals(wbsVO.getId())).findFirst();
                    if (firstPayload.isPresent()) {
                        wbsVO.setWbsCode(firstPayload.get().getWbsCode());
                        wbsVO.setWbsName(firstPayload.get().getWbsName());
                    }
                    if (wbsVO.getWbsCode().equals(entityDo.getWbsCode())) {
                        first = wbsVO;
                    }
                    if (wbsVO.getWbsName().equals(entityDo.getWbsName())) {
                        first0 = wbsVO;
                    }
                }
                if (first != null) {
                    if (entityDo.getId() == null || !first.getId().equals(entityDo.getId())) {
                        throw TwException.error("", "【" + entityDo.getWbsName() + "】wbs编码已存在，请核验！");
                    }
                }
                if (first0 != null) {
                    if (entityDo.getId() == null || !first0.getId().equals(entityDo.getId())) {
                        throw TwException.error("", "【" + entityDo.getWbsName() + "】节点描述已存在，请核验！");
                    }
                }
            }

        });
    }
}
