package com.elitesland.tw.tw5.server.prd.adm.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.adm.payload.AdmTripTicketPayload;
import com.elitesland.tw.tw5.api.prd.adm.query.AdmTripTicketQuery;
import com.elitesland.tw.tw5.api.prd.adm.vo.AdmTripTicketVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccReimDO;
import com.elitesland.tw.tw5.server.prd.adm.entity.AdmTripTicketDO;
import com.elitesland.tw.tw5.server.prd.adm.entity.QAdmBusitripApplyDO;
import com.elitesland.tw.tw5.server.prd.adm.entity.QAdmTripTicketDO;
import com.elitesland.tw.tw5.server.prd.adm.repo.AdmTripTicketRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 行政订票管理
 *
 * @author carl
 * @date 2023-10-30
 */
@Repository
@RequiredArgsConstructor
public class AdmTripTicketDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AdmTripTicketRepo repo;
    private final QAdmTripTicketDO qdo = QAdmTripTicketDO.admTripTicketDO;
    private final QAdmBusitripApplyDO qdoApply = QAdmBusitripApplyDO.admBusitripApplyDO;
    private final QAccReimDO qAccReimDO = QAccReimDO.accReimDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AdmTripTicketVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AdmTripTicketVO.class,
                        qdo.id,
                        qdo.remark,
                        qdo.createUserId,
                        qdo.creator,
                        qdo.createTime,
                        qdo.modifyUserId,
                        //qdo.updater,
                        //qdo.modifyTime,
                        // 申请id
                        qdo.applyId,
                        // 订票费用类型
                        qdo.ticketExpType,
                        // 出差人资源id
                        qdo.tripResId,
                        // 订票人资源id
                        qdo.bookingResId,
                        // 出发地
                        qdo.fromPlace,
                        // 目的地
                        qdo.toPlace,
                        // 交通工具
                        qdo.vehicle,
                        // 出差日期
                        qdo.tripDate,
                        // 时间段
                        qdo.timespan,
                        // 订票日期
                        qdo.bookingDate,
                        // 车次航班
                        qdo.vehicleNo,
                        // 金额
                        qdo.expAmt,
                        // 票务使用状态
                        qdo.useStatus,
                        // 购票渠道
                        qdo.ticketPurchasingChannel,
                        // 报销状态
                        qdo.reimbursementStatus,
                        qdoApply.applyName,
                        qdoApply.applyNo,
                        qAccReimDO.createTime.as("reimTime"),
                        qAccReimDO.expenseByType
                )).from(qdo).leftJoin(qdoApply).on(qdo.applyId.eq(qdoApply.id))
                .leftJoin(qAccReimDO).on(qAccReimDO.id.eq(qdo.reimId));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AdmTripTicketVO> getJpaQueryWhere(AdmTripTicketQuery query) {
        JPAQuery<AdmTripTicketVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AdmTripTicketQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo).leftJoin(qdoApply).on(qdo.applyId.eq(qdoApply.id));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AdmTripTicketQuery query) {
        List<Predicate> list = new ArrayList<>();
        list.add(qdoApply.deleteFlag.eq(0));
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 申请id 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyId())) {
            list.add(qdo.applyId.eq(query.getApplyId()));
        }
        /** 申请单名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getApplyName())) {
            list.add(qdoApply.applyName.like(SqlUtil.toSqlLikeString(query.getApplyName())));
        }
        /** 订票费用类型 精确 */
        if (!ObjectUtils.isEmpty(query.getTicketExpType())) {
            list.add(qdo.ticketExpType.eq(query.getTicketExpType()));
        }
        /** 出差人资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getTripResId())) {
            list.add(qdo.tripResId.eq(query.getTripResId()));
        }
        /** 订票人资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getBookingResId())) {
            list.add(qdo.bookingResId.eq(query.getBookingResId()));
        }
        /** 出发地 精确 */
        if (!ObjectUtils.isEmpty(query.getFromPlace())) {
            list.add(qdo.fromPlace.eq(query.getFromPlace()));
        }
        /** 目的地 精确 */
        if (!ObjectUtils.isEmpty(query.getToPlace())) {
            list.add(qdo.toPlace.eq(query.getToPlace()));
        }
        /** 交通工具 精确 */
        if (!ObjectUtils.isEmpty(query.getVehicle())) {
            list.add(qdo.vehicle.eq(query.getVehicle()));
        }
        /** 出差日期 精确 */
        if (!ObjectUtils.isEmpty(query.getTripBeginDate())) {
            list.add(qdo.tripDate.goe(query.getTripBeginDate()));
        }
        /** 出差日期 精确 */
        if (!ObjectUtils.isEmpty(query.getTripEndDate())) {
            list.add(qdo.tripDate.loe(query.getTripEndDate()));
        }
        /** 时间段 精确 */
        if (!ObjectUtils.isEmpty(query.getTimespan())) {
            list.add(qdo.timespan.eq(query.getTimespan()));
        }
        /** 订票日期 精确 */
        if (!ObjectUtils.isEmpty(query.getBookingBeginDate())) {
            list.add(qdo.bookingDate.goe(query.getBookingBeginDate()));
        }
        /** 订票日期 精确 */
        if (!ObjectUtils.isEmpty(query.getBookingEndDate())) {
            list.add(qdo.bookingDate.loe(query.getBookingEndDate()));
        }
        /** 车次航班 模糊 */
        if (!ObjectUtils.isEmpty(query.getVehicleNo())) {
            list.add(qdo.vehicleNo.like(SqlUtil.toSqlLikeString(query.getVehicleNo())));
        }
        /** 金额 精确 */
        if (!ObjectUtils.isEmpty(query.getExpAmt())) {
            list.add(qdo.expAmt.eq(query.getExpAmt()));
        }
        /** 票务使用状态 精确 */
        if (!ObjectUtils.isEmpty(query.getUseStatus())) {
            list.add(qdo.useStatus.eq(query.getUseStatus()));
        }
        /** 购票渠道 精确 */
        if (!ObjectUtils.isEmpty(query.getTicketPurchasingChannel())) {
            list.add(qdo.ticketPurchasingChannel.eq(query.getTicketPurchasingChannel()));
        }
        /** 报销状态 精确 */
        if (!ObjectUtils.isEmpty(query.getReimbursementStatus())) {
            list.add(qdo.reimbursementStatus.eq(query.getReimbursementStatus()));
        }
        //id集合
        if (!ObjectUtils.isEmpty(query.getIdList())) {
            list.add(qdo.id.in(query.getIdList()));
        }
        //报销单id
        if (!ObjectUtils.isEmpty(query.getReimId())) {
            list.add(qdo.reimId.eq(query.getReimId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AdmTripTicketVO queryByKey(Long id) {
        JPAQuery<AdmTripTicketVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AdmTripTicketVO> queryListDynamic(AdmTripTicketQuery query) {
        JPAQuery<AdmTripTicketVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AdmTripTicketVO> queryPaging(AdmTripTicketQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AdmTripTicketVO> jpaQuery = getJpaQueryWhere(query);
        List<AdmTripTicketVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AdmTripTicketVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AdmTripTicketDO save(AdmTripTicketDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AdmTripTicketDO> saveAll(List<AdmTripTicketDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AdmTripTicketPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 申请id
        if (payload.getApplyId() != null) {
            update.set(qdo.applyId, payload.getApplyId());
        }
        // 订票费用类型
        if (payload.getTicketExpType() != null) {
            update.set(qdo.ticketExpType, payload.getTicketExpType());
        }
        // 出差人资源id
        if (payload.getTripResId() != null) {
            update.set(qdo.tripResId, payload.getTripResId());
        }
        // 订票人资源id
        if (payload.getBookingResId() != null) {
            update.set(qdo.bookingResId, payload.getBookingResId());
        }
        // 出发地
        if (payload.getFromPlace() != null) {
            update.set(qdo.fromPlace, payload.getFromPlace());
        }
        // 目的地
        if (payload.getToPlace() != null) {
            update.set(qdo.toPlace, payload.getToPlace());
        }
        // 交通工具
        if (payload.getVehicle() != null) {
            update.set(qdo.vehicle, payload.getVehicle());
        }
        // 出差日期
        if (payload.getTripDate() != null) {
            update.set(qdo.tripDate, payload.getTripDate());
        }
        // 时间段
        if (payload.getTimespan() != null) {
            update.set(qdo.timespan, payload.getTimespan());
        }
        // 订票日期
        if (payload.getBookingDate() != null) {
            update.set(qdo.bookingDate, payload.getBookingDate());
        }
        // 车次航班
        if (payload.getVehicleNo() != null) {
            update.set(qdo.vehicleNo, payload.getVehicleNo());
        }
        // 金额
        if (payload.getExpAmt() != null) {
            update.set(qdo.expAmt, payload.getExpAmt());
        }
        // 票务使用状态
        if (payload.getUseStatus() != null) {
            update.set(qdo.useStatus, payload.getUseStatus());
        }
        // 购票渠道
        if (payload.getTicketPurchasingChannel() != null) {
            update.set(qdo.ticketPurchasingChannel, payload.getTicketPurchasingChannel());
        }
        // 报销状态
        if (payload.getReimbursementStatus() != null) {
            update.set(qdo.reimbursementStatus, payload.getReimbursementStatus());
        }
        if (payload.getReimId() != null) {
            update.set(qdo.reimId, payload.getReimId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 申请id
            if (nullFields.contains("applyId")) {
                update.setNull(qdo.applyId);
            }
            // 订票费用类型
            if (nullFields.contains("ticketExpType")) {
                update.setNull(qdo.ticketExpType);
            }
            // 出差人资源id
            if (nullFields.contains("tripResId")) {
                update.setNull(qdo.tripResId);
            }
            // 订票人资源id
            if (nullFields.contains("bookingResId")) {
                update.setNull(qdo.bookingResId);
            }
            // 出发地
            if (nullFields.contains("fromPlace")) {
                update.setNull(qdo.fromPlace);
            }
            // 目的地
            if (nullFields.contains("toPlace")) {
                update.setNull(qdo.toPlace);
            }
            // 交通工具
            if (nullFields.contains("vehicle")) {
                update.setNull(qdo.vehicle);
            }
            // 出差日期
            if (nullFields.contains("tripDate")) {
                update.setNull(qdo.tripDate);
            }
            // 时间段
            if (nullFields.contains("timespan")) {
                update.setNull(qdo.timespan);
            }
            // 订票日期
            if (nullFields.contains("bookingDate")) {
                update.setNull(qdo.bookingDate);
            }
            // 车次航班
            if (nullFields.contains("vehicleNo")) {
                update.setNull(qdo.vehicleNo);
            }
            // 金额
            if (nullFields.contains("expAmt")) {
                update.setNull(qdo.expAmt);
            }
            // 票务使用状态
            if (nullFields.contains("useStatus")) {
                update.setNull(qdo.useStatus);
            }
            // 购票渠道
            if (nullFields.contains("ticketPurchasingChannel")) {
                update.setNull(qdo.ticketPurchasingChannel);
            }
            // 报销状态
            if (nullFields.contains("reimbursementStatus")) {
                update.setNull(qdo.reimbursementStatus);
            }
            // 报销单ID
            if (nullFields.contains("reimId")) {
                update.setNull(qdo.reimId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据出差申请id逻辑删除
     *
     * @param applyId 出差申请id
     * @return 删除的行数
     */
    public long deleteSoftByApplyId(Long applyId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.applyId.eq(applyId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据出差申请单查询
     *
     * @param applyId
     * @return
     */
    public List<AdmTripTicketVO> queryListByApplyId(Long applyId) {
        JPAQuery<AdmTripTicketVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.applyId.eq(applyId));
        return jpaQuery.fetch();
    }
}

