package com.elitesland.tw.tw5.server.prd.budget.controller;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.budget.payload.BudgetChangePayload;
import com.elitesland.tw.tw5.api.prd.budget.payload.BudgetPayload;
import com.elitesland.tw.tw5.api.prd.budget.query.BudgetQuery;
import com.elitesland.tw.tw5.api.prd.budget.service.BudgetCommonService;
import com.elitesland.tw.tw5.api.prd.budget.service.BudgetService;
import com.elitesland.tw.tw5.api.prd.budget.vo.BudgetChangeVO;
import com.elitesland.tw.tw5.api.prd.budget.vo.BudgetUsageDetailsVO;
import com.elitesland.tw.tw5.api.prd.budget.vo.BudgetVO;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectVO;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.udc.UdcNameClass;
import io.seata.spring.annotation.GlobalTransactional;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.Arrays;
import java.util.List;

/**
 * 预算基本信息
 *
 * @author xxb
 * @date 2023-11-01
 */
@Api(tags = "预算基本信息--2.0")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/budget")
@Slf4j
public class BudgetController {

    private final BudgetService budgetService;

    private final BudgetCommonService budgetCommonService;


    /**
     * 通过 资源 查询预算
     *
     * @param sourceId
     * @param sourceType
     * @return
     */
    @GetMapping("/querySimpleBySource")
    @UdcNameClass
    @ApiOperation("通过 资源 查询预算--简单预算查询")
    public TwOutputUtil<BudgetVO> querySimpleBySource(Long sourceId, String sourceType) {
        return TwOutputUtil.ok(budgetCommonService.querySimpleBySource(sourceId, sourceType));
    }

    /**
     * 通过 资源 查询预算
     *
     * @param sourceId
     * @param sourceType
     * @return
     */
    @GetMapping("/queryBySource")
    @UdcNameClass
    @ApiOperation("通过 资源 查询预算--预算详情")
    public TwOutputUtil<BudgetVO> queryBySource(Long sourceId, String sourceType) {
        return TwOutputUtil.ok(budgetService.queryBySource(sourceId, sourceType));
    }

    /**
     * 首次预算-新增或修改
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/saveFirstBudget")
    @ApiOperation("首次预算-新增或修改")
//    @GlobalTransactional
    public TwOutputUtil saveFirstBudget(@RequestBody BudgetPayload payload) {
        return TwOutputUtil.ok(budgetService.saveFirstBudget(payload));
    }


    /**
     * BU预算、商机预算 -新增或修改
     *
     * @param payload
     * @return
     */
    @PostMapping("/saveOPPOAndBUTypeBudget")
    @ApiOperation("BU预算、商机预算 -新增或修改")
    @GlobalTransactional
    public TwOutputUtil saveOPPOAndBUTypeBudget(@RequestBody BudgetPayload payload) {
        budgetService.saveOPPOAndBUTypeBudget(payload);
        return TwOutputUtil.ok();
    }


    /**
     * 预算信息-预算主键查询
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryBudgetByKey")
    @UdcNameClass
    @ApiOperation("预算信息-预算主键查询")
    public TwOutputUtil<BudgetVO> queryBudgetByKey(Long key) {
        return TwOutputUtil.ok(budgetService.queryBudgetByKey(key));
    }

    /**
     * 预算变更-新增或修改
     *
     * @param budgetChangePayload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/saveBudgetChange")
    @ApiOperation("预算变更-新增或修改")
    @GlobalTransactional
    public TwOutputUtil saveBudgetChange(@RequestBody BudgetChangePayload budgetChangePayload) {
        budgetService.saveBudgetChange(budgetChangePayload);
        return TwOutputUtil.ok();
    }

    /**
     * 预算变更信息-变更主键查询
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryBudgetChangeByKey")
    @UdcNameClass
    @ApiOperation("预算变更信息-变更主键查询")
    public TwOutputUtil<BudgetChangeVO> queryBudgetChangeByKey(Long key) {
        return TwOutputUtil.ok(budgetService.queryBudgetChangeByKey(key));
    }


    /**
     * 预算基本信息-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/paging")
    @UdcNameClass
    @ApiOperation("预算基本信息-分页")
    public TwOutputUtil<PagingVO<BudgetVO>> paging(BudgetQuery query) {
        return TwOutputUtil.ok(budgetService.queryPaging(query));
    }


    /**
     * 预算列表
     *
     * @return
     */
    @GetMapping("/list")
    @UdcNameClass
    @ApiOperation("预算基本信息-列表")
    public TwOutputUtil<List<PmsProjectVO>> queryBudgetList(BudgetQuery query) {
        return TwOutputUtil.ok(budgetService.queryList(query));
    }


    /**
     * 根据科目模板id查找预算科目
     *
     * @param subjectTemplateId
     * @return
     */
    @GetMapping("/queryBySubjectTemplateId")
    @UdcNameClass
    @ApiOperation("根据科目模板id查找预算科目")
    public TwOutputUtil<List<BudgetUsageDetailsVO>> queryBySubjectTemplateId(String sourceType, Long subjectTemplateId) {
        return TwOutputUtil.ok(budgetCommonService.queryBySubjectTemplateId(sourceType, subjectTemplateId));
    }

    /**
     * 预算基本信息-删除
     *
     * @param keys 主键id
     * @return result
     */
    @DeleteMapping("/deleteSoft")
    @ApiOperation("预算基本信息-删除")

    public TwOutputUtil deleteSoft(Long[] keys) {
        budgetService.deleteSoft(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 预算信息-预算主键查询（为了拨付）
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryBudgetForAppropriation")
    @UdcNameClass
    @ApiOperation("预算信息-预算主键查询（为了拨付）")
    public TwOutputUtil<BudgetVO> queryBudgetForAppropriation(Long key) {
        return TwOutputUtil.ok(budgetCommonService.queryBudgetForAppropriation(key));
    }

    /**
     * 没有建立过预算的项目列表
     *
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryProjectNoBudget")
    // @UdcNameClass
    @ApiOperation("没有建立过预算的项目列表")
    public TwOutputUtil<List<PmsProjectVO>> queryProjectNoBudget() {
        return TwOutputUtil.ok(budgetService.queryProjectNoBudget());
    }


    @GetMapping("/queryBudgetUsageDetails")
    @ApiOperation("当量、费用 使用情况")
    public TwOutputUtil<List<BudgetUsageDetailsVO>> queryBudgetUsageDetails(Long sourceId, String sourceType) {
        return TwOutputUtil.ok(budgetCommonService.queryBudgetUsageDetails(sourceId, sourceType));
    }

    /**
     * 同步科目模板
     *
     * @param keys   预算主键
     * @param tmplId 模板id
     * @return result
     */
    @GetMapping("/synchronizationTemp")
    @ApiOperation("同步科目模板")
    public TwOutputUtil synchronizationTemp(Long[] keys, Long tmplId) {
        budgetService.synchronizationTemp(Arrays.asList(keys), tmplId);
        return TwOutputUtil.ok();
    }

}
