package com.elitesland.tw.tw5.server.prd.budget.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.budget.payload.BudgetAppropriationDetailPayload;
import com.elitesland.tw.tw5.api.prd.budget.query.BudgetAppropriationDetailQuery;
import com.elitesland.tw.tw5.api.prd.budget.vo.BudgetAppropriationDetailVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.budget.entity.BudgetAppropriationDetailDO;
import com.elitesland.tw.tw5.server.prd.budget.entity.QBudgetAppropriationDetailDO;
import com.elitesland.tw.tw5.server.prd.budget.repo.BudgetAppropriationDetailRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 预算拨付明细
 *
 * @author xxb
 * @date 2023-11-01
 */
@Repository
@RequiredArgsConstructor
public class BudgetAppropriationDetailDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BudgetAppropriationDetailRepo repo;
    private final QBudgetAppropriationDetailDO qdo = QBudgetAppropriationDetailDO.budgetAppropriationDetailDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BudgetAppropriationDetailVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BudgetAppropriationDetailVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 拨款ID
                qdo.appropriationId,
                // 拨款金额
                qdo.allocateCost,
                // 活动id
                qdo.activityId,
                // 拨付当量
                qdo.allocateEqva,
                // 预算科目明细ID
                qdo.budgetSubjectDetailId,
                // 可配置字段1
                qdo.configurableField1,
                // 可配置字段2
                qdo.configurableField2,
                // 可配置字段3
                qdo.configurableField3
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BudgetAppropriationDetailVO> getJpaQueryWhere(BudgetAppropriationDetailQuery query) {
        JPAQuery<BudgetAppropriationDetailVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BudgetAppropriationDetailQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BudgetAppropriationDetailQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 拨款ID 精确 */
        if (!ObjectUtils.isEmpty(query.getAppropriationId())) {
            list.add(qdo.appropriationId.eq(query.getAppropriationId()));
        }
        /** 拨款金额 精确 */
        if (!ObjectUtils.isEmpty(query.getAllocateCost())) {
            list.add(qdo.allocateCost.eq(query.getAllocateCost()));
        }
        /** 活动id 精确 */
        if (!ObjectUtils.isEmpty(query.getActivityId())) {
            list.add(qdo.activityId.eq(query.getActivityId()));
        }
        /** 拨付当量 精确 */
        if (!ObjectUtils.isEmpty(query.getAllocateEqva())) {
            list.add(qdo.allocateEqva.eq(query.getAllocateEqva()));
        }
        /** 预算科目明细ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBudgetSubjectDetailId())) {
            list.add(qdo.budgetSubjectDetailId.eq(query.getBudgetSubjectDetailId()));
        }
        /** 可配置字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigurableField1())) {
            list.add(qdo.configurableField1.eq(query.getConfigurableField1()));
        }
        /** 可配置字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigurableField2())) {
            list.add(qdo.configurableField2.eq(query.getConfigurableField2()));
        }
        /** 可配置字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigurableField3())) {
            list.add(qdo.configurableField3.eq(query.getConfigurableField3()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BudgetAppropriationDetailVO queryByKey(Long id) {
        JPAQuery<BudgetAppropriationDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BudgetAppropriationDetailVO> queryListDynamic(BudgetAppropriationDetailQuery query) {
        JPAQuery<BudgetAppropriationDetailVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BudgetAppropriationDetailVO> queryPaging(BudgetAppropriationDetailQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<BudgetAppropriationDetailVO> jpaQuery = getJpaQueryWhere(query);
        List<BudgetAppropriationDetailVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BudgetAppropriationDetailVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BudgetAppropriationDetailDO save(BudgetAppropriationDetailDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BudgetAppropriationDetailDO> saveAll(List<BudgetAppropriationDetailDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BudgetAppropriationDetailPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 拨款ID
        if (payload.getAppropriationId() != null) {
            update.set(qdo.appropriationId, payload.getAppropriationId());
        }
        // 拨款金额
        if (payload.getAllocateCost() != null) {
            update.set(qdo.allocateCost, payload.getAllocateCost());
        }
        // 活动id
        if (payload.getActivityId() != null) {
            update.set(qdo.activityId, payload.getActivityId());
        }
        // 拨付当量
        if (payload.getAllocateEqva() != null) {
            update.set(qdo.allocateEqva, payload.getAllocateEqva());
        }
        // 预算科目明细ID
        if (payload.getBudgetSubjectDetailId() != null) {
            update.set(qdo.budgetSubjectDetailId, payload.getBudgetSubjectDetailId());
        }
        // 可配置字段1
        if (payload.getConfigurableField1() != null) {
            update.set(qdo.configurableField1, payload.getConfigurableField1());
        }
        // 可配置字段2
        if (payload.getConfigurableField2() != null) {
            update.set(qdo.configurableField2, payload.getConfigurableField2());
        }
        // 可配置字段3
        if (payload.getConfigurableField3() != null) {
            update.set(qdo.configurableField3, payload.getConfigurableField3());
        }
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 拨款ID
            if (nullFields.contains("appropriationId")) {
                update.setNull(qdo.appropriationId);
            }
            // 拨款金额
            if (nullFields.contains("allocateCost")) {
                update.setNull(qdo.allocateCost);
            }
            // 活动id
            if (nullFields.contains("activityId")) {
                update.setNull(qdo.activityId);
            }
            // 拨付当量
            if (nullFields.contains("allocateEqva")) {
                update.setNull(qdo.allocateEqva);
            }
            // 预算科目明细ID
            if (nullFields.contains("budgetSubjectDetailId")) {
                update.setNull(qdo.budgetSubjectDetailId);
            }
            // 可配置字段1
            if (nullFields.contains("configurableField1")) {
                update.setNull(qdo.configurableField1);
            }
            // 可配置字段2
            if (nullFields.contains("configurableField2")) {
                update.setNull(qdo.configurableField2);
            }
            // 可配置字段3
            if (nullFields.contains("configurableField3")) {
                update.setNull(qdo.configurableField3);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据主键数据逻辑删除
     *
     * @param appId 主键
     */
    public long deleteSoftByAppId(Long appId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.appropriationId.eq(appId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}

