package com.elitesland.tw.tw5.server.prd.crm.service;


import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmGiftInfoExcelExport;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmGiftInfoPayload;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmPeoplePayload;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmGiftInfoQuery;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmGiftListQuery;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmGiftInfoService;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmPeopleService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmGiftInfoVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeRefVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.crm.convert.CrmGiftInfoConvert;
import com.elitesland.tw.tw5.server.prd.crm.dao.CrmGiftInfoDAO;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmGiftInfoDO;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmGiftInfoRepo;
import com.elitesland.tw.tw5.server.prd.my.dao.PrdUserDAO;
import com.elitesland.tw.tw5.server.prd.system.dao.PrdSystemRoleDAO;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;


/**
 * 礼品单明细
 *
 * @author xxb
 * @date 2023-08-31
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class CrmGiftInfoServiceImpl extends BaseServiceImpl implements CrmGiftInfoService {

    private final CrmGiftInfoRepo crmGiftInfoRepo;
    private final CrmGiftInfoDAO crmGiftInfoDAO;

    private final PrdSystemRoleDAO systemRoleDAO;

    private final PrdUserDAO prdUserDAO;

    private final UdcUtil udcUtil;

    private final CacheUtil cacheUtil;

    private final CrmPeopleService crmPeopleService;

    @Override
    public PagingVO<CrmGiftInfoVO> queryPaging(CrmGiftInfoQuery query) {
        return crmGiftInfoDAO.queryPaging(query);
    }

    @Override
    public PagingVO<CrmGiftInfoVO> pagingByGiftList(CrmGiftListQuery query) {
        //自己只能查看自己的礼品单，直属领导能看到自己的和下属的礼品单，刘志军可以查看所有的礼品单
        Long loginUserId = GlobalUtil.getLoginUserId();
        List<String> roleCodes = systemRoleDAO.queryUserRoleCodes(loginUserId);
        if (roleCodes.contains(RoleEnum.GIFT_APPLY_QUERY_ALL.getCode())) {
            PagingVO<CrmGiftInfoVO> pagingVO = crmGiftInfoDAO.pagingByGiftList(query);
            if (pagingVO != null && !ObjectUtils.isEmpty(pagingVO.getRecords())) {
                pagingVO.getRecords().forEach(v -> transferDatas(v));
            }
            return pagingVO;
        }
        getUser(query, loginUserId);
        PagingVO<CrmGiftInfoVO> pagingVO = crmGiftInfoDAO.pagingByGiftList(query);
        if (pagingVO != null && !ObjectUtils.isEmpty(pagingVO.getRecords())) {
            pagingVO.getRecords().forEach(v -> transferDatas(v));
        }
        return pagingVO;
    }

    @Override
    public List<CrmGiftInfoVO> byGiftList(CrmGiftListQuery query) {
        //自己只能查看自己的礼品单，直属领导能看到自己的和下属的礼品单，刘志军可以查看所有的礼品单
        Long loginUserId = GlobalUtil.getLoginUserId();
        List<String> roleCodes = systemRoleDAO.queryUserRoleCodes(loginUserId);
        if (roleCodes.contains(RoleEnum.GIFT_APPLY_QUERY_ALL.getCode())) {
            List<CrmGiftInfoVO> list = crmGiftInfoDAO.byGiftList(query);
            list.forEach(v -> transferDatas(v));
            return list;
        }
        getUser(query, loginUserId);
        List<CrmGiftInfoVO> list = crmGiftInfoDAO.byGiftList(query);
        list.forEach(v -> transferDatas(v));
        return list;
    }

    @Override
    public List<CrmGiftInfoVO> queryListDynamic(CrmGiftInfoQuery query) {
        if (ObjectUtils.isEmpty(query.getGiftListId())) {
            return new ArrayList<>();
        }
        return crmGiftInfoDAO.queryListDynamic(query);
    }

    @Override
    public CrmGiftInfoVO queryByKey(Long key) {
        CrmGiftInfoDO entity = crmGiftInfoRepo.findById(key).orElseGet(CrmGiftInfoDO::new);
        Assert.notNull(entity.getId(), "不存在");
        CrmGiftInfoVO vo = CrmGiftInfoConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmGiftInfoVO insert(CrmGiftInfoPayload payload) {
        //验证数据
        checkDatas(payload);
        CrmGiftInfoDO entityDo = CrmGiftInfoConvert.INSTANCE.toDo(payload);
        return CrmGiftInfoConvert.INSTANCE.toVo(crmGiftInfoRepo.save(entityDo));
    }

    @Override
    public void batchInsert(List<CrmGiftInfoPayload> crmGiftInfoPayloads) {
        List<CrmGiftInfoDO> infoDOS = new ArrayList<>();
        for (CrmGiftInfoPayload payload : crmGiftInfoPayloads) {
            //验证数据
            checkDatas(payload);
            infoDOS.add(CrmGiftInfoConvert.INSTANCE.toDo(payload));
        }
        crmGiftInfoRepo.saveAll(infoDOS);
    }

    @Override
    public void batchUpdate(List<CrmGiftInfoPayload> crmGiftInfoPayloads) {
        List<CrmGiftInfoDO> insertList = new ArrayList<>();
        List<CrmGiftInfoDO> updateList = new ArrayList<>();
        List<Long> updateIds = new ArrayList<>();
        for (CrmGiftInfoPayload payload : crmGiftInfoPayloads) {
            //验证数据
            checkDatas(payload);
            if (ObjectUtils.isEmpty(payload.getId())) {
                insertList.add(CrmGiftInfoConvert.INSTANCE.toDo(payload));
            } else {
                updateList.add(CrmGiftInfoConvert.INSTANCE.toDo(payload));
                updateIds.add(payload.getId());
            }
        }
        // 判断下 数据库里是否有旧值  好确定 是否有要删除的
        CrmGiftInfoQuery query = new CrmGiftInfoQuery();
        query.setGiftListId(crmGiftInfoPayloads.get(0).getGiftListId());
        List<CrmGiftInfoVO> fromDB = queryListDynamic(query);
        if (!ObjectUtils.isEmpty(fromDB)) {
            List<Long> delList = fromDB.stream().filter(v -> !updateIds.contains(v.getId())).map(v -> v.getId()).collect(Collectors.toList());
            deleteSoft(delList);
        }

        //  新增 或修改
        if (!ObjectUtils.isEmpty(insertList)) {
            crmGiftInfoDAO.saveAll(insertList);
        }
        if (!ObjectUtils.isEmpty(updateList)) {
            crmGiftInfoDAO.saveAll(updateList);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmGiftInfoVO update(CrmGiftInfoPayload payload) {
        //验证数据
        checkDatas(payload);
        CrmGiftInfoDO entity = crmGiftInfoRepo.findById(payload.getId()).orElseGet(CrmGiftInfoDO::new);
        Assert.notNull(entity.getId(), "不存在");
        CrmGiftInfoDO entityDo = CrmGiftInfoConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return CrmGiftInfoConvert.INSTANCE.toVo(crmGiftInfoRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(CrmGiftInfoPayload payload) {
        //验证数据
        checkDatas(payload);
        CrmGiftInfoDO entity = crmGiftInfoRepo.findById(payload.getId()).orElseGet(CrmGiftInfoDO::new);
        Assert.notNull(entity.getId(), "不存在");
        long result = crmGiftInfoDAO.updateByKeyDynamic(payload);
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            crmGiftInfoDAO.deleteSoft(keys);
        }
    }

    @Override
    public void downloadBatch(HttpServletResponse response, CrmGiftListQuery query) {
        //定义文件名称
        String sheetName = "礼品单列表";
        //导出顺序是反的
        OrderItem orderItem = OrderItem.desc("applyDate");
        query.defaultOrder(orderItem);
        //获取想要导出的数据
        List<CrmGiftInfoVO> records = byGiftList(query);
        List<CrmGiftInfoExcelExport> resultList = CrmGiftInfoConvert.INSTANCE.voListVoExcelExport(records);
        resultList = udcUtil.translateList(resultList);
        //对文件名进行固定格式编码
        String fileName = null;
        try {
            fileName = URLEncoder.encode(sheetName + System.currentTimeMillis() + ".xlsx", "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        //设置请求响应内容类型
        //作用:使客户端浏览器，区分不同种类的数据，并根据不同的MIME调用浏览器内不同的程序嵌入模块来处理相应的数据。
        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        //设置请求响应内容编码方式
        response.setCharacterEncoding("utf-8");
        //文件下载，指定默认名
        response.addHeader("Content-Disposition", "attachment;filename=" + fileName);

        final ExcelWriterSheetBuilder sheet;
        try {
            sheet = EasyExcel.write(response.getOutputStream(), CrmGiftInfoVO.class)
                    .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                    .sheet(sheetName);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        // 列
        com.elitesland.tw.tw5.server.common.excel.ExcelUtil.excelHelper(sheet, CrmGiftInfoExcelExport.class, null);
        //写入
        resultList.forEach(v -> v.setApplyDateString(v.getApplyDate().format(DateTimeFormatter.ofPattern("yyyy-MM-dd"))));
        sheet.doWrite(resultList);
    }


    /**
     * 获取 申请人集合
     *
     * @param query
     * @param loginUserId
     */
    private void getUser(CrmGiftListQuery query, Long loginUserId) {
        List<Long> userIds = new ArrayList<>();
        userIds.add(loginUserId);
        // 查出来userId的所有下级userId
        final List<PrdOrgEmployeeRefVO> empRef = prdUserDAO.queryLowListByKey(null, loginUserId);
        empRef.forEach(v -> {
            if (!userIds.contains(v.getUserId())) {
                userIds.add(v.getUserId());
            }
        });
        query.setUserIds(userIds);
    }

    /**
     * 全局翻译
     *
     * @param vo
     */
    void transferDatas(CrmGiftInfoVO vo) {
        vo.setGiftTypeName1(cacheUtil.transferSystemSelection("GIFT_TYPE", vo.getGiftType1()));
        vo.setGiftTypeName2(cacheUtil.transferSystemSelection("GIFT_TYPE:Festival", vo.getGiftType2()));
        vo.setApplyStatusName(cacheUtil.transferSystemSelection("gift:status", vo.getApplyStatus()));
        vo.setCreator(cacheUtil.getUserName(vo.getCreateUserId()));
        if (ObjectUtils.isEmpty(vo.getGiftType2())) {
            vo.setGiftTypeName(vo.getGiftTypeName1());
        } else {
            vo.setGiftTypeName(vo.getGiftTypeName1() + "/" + vo.getGiftTypeName2());
        }
    }

    /**
     * 验证数据是否为空
     */
    void checkDatas(CrmGiftInfoPayload payload) {
        if (ObjectUtils.isEmpty(payload.getGiftListId())) {
            throw TwException.error("", "礼品单ID不可为空");
        }
        if (ObjectUtils.isEmpty(payload.getSpecialUsersFlag())) {
            throw TwException.error("", "特殊申请用户标志不可为空");
        }
        if (ObjectUtils.isEmpty(payload.getGiftName())) {
            throw TwException.error("", "礼品名称不可为空");
        }
        if (ObjectUtils.isEmpty(payload.getGiftNumber())) {
            throw TwException.error("", "申请数量不可为空");
        }
        if (ObjectUtils.isEmpty(payload.getPeopleName())) {
            throw TwException.error("", "联系人不可为空");
        }
        if (ObjectUtils.isEmpty(payload.getMobile())) {
            throw TwException.error("", "联系方式不可为空");
        }
        if (!isValidPhoneNumber(payload.getMobile())) {
            throw TwException.error("", "手机号无效");
        }
        if (ObjectUtils.isEmpty(payload.getLocationDetail())) {
            throw TwException.error("", "礼品寄送地址不可为空");
        }
        if (1 != payload.getSpecialUsersFlag().intValue()) {
            if (ObjectUtils.isEmpty(payload.getCustomerId())) {
                throw TwException.error("", "客户方不可为空");
            }
            if (ObjectUtils.isEmpty(payload.getCustName())) {
                throw TwException.error("", "客户方不可为空");
            }
            if (ObjectUtils.isEmpty(payload.getReason())) {
                throw TwException.error("", "赠送理由不可为空");
            }
            //创建人脉
            createPeople(payload);

        } else {
            if (!ObjectUtils.isEmpty(payload.getCustomerId())) {
                //创建人脉
                createPeople(payload);
            }
        }
    }

    /**
     * 创建人脉
     *
     * @param payload
     */
    void createPeople(CrmGiftInfoPayload payload) {
        // 普通销售客户  无联系人，无联系方式，可手输，并反向添加到人脉经营中（如联系人不是自己需要的，等于无联系人，无联系方式）
        if (ObjectUtils.isEmpty(payload.getId()) && ObjectUtils.isEmpty(payload.getPeopleId())) {
            CrmPeoplePayload peoplePayload = new CrmPeoplePayload();
            peoplePayload.setCustomerId(payload.getCustomerId());
            peoplePayload.setPeopleName(payload.getPeopleName());
            peoplePayload.setMobile(payload.getMobile());
            peoplePayload.setLocationDetail(payload.getLocationDetail());
            peoplePayload.setJobs(payload.getJobDetail());
            peoplePayload.setBu(payload.getCustOperBu());
            crmPeopleService.insertOrUpdate(peoplePayload);
        }
    }

    /**
     * 验证手机号
     *
     * @param phoneNumber
     * @return
     */
    public static boolean isValidPhoneNumber(String phoneNumber) {
        // 定义中国大陆手机号的正则表达式
        String regex = "^1[3456789]\\d{9}$";

        // 使用Pattern和Matcher进行匹配
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(phoneNumber);

        return matcher.matches();
    }

}
