package com.elitesland.tw.tw5.server.prd.crm.visit.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgEmployeeService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeRefVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemUserRoleVO;
import com.elitesland.tw.tw5.api.prd.visit.payload.VisitPlanPayload;
import com.elitesland.tw.tw5.api.prd.visit.query.VisitPlanQuery;
import com.elitesland.tw.tw5.api.prd.visit.vo.VisitPlanVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.crm.visit.constants.VisitTaskPlanStatusEnum;
import com.elitesland.tw.tw5.server.prd.crm.visit.entity.QVisitPlanDO;
import com.elitesland.tw.tw5.server.prd.crm.visit.entity.QVisitPlanDetailDO;
import com.elitesland.tw.tw5.server.prd.crm.visit.entity.VisitPlanDO;
import com.elitesland.tw.tw5.server.prd.crm.visit.repo.VisitPlanRepo;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBookAddressDO;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBusinessPartnerDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 销售拜访计划
 *
 * @author liwenpeng
 * @date 2023-03-14
 */
@Repository
@RequiredArgsConstructor
public class VisitPlanDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final VisitPlanRepo repo;
    private final QVisitPlanDO qdo = QVisitPlanDO.visitPlanDO;
  //  private final QCrmCustomerDO customerDO = QCrmCustomerDO.crmCustomerDO;
    private final QBusinessPartnerDO qBusinessPartnerDO = QBusinessPartnerDO.businessPartnerDO;
    private final QBookAddressDO qBookAddressDO = QBookAddressDO.bookAddressDO;
    private final CacheUtil cacheUtil;
    private final PrdOrgEmployeeService prdOrgEmployeeService;
    private final QVisitPlanDetailDO qVisitPlanDetailDO = QVisitPlanDetailDO.visitPlanDetailDO;
    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<VisitPlanVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(VisitPlanVO.class,
                        qdo.id,
                        qdo.remark,
                        qdo.createUserId,
                        qdo.creator,
                        qdo.createTime,
                        qdo.modifyUserId,
                        qdo.updater,
                        qdo.modifyTime,
                        // 计划类型
                        qdo.visitPlanType,
                        qdo.visitPlanStatus,
                        // 计划名称
                        qdo.visitPlanName,
                        // 拜访客户id
                        qdo.customId,
                        qdo.partnerId,
                        qBusinessPartnerDO.partnerName.as("customerName"),
                        qBusinessPartnerDO.bookId,
//                customerDO.customerName,
                        // 客户地址
                        qdo.customAdress,
                        qdo.longitudeLatitude,
                        // 业务合作伙伴-地址信息ID
                        qdo.bookAddressId,
                        qBookAddressDO.detailAddress.as("bookAddressDesc"),
                        // 拜访时间
                        qdo.visitDateFrom,
                        // 拜访时间
                        qdo.visitDateTo,
                        // 拜访次数
                        qdo.visitTimes,
                        // 添加提醒
                        qdo.remindTime,
                        qdo.extString1,
                        qdo.extString2,
                        qdo.extString3,
                        qdo.extString4,
                        qdo.extString5,
                        // 关联信息类型（线索、商机OPPO、经营计划、销售目标）
                        qdo.objType,
                        // 关联信息主键（关联主键)
                        qdo.objId,
                        // 关联信息名称（冗余）
                        qdo.objName
                )).from(qdo)
                .leftJoin(qBusinessPartnerDO).on(qBusinessPartnerDO.id.eq(qdo.partnerId))
                .leftJoin(qBookAddressDO).on(qBookAddressDO.id.eq(qdo.bookAddressId))
 //            .leftJoin(customerDO).on(customerDO.id.eq(qdo.customId))
                ;
    }


    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<VisitPlanVO> getJpaQueryWhere(VisitPlanQuery query) {
        JPAQuery<VisitPlanVO> jpaQuery = getJpaQuerySelect();
        if(query.getPermissionFlag()){
            jpaQuery.leftJoin(qVisitPlanDetailDO).on(qdo.id.eq(qVisitPlanDetailDO.visitPlanId));
        }
        jpaQuery.distinct();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(VisitPlanQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.countDistinct())
                .from(qdo)
                .leftJoin(qBusinessPartnerDO).on(qBusinessPartnerDO.id.eq(qdo.partnerId))
                .leftJoin(qBookAddressDO).on(qBookAddressDO.id.eq(qdo.bookAddressId));
        if(query.getPermissionFlag()){
            jpaQuery.leftJoin(qVisitPlanDetailDO).on(qdo.id.eq(qVisitPlanDetailDO.visitPlanId));
        }
        long total = jpaQuery.where(where(query)).fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(VisitPlanQuery query) {
        List<Predicate> list = new ArrayList<>();

        /** 计划类型 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitPlanType())) {
            list.add(qdo.visitPlanType.eq(query.getVisitPlanType()));
        }
        /** 计划状态 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitPlanStatus())) {
            list.add(qdo.visitPlanStatus.eq(query.getVisitPlanStatus()));
        }

        /** 计划名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getVisitPlanName())) {
            list.add(qdo.visitPlanName.like(SqlUtil.toSqlLikeString(query.getVisitPlanName())));
        }
        /** 拜访客户id 精确 */
        if (!ObjectUtils.isEmpty(query.getCustomId())) {
            list.add(qdo.customId.eq(query.getCustomId()));
        }
        if (!ObjectUtils.isEmpty(query.getPartnerId())) {
            list.add(qdo.partnerId.eq(query.getPartnerId()));
        }
        /** 客户地址 模糊 */
        if (!ObjectUtils.isEmpty(query.getCustomAdress())) {
            list.add(qdo.customAdress.like(SqlUtil.toSqlLikeString(query.getCustomAdress())));
        }
        /** 拜访次数 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitTimes())) {
            list.add(qdo.visitTimes.eq(query.getVisitTimes()));
        }
        /** 创建人 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            list.add(qdo.createUserId.eq(query.getCreateUserId()));
        }
        /** 拜访时间 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitDateFrom())) {
            list.add(qdo.visitDateFrom.goe(query.getVisitDateFrom()));
        }
        if (!ObjectUtils.isEmpty(query.getVisitDateTo())) {
            list.add(qdo.visitDateTo.loe(query.getVisitDateTo()));
        }
        /** 创建人 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            list.add(qdo.createUserId.eq(query.getCreateUserId()));
        }

        /** 拜访人 */
        if (!ObjectUtils.isEmpty(query.getVisitPersonNameStr())) {
            list.add(qdo.extString1.contains(query.getVisitPersonNameStr()));
        }

        if (!ObjectUtils.isEmpty(query.getAccompanyPersonNameStr())) {
            list.add(qdo.extString2.contains(query.getAccompanyPersonNameStr()));
        }

        /**
         * 当没有管理员等角色时
         * 1.销售资源角色 可以查询到本人 下级 下级的下级...作为销售资源角色自己创建的数据
         * 2.拜访人员/协防人员  本人 本人的下级 下级的下级...作为拜访人员/协防人员的数据
         */
        if (query.getPermissionFlag()) {
            Long userId = query.getLoginUserId();
            BooleanExpression booleanExpression = qdo.createUserId.eq(userId);
            booleanExpression = booleanExpression.or(qVisitPlanDetailDO.visitPersonId.in(userId).and(qVisitPlanDetailDO.deleteFlag.eq(0)));
            list.add(booleanExpression);
        }
        return ExpressionUtils.allOf(list);
    }

    //拜访计划权限
    private void dataPermissionFlag(List<Predicate> list) {
        Long userId = GlobalUtil.getLoginUserId();
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.PLAT_SALE_PIC.getCode(), RoleEnum.PLAT_ALL_PIC.getCode(), RoleEnum.SALE_VISIT_PIC.getCode()));
        if (rolePermission) {
            return;
        }
        BooleanExpression booleanExpression = qdo.deleteFlag.eq(0);
        //先查询下级
        List<PrdOrgEmployeeRefVO> prdOrgEmployeeRefVOS = prdOrgEmployeeService.queryLowChildListByKey(null, userId);
        if (!CollectionUtils.isEmpty(prdOrgEmployeeRefVOS)) {
            List<Long> userIdList = prdOrgEmployeeRefVOS.stream().filter(p -> p.getUserId() != null).map(s -> s.getUserId()).collect(Collectors.toList());
            Set userIdSet = new HashSet(userIdList);
            //把自己也加进去
            userIdSet.add(userId);
            // 查询自己和自己下级是销售资源角色的数据
            List<PrdSystemUserRoleVO> prdSystemUserRoleVOS = prdOrgEmployeeService.queryUserListRole(new ArrayList<>(userIdSet), Arrays.asList(RoleEnum.SALE_RES.getCode()));
            if (!CollectionUtils.isEmpty(prdSystemUserRoleVOS)) {
                List<Long> saleUserIds = prdSystemUserRoleVOS.stream().filter(p -> p.getUserId() != null).map(s -> s.getUserId()).collect(Collectors.toList());
                booleanExpression.or(qdo.createUserId.in(saleUserIds));
            }
            // 查询下级人员是拜访人员或者协防人员（因为有一张明细表）
            booleanExpression.or(qVisitPlanDetailDO.visitPersonId.in(userIdSet));
        } else {
            //
            if (cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SALE_RES.getCode()))) {
                booleanExpression.or(qdo.createUserId.in(userId));
            }
            booleanExpression.or(qVisitPlanDetailDO.visitPersonId.in(userId));
        }
        list.add(booleanExpression);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public VisitPlanVO queryByKey(Long id) {
        JPAQuery<VisitPlanVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    public int verifyVisitPlanName(String visitPlanName) {
        JPAQuery<VisitPlanVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.visitPlanName.eq(visitPlanName));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch().size();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<VisitPlanVO> queryListDynamic(VisitPlanQuery query) {
        JPAQuery<VisitPlanVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    public VisitPlanVO findByKey(Long key) {
        JPAQuery<VisitPlanVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQuerySelect.where(qdo.id.eq(key));
        jpaQuerySelect.where(qdo.deleteFlag.eq(0));
        return jpaQuerySelect.fetchFirst();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<VisitPlanVO> queryPaging(VisitPlanQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<VisitPlanVO> jpaQuery = getJpaQueryWhere(query);
        List<VisitPlanVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<VisitPlanVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public VisitPlanDO save(VisitPlanDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<VisitPlanDO> saveAll(List<VisitPlanDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    public long updateByKeyDynamic(VisitPlanPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 计划类型
        if (payload.getVisitPlanType() != null) {
            update.set(qdo.visitPlanType, payload.getVisitPlanType());
        }
        // 计划状态
        if (payload.getVisitPlanStatus() != null) {
            update.set(qdo.visitPlanStatus, payload.getVisitPlanStatus());
        }
        // 计划名称
        if (payload.getVisitPlanName() != null) {
            update.set(qdo.visitPlanName, payload.getVisitPlanName());
        }
        // 拜访客户id
        if (payload.getCustomId() != null) {
            update.set(qdo.customId, payload.getCustomId());
        }
        if (payload.getPartnerId() != null) {
            update.set(qdo.partnerId, payload.getPartnerId());
        }
        // 客户地址
        if (payload.getCustomAdress() != null) {
            update.set(qdo.customAdress, payload.getCustomAdress());
        }
        if (payload.getVisitDateFrom() != null) {
            update.set(qdo.visitDateFrom, payload.getVisitDateFrom());
        }
        if (payload.getVisitDateTo() != null) {
            update.set(qdo.visitDateTo, payload.getVisitDateTo());
        }

        // 拜访次数
        if (payload.getVisitTimes() != null) {
            update.set(qdo.visitTimes, payload.getVisitTimes());
        }

        // 提醒
        if (payload.getRemindTime() != null) {
            update.set(qdo.remindTime, payload.getRemindTime());
        }
        // 关联信息类型（线索、商机OPPO、经营计划、销售目标）
        if (payload.getObjType() != null) {
            update.set(qdo.objType, payload.getObjType());
        }
        // 关联信息主键（关联主键)
        if (payload.getObjId() != null) {
            update.set(qdo.objId, payload.getObjId());
        }
        // 关联信息名称（冗余）
        if (payload.getObjName() != null) {
            update.set(qdo.objName, payload.getObjName());
        }
        if (payload.getExtString1() != null) {
            update.set(qdo.extString1, payload.getExtString1());
        }
        if (payload.getExtString2() != null) {
            update.set(qdo.extString2, payload.getExtString2());
        }

        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            if (nullFields.contains("remindTime")) {
                update.setNull(qdo.remindTime);
            }
            // 计划类型
            if (nullFields.contains("visitPlanType")) {
                update.setNull(qdo.visitPlanType);
            }
            // 计划名称
            if (nullFields.contains("visitPlanName")) {
                update.setNull(qdo.visitPlanName);
            }
            // 拜访客户id
            if (nullFields.contains("customId")) {
                update.setNull(qdo.customId);
            }
            // 客户地址
            if (nullFields.contains("customAdress")) {
                update.setNull(qdo.customAdress);
            }
            // 拜访次数
            if (nullFields.contains("visitTimes")) {
                update.setNull(qdo.visitTimes);
            }
            // 拜访次数
            if (nullFields.contains("visitDateFrom")) {
                update.setNull(qdo.visitDateFrom);
            }
            // 拜访次数
            if (nullFields.contains("visitDateTo")) {
                update.setNull(qdo.visitDateTo);
            }
            // 关联信息类型（线索、商机OPPO、经营计划、销售目标）
            if (nullFields.contains("objType")) {
                update.setNull(qdo.objType);
            }
            // 关联信息主键（关联主键)
            if (nullFields.contains("objId")) {
                update.setNull(qdo.objId);
            }
            // 关联信息名称（冗余）
            if (nullFields.contains("objName")) {
                update.setNull(qdo.objName);
            }
            // 关联信息名称（冗余）
            if (nullFields.contains("extString1")) {
                update.setNull(qdo.extString1);
            }
            if (nullFields.contains("extString2")) {
                update.setNull(qdo.extString2);
            }
        }

        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    public long completeVisitPlan(Long visitPlanId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(visitPlanId));

        update.set(qdo.visitPlanStatus, VisitTaskPlanStatusEnum.completed.getCode());
        SqlUtil.updateCommonJpaQuery(update, qdo._super);

        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

