package com.elitesland.tw.tw5.server.prd.crm.visit.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.visit.payload.VisitSignRecordPayload;
import com.elitesland.tw.tw5.api.prd.visit.query.VisitSignRecordQuery;
import com.elitesland.tw.tw5.api.prd.visit.vo.VisitSignRecordVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;

import com.elitesland.tw.tw5.server.prd.crm.visit.entity.QVisitSignRecordDO;
import com.elitesland.tw.tw5.server.prd.crm.visit.entity.VisitSignRecordDO;
import com.elitesland.tw.tw5.server.prd.crm.visit.repo.VisitSignRecordRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 销售拜访任务执行情况签到记录
 *
 * @author duwh
 * @date 2023-03-13
 */
@Repository
@RequiredArgsConstructor
public class VisitSignRecordDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final VisitSignRecordRepo repo;
    private final QVisitSignRecordDO qdo = QVisitSignRecordDO.visitSignRecordDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<VisitSignRecordVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(VisitSignRecordVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 拜访计划主键
                qdo.planId,
                // 拜访任务主键crm_visit_task.id
                qdo.taskId,
                // 任务名称（冗余）crm_visit_task.task_name
                qdo.taskName,
                qdo.type,
                // 签到坐标（最新）
                qdo.locations,
                // 签到地点（最新）
                qdo.address,
                // 签到时间（最新）
                qdo.signTime,
                // 签到图片（最新）
                qdo.fileCodes,
                // 拓展字段1
                qdo.ext1,
                // 拓展字段2
                qdo.ext2,
                // 拓展字段3
                qdo.ext3
        )).from(qdo);
    }

    /**
    * 拼装查询条件
    *
    * @param query 查询参数
    * @return jpaQuery对象
    */
    private JPAQuery<VisitSignRecordVO> getJpaQueryWhere(VisitSignRecordQuery query) {
        JPAQuery<VisitSignRecordVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(VisitSignRecordQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
    * 查询条件封装
    *
    * @param query 条件
    * @return {@link Predicate}
    */
    private Predicate where(VisitSignRecordQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 拜访计划主键 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanId())) {
            list.add(qdo.planId.eq(query.getPlanId()));
        }
        /** 拜访任务主键crm_visit_task.id 精确 */
        if (!ObjectUtils.isEmpty(query.getTaskId())) {
            list.add(qdo.taskId.eq(query.getTaskId()));
        }
        /** 任务名称（冗余）crm_visit_task.task_name 模糊 */
        if (!ObjectUtils.isEmpty(query.getTaskName())) {
             list.add(qdo.taskName.like(SqlUtil.toSqlLikeString(query.getTaskName())));
        }
        if (!ObjectUtils.isEmpty(query.getType())) {
            list.add(qdo.type.eq(query.getType()));
        }
        /** 签到坐标（最新） 模糊 */
        if (!ObjectUtils.isEmpty(query.getLocations())) {
             list.add(qdo.locations.like(SqlUtil.toSqlLikeString(query.getLocations())));
        }
        /** 签到地点（最新） 模糊 */
        if (!ObjectUtils.isEmpty(query.getAddress())) {
             list.add(qdo.address.like(SqlUtil.toSqlLikeString(query.getAddress())));
        }
        /** 签到图片（最新） 模糊 */
        if (!ObjectUtils.isEmpty(query.getFileCodes())) {
             list.add(qdo.fileCodes.like(SqlUtil.toSqlLikeString(query.getFileCodes())));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public VisitSignRecordVO queryByKey(Long id) {
        JPAQuery<VisitSignRecordVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<VisitSignRecordVO> queryListDynamic(VisitSignRecordQuery query) {
        JPAQuery<VisitSignRecordVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<VisitSignRecordVO> queryPaging(VisitSignRecordQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<VisitSignRecordVO> jpaQuery = getJpaQueryWhere(query);
        List<VisitSignRecordVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<VisitSignRecordVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public VisitSignRecordDO save(VisitSignRecordDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<VisitSignRecordDO> saveAll(List<VisitSignRecordDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(VisitSignRecordPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
                // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
                // 拜访计划主键
        if (payload.getPlanId() != null) {
            update.set(qdo.planId, payload.getPlanId());
        }
                // 拜访任务主键crm_visit_task.id
        if (payload.getTaskId() != null) {
            update.set(qdo.taskId, payload.getTaskId());
        }
                // 任务名称（冗余）crm_visit_task.task_name
        if (payload.getTaskName() != null) {
            update.set(qdo.taskName, payload.getTaskName());
        }
                // 签到坐标（最新）
        if (payload.getLocations() != null) {
            update.set(qdo.locations, payload.getLocations());
        }
                // 签到地点（最新）
        if (payload.getAddress() != null) {
            update.set(qdo.address, payload.getAddress());
        }
                // 签到图片（最新）
        if (payload.getFileCodes() != null) {
            update.set(qdo.fileCodes, payload.getFileCodes());
        }
                // 拓展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
                // 拓展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
                // 拓展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 拜访计划主键
            if (nullFields.contains("planId")) {
                update.setNull(qdo.planId);
            }
            // 拜访任务主键crm_visit_task.id
            if (nullFields.contains("taskId")) {
                update.setNull(qdo.taskId);
            }
            // 任务名称（冗余）crm_visit_task.task_name
            if (nullFields.contains("taskName")) {
                update.setNull(qdo.taskName);
            }
            // 签到坐标（最新）
            if (nullFields.contains("locations")) {
                update.setNull(qdo.locations);
            }
            // 签到地点（最新）
            if (nullFields.contains("address")) {
                update.setNull(qdo.address);
            }
            // 签到图片（最新）
            if (nullFields.contains("fileCodes")) {
                update.setNull(qdo.fileCodes);
            }
            // 拓展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

