package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchasePaymentPlanPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.LatePaymentPlanQuery;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchaseAgreementPaymentQuery;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchasePaymentPlanQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.LatePaymentPlanVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchasePaymentPlanVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.my.entity.QResActivityDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.PurchasePaymentPlanDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchaseContractManagerDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchasePaymentDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchasePaymentPlanDO;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchasePaymentEnum;
import com.elitesland.tw.tw5.server.prd.purchase.repo.PurchasePaymentPlanRepo;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConReceivablePlanDO;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

/**
 * 采购合同管理
 *
 * @author lipengfei
 * @date 2023-05-30
 */
@Repository
@RequiredArgsConstructor
public class PurchasePaymentPlanDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PurchasePaymentPlanRepo repo;
    private final QPurchasePaymentPlanDO qdo = QPurchasePaymentPlanDO.purchasePaymentPlanDO;
    private final QResActivityDO qResActivityDO = QResActivityDO.resActivityDO;
    private final QConReceivablePlanDO qConReceivablePlanDO = QConReceivablePlanDO.conReceivablePlanDO;
    private final QPurchasePaymentDO qPurchasePaymentDO = QPurchasePaymentDO.purchasePaymentDO;
    private final QPurchaseContractManagerDO qPurchaseContractManagerDO = QPurchaseContractManagerDO.purchaseContractManagerDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PurchasePaymentPlanVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PurchasePaymentPlanVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 付款阶段
                qdo.paymentStage,
                // 付款金额
                qdo.paymentAmt,
                // 本次付款金额
                qdo.currentPaymentAmt,
                // 付款比例
                qdo.paymentProportion,
                // 预计付款日期
                qdo.estimatedPaymentDate,
                // 付款申请单id
                qdo.paymentApplyId,
                // 付款申请单号
                qPurchasePaymentDO.paymentNo.as("paymentApplyNo"),
                // 付款申请单状态
                qPurchasePaymentDO.state.as("payApplyStatus"),
                // 付款申请单类型
                qPurchasePaymentDO.paymentApplicationType.as("paymentApplicationType"),
                // 单据id
                qdo.contractId,
                // 单据编号
                qdo.contractNo,
                // 付款状态
                qdo.paymentStatus,
                // 核销状态
                qdo.writeoffStatus,
                // 约束里程碑
                qdo.milestone,
                // 约束里程碑名称
                qResActivityDO.actName.as("milestoneName"),
                // 单据类型
                qdo.docType,
                // 付款时间
                qdo.paymentDate,
                // 约束合同节点
                qdo.contractNode,
                // 约束合同节点 名称
                qConReceivablePlanDO.receStage.as("contractNodeName"),
                // 合同收款计划状态（合同节点状态）
                qConReceivablePlanDO.receStatus.as("receStatus"),
                // 当期应收款金额
                qConReceivablePlanDO.receAmt,
                // 已收款金额
                qConReceivablePlanDO.actualRecvAmt.as("actualReceAmt"),
                // 预付款中付款计划id
                qdo.prePaymentPlanId
        )).from(qdo)
                .leftJoin(qPurchasePaymentDO).on(qdo.paymentApplyId.eq(qPurchasePaymentDO.id)
                        .and(qPurchasePaymentDO.deleteFlag.eq(0)))
                .leftJoin(qConReceivablePlanDO).on(qdo.contractNode.eq(qConReceivablePlanDO.id)
                        .and(qConReceivablePlanDO.deleteFlag.eq(0)))
                .leftJoin(qResActivityDO).on(qdo.milestone.eq(qResActivityDO.id)
                        .and(qResActivityDO.deleteFlag.eq(0)))
                .where(qdo.deleteFlag.eq(0));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PurchasePaymentPlanVO> getJpaQueryWhere(PurchasePaymentPlanQuery query) {
        JPAQuery<PurchasePaymentPlanVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PurchasePaymentPlanQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PurchasePaymentPlanQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 付款阶段 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentStage())) {
            list.add(qdo.paymentStage.eq(query.getPaymentStage()));
        }
        /** 付款金额 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentAmt())) {
            list.add(qdo.paymentAmt.eq(query.getPaymentAmt()));
        }
        /** 本次付款金额 精确 */
        if (!ObjectUtils.isEmpty(query.getCurrentPaymentAmt())) {
            list.add(qdo.currentPaymentAmt.eq(query.getCurrentPaymentAmt()));
        }
        /** 付款比例 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentProportion())) {
            list.add(qdo.paymentProportion.eq(query.getPaymentProportion()));
        }
        /** 预计付款日期 精确 */
        if (!ObjectUtils.isEmpty(query.getEstimatedPaymentDate())) {
            list.add(qdo.estimatedPaymentDate.eq(query.getEstimatedPaymentDate()));
        }
        /** 付款申请单号 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentApplyId())) {
            list.add(qdo.paymentApplyId.eq(query.getPaymentApplyId()));
        }
        /** 单据id 精确 */
        if (!ObjectUtils.isEmpty(query.getContractId())) {
            list.add(qdo.contractId.eq(query.getContractId()));
        }
        /** 单据编号 精确 */
        if (!ObjectUtils.isEmpty(query.getContractNo())) {
            list.add(qdo.contractNo.eq(query.getContractNo()));
        }
        /** 付款状态 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentStatus())) {
            list.add(qdo.paymentStatus.eq(query.getPaymentStatus()));
        }
        /** 核销状态 精确 */
        if (!ObjectUtils.isEmpty(query.getWriteoffStatus())) {
            list.add(qdo.writeoffStatus.eq(query.getWriteoffStatus()));
        }
        /** 约束里程碑 精确 */
        if (!ObjectUtils.isEmpty(query.getMilestone())) {
            list.add(qdo.milestone.eq(query.getMilestone()));
        }
        /** 单据类型 精确 */
        if (!ObjectUtils.isEmpty(query.getDocType())) {
            list.add(qdo.docType.eq(query.getDocType()));
        }
        /** 付款时间 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentDate())) {
            list.add(qdo.paymentDate.eq(query.getPaymentDate()));
        }
        /** 约束合同节点 精确 */
        if (!ObjectUtils.isEmpty(query.getContractNode())) {
            list.add(qdo.contractNode.eq(query.getContractNode()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PurchasePaymentPlanVO queryByKey(Long id) {
        JPAQuery<PurchasePaymentPlanVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PurchasePaymentPlanVO> queryListDynamic(PurchasePaymentPlanQuery query) {
        JPAQuery<PurchasePaymentPlanVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PurchasePaymentPlanVO> queryPaging(PurchasePaymentPlanQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PurchasePaymentPlanVO> jpaQuery = getJpaQueryWhere(query);
        List<PurchasePaymentPlanVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PurchasePaymentPlanVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PurchasePaymentPlanDO save(PurchasePaymentPlanDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PurchasePaymentPlanDO> saveAll(List<PurchasePaymentPlanDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PurchasePaymentPlanPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 付款阶段
        if (payload.getPaymentStage() != null) {
            update.set(qdo.paymentStage, payload.getPaymentStage());
        }
        // 付款金额
        if (payload.getPaymentAmt() != null) {
            update.set(qdo.paymentAmt, payload.getPaymentAmt());
        }
        // 本次付款金额
        if (payload.getCurrentPaymentAmt() != null) {
            update.set(qdo.currentPaymentAmt, payload.getCurrentPaymentAmt());
        }
        // 付款比例
        if (payload.getPaymentProportion() != null) {
            update.set(qdo.paymentProportion, payload.getPaymentProportion());
        }
        // 预计付款日期
        if (payload.getEstimatedPaymentDate() != null) {
            update.set(qdo.estimatedPaymentDate, payload.getEstimatedPaymentDate());
        }
        // 付款申请单号
        if (payload.getPaymentApplyId() != null) {
            update.set(qdo.paymentApplyId, payload.getPaymentApplyId());
        }
        // 单据id
        if (payload.getContractId() != null) {
            update.set(qdo.contractId, payload.getContractId());
        }
        // 单据编号
        if (payload.getContractNo() != null) {
            update.set(qdo.contractNo, payload.getContractNo());
        }
        // 付款状态
        if (payload.getPaymentStatus() != null) {
            update.set(qdo.paymentStatus, payload.getPaymentStatus());
        }
        // 核销状态
        if (payload.getWriteoffStatus() != null) {
            update.set(qdo.writeoffStatus, payload.getWriteoffStatus());
        }
        // 约束里程碑
        if (payload.getMilestone() != null) {
            update.set(qdo.milestone, payload.getMilestone());
        }
        // 单据类型
        if (payload.getDocType() != null) {
            update.set(qdo.docType, payload.getDocType());
        }
        // 付款时间
        if (payload.getPaymentDate() != null) {
            update.set(qdo.paymentDate, payload.getPaymentDate());
        }
        // 约束合同节点
        if (payload.getContractNode() != null) {
            update.set(qdo.contractNode, payload.getContractNode());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 付款阶段
            if (nullFields.contains("paymentStage")) {
                update.setNull(qdo.paymentStage);
            }
            // 付款金额
            if (nullFields.contains("paymentAmt")) {
                update.setNull(qdo.paymentAmt);
            }
            // 本次付款金额
            if (nullFields.contains("currentPaymentAmt")) {
                update.setNull(qdo.currentPaymentAmt);
            }
            // 付款比例
            if (nullFields.contains("paymentProportion")) {
                update.setNull(qdo.paymentProportion);
            }
            // 预计付款日期
            if (nullFields.contains("estimatedPaymentDate")) {
                update.setNull(qdo.estimatedPaymentDate);
            }
            // 付款申请单号
            if (nullFields.contains("paymentApplyId")) {
                update.setNull(qdo.paymentApplyId);
            }
            // 单据id
            if (nullFields.contains("contractId")) {
                update.setNull(qdo.contractId);
            }
            // 单据编号
            if (nullFields.contains("contractNo")) {
                update.setNull(qdo.contractNo);
            }
            // 付款状态
            if (nullFields.contains("paymentStatus")) {
                update.setNull(qdo.paymentStatus);
            }
            // 核销状态
            if (nullFields.contains("writeoffStatus")) {
                update.setNull(qdo.writeoffStatus);
            }
            // 约束里程碑
            if (nullFields.contains("milestone")) {
                update.setNull(qdo.milestone);
            }
            // 单据类型
            if (nullFields.contains("docType")) {
                update.setNull(qdo.docType);
            }
            // 付款时间
            if (nullFields.contains("paymentDate")) {
                update.setNull(qdo.paymentDate);
            }
            // 约束合同节点
            if (nullFields.contains("contractNode")) {
                update.setNull(qdo.contractNode);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
    /**
     * 逻辑删除
     *
     * @param contractIdkeys 主集合
     * @return 删除的行数
     */
    public long logicalDeleteByContractIds(List<Long> contractIdkeys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.contractId.in(contractIdkeys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据合同编号查询付款计划
     * @param docNo 合同编号
     * @param applyIdIsNull 付款申请单Id是否为null
     * @return  付款计划
     */
    public List<PurchasePaymentPlanVO> findListByConNo(String docNo, Boolean applyIdIsNull) {
        JPAQuery<PurchasePaymentPlanVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PurchasePaymentPlanVO.class,
                qdo.id,
                // 付款阶段
                qdo.paymentStage,
                // 预计付款日期
                qdo.estimatedPaymentDate,
                // 本次付款金额
                qdo.currentPaymentAmt,
                // 付款金额
                qdo.paymentAmt,
                // 单据类型
                qdo.docType,
                // 单据编号
                qdo.contractNo,
                // 约束里程碑
                qdo.milestone,
                // 约束里程碑名称
                qResActivityDO.actName.as("milestoneName"),
                // 约束合同节点
                qdo.contractNode,
                // 合同约束节点名称
                qConReceivablePlanDO.receStage.as("contractNodeName")
        )).from(qdo)
                .leftJoin(qResActivityDO).on(qdo.milestone.eq(qResActivityDO.id)
                        .and(qResActivityDO.deleteFlag.eq(0)))
                .leftJoin(qConReceivablePlanDO).on(qdo.contractNode.eq(qConReceivablePlanDO.id)
                        .and(qConReceivablePlanDO.deleteFlag.eq(0)))
                .where(qdo.contractNo.eq(docNo));
        if (applyIdIsNull) {
            jpaQuery.where(qdo.paymentApplyId.isNull());
        }
        return jpaQuery.fetch();
    }

    /**
     * 根据付款申请单ID查询付款计划
     * @param paymentApplyId 付款申请单Id
     * @return  付款计划
     */
    public List<PurchasePaymentPlanVO> queryListByPaymentApplyId(Long paymentApplyId) {
        JPAQuery<PurchasePaymentPlanVO> jpaQuery = getJpaQuerySelect().where(qdo.paymentApplyId.eq(paymentApplyId));
        return jpaQuery.fetch();
    }

    /**
     * 将付款申请单Id置空
     * @param paymentApplyId 付款申请单Id
     */
    public void resettingPlanByPaymentId(Long paymentApplyId) {
        JPAUpdateClause resetting = jpaQueryFactory.update(qdo)
                .set(qdo.currentPaymentAmt, BigDecimal.ZERO)
                .setNull(qdo.paymentApplyId)
                .where(qdo.paymentApplyId.eq(paymentApplyId));
        resetting.execute();
    }

    /**
     * 关联付款申请单及付款计划
     * @param purchasePaymentPlanPayload 付款计划
     */
    public void updatePlan(PurchasePaymentPlanPayload purchasePaymentPlanPayload) {
        // 本次付款金额
        BigDecimal currentPayment = purchasePaymentPlanPayload.getCurrentPaymentAmt() == null ? BigDecimal.ZERO : purchasePaymentPlanPayload.getCurrentPaymentAmt();
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.currentPaymentAmt, currentPayment)
                .set(qdo.docType, purchasePaymentPlanPayload.getDocType())
                .set(qdo.paymentDate,purchasePaymentPlanPayload.getPaymentDate())
                .set(qdo.paymentApplyId, purchasePaymentPlanPayload.getPaymentApplyId())
                .where(qdo.id.eq(purchasePaymentPlanPayload.getId()));
        update.execute();
    }

    /**
     * 根据付款申请单Id删除付款计划
     * @param id 付款申请单Id
     */
    public void deleteByPaymentId(Long id) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.paymentApplyId.eq(id));
        update.execute();
    }

    /**
     * 逾期付款计划
     * @param query 查询条件
     * @return 结果
     */
    public PagingVO<LatePaymentPlanVO> findLatePaymentPlanList(LatePaymentPlanQuery query) {
        JPAQuery<LatePaymentPlanVO> jpaQuery = jpaQueryFactory.select(Projections.bean(LatePaymentPlanVO.class,
                qdo.id,
                // 采购合同ID
                qPurchaseContractManagerDO.id.as("purchaseConId"),
                // 采购合同编号
                qPurchaseContractManagerDO.purchaseContractNo.as("purchaseConNo"),
                // 采购合同名称
                qPurchaseContractManagerDO.purchaseContractName.as("purchaseConName"),
                // 供应商BookId
                qPurchaseContractManagerDO.supplierBookId,
                // 采购负责人
                qPurchaseContractManagerDO.purchaseInchargeResId,
                // 付款阶段
                qdo.paymentStage,
                // 本次付款金额
                qdo.currentPaymentAmt,
                // 付款比例
                qdo.paymentProportion,
                // 预计付款日期
                qdo.estimatedPaymentDate,
                // 备注
                qdo.remark,
                // 合同节点状态
                qConReceivablePlanDO.receStatus.as("receivableStatus"),
                // 已收款金额
                qConReceivablePlanDO.actualRecvAmt,
                // 约束里程碑
                qResActivityDO.actName.as("milestoneName")
        )).from(qdo);
        jpaQuery.leftJoin(qPurchaseContractManagerDO).on(qdo.contractId.eq(qPurchaseContractManagerDO.id)
                        .and(qPurchaseContractManagerDO.deleteFlag.eq(0)))
                .leftJoin(qConReceivablePlanDO).on(qdo.contractNode.eq(qConReceivablePlanDO.id)
                        .and(qConReceivablePlanDO.deleteFlag.eq(0)))
                .leftJoin(qResActivityDO).on(qdo.milestone.eq(qResActivityDO.id)
                        .and(qResActivityDO.deleteFlag.eq(0)))
                .where(qdo.paymentApplyId.isNull()
                        .and(qPurchaseContractManagerDO.id.isNotNull()));
        // 采购合同编号或名称
        if (!ObjectUtils.isEmpty(query.getPurchaseConNoOrName())) {
           jpaQuery.where(qPurchaseContractManagerDO.purchaseContractNo.like("%" + query.getPurchaseConNoOrName() + "%")
                   .or(qPurchaseContractManagerDO.purchaseContractName.like("%" + query.getPurchaseConNoOrName() + "%")));
        }
        // 供应商
        if (!ObjectUtils.isEmpty(query.getSupplierBookId())) {
            jpaQuery.where(qPurchaseContractManagerDO.supplierBookId.eq(query.getSupplierBookId()));
        }
        // 采购负责人
        if (!ObjectUtils.isEmpty(query.getPurchaseInchargeResId())) {
            jpaQuery.where(qPurchaseContractManagerDO.purchaseInchargeResId.eq(query.getPurchaseInchargeResId()));
        }
        // 预计付款日期区间
        if (!ObjectUtils.isEmpty(query.getEstimatedPaymentDateStart()) && !ObjectUtils.isEmpty(query.getEstimatedPaymentDateEnd())) {
            jpaQuery.where(qdo.estimatedPaymentDate.between(query.getEstimatedPaymentDateStart(), query.getEstimatedPaymentDateEnd()));
        }
        // 是否逾期（是：查询预计付款日期大于等于当前日期，否：查询预计付款日期大于等于当前日期/预计付款日期为空）
        if (query.getLateFlag()) {
            jpaQuery.where(qdo.estimatedPaymentDate.before(LocalDate.now()));
        }else {
            jpaQuery.where(qdo.estimatedPaymentDate.before(LocalDate.now())
                    .or(qdo.estimatedPaymentDate.isNull()));
        }
        List<LatePaymentPlanVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<LatePaymentPlanVO>builder().records(result).total(jpaQuery.fetchCount()).build();
    }

    /**
     * 根据采购协议编号查询付款明细
     * @param query 查询条件
     * @return 付款明细
     */
    public PagingVO<PurchasePaymentPlanVO> findPlanListByAgreementNo(PurchaseAgreementPaymentQuery query) {
        if (ObjectUtils.isEmpty(query.getAgreementNo())) {
            return null;
        }
        JPAQuery<PurchasePaymentPlanVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PurchasePaymentPlanVO.class,
            // 付款计划表Id
            qdo.id,
            // 付款阶段
            qdo.paymentStage,
            // 本次付款金额
            qdo.currentPaymentAmt,
            // 付款金额
            qdo.paymentAmt,
            // 预计付款日期
            qdo.estimatedPaymentDate,
            // 付款状态
            qdo.paymentStatus,
            // 付款时间
            qdo.paymentDate,
            // 付款申请单Id
            qdo.paymentApplyId,
            // 付款申请单号
            qPurchasePaymentDO.paymentNo.as("paymentApplyNo"),
            // 付款申请单类型
            qPurchasePaymentDO.paymentApplicationType,
            // 付款申请单状态
            qPurchasePaymentDO.state.as("payApplyStatus"),
            // 付款日期
            qPurchasePaymentDO.payDate
        )).from(qdo)
                .leftJoin(qPurchasePaymentDO).on(qdo.paymentApplyId.eq(qPurchasePaymentDO.id)
                        .and(qPurchasePaymentDO.deleteFlag.eq(0)))
                .where(qPurchasePaymentDO.docNo.eq(query.getAgreementNo()).and(qPurchasePaymentDO.docType.eq(PurchasePaymentEnum.PaymentDocType.AGREEMENT.getCode())));
        // 过滤掉预付款核销的
        if (query.getIsIgnoreWriteOff()) {
            jpaQuery.where(qPurchasePaymentDO.paymentApplicationType.ne(PurchasePaymentEnum.PaymentType.ADVANCE_PAY_WRITE_OFF.getCode())
                    .or(qPurchasePaymentDO.paymentApplicationType.isNull()));
        }
        List<PurchasePaymentPlanVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PurchasePaymentPlanVO>builder().records(result).total(jpaQuery.fetchCount()).build();
    }

    /**
     * 询查采购付款计划
     * @param purConId 采购合同Id
     * @param isIgnoreWriteOff 是否忽略掉预付款核销
     * @return 付款计划
     */
    public List<PurchasePaymentPlanVO> queryListByConId(Long purConId, Boolean isIgnoreWriteOff) {
        JPAQuery<PurchasePaymentPlanVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQuerySelect.where(qdo.contractId.eq(purConId)
                .and(qdo.contractNo.like("PCN" + "%")));
        // 过滤掉预付款核销的
        if (isIgnoreWriteOff) {
            jpaQuerySelect.where(qPurchasePaymentDO.paymentApplicationType.ne(PurchasePaymentEnum.PaymentType.ADVANCE_PAY_WRITE_OFF.getCode())
                    .or(qPurchasePaymentDO.paymentApplicationType.isNull()));
        }
        return jpaQuerySelect.fetch();
    }

    /**
     * 根据采购合同or采购协议编号查询预付款付款计划
     * @param conOrAgreementNo 采购合同or采购协议编号
     * @param prePaymentNo 预付款申请单编号
     * @return 付款计划
     */
    public List<PurchasePaymentPlanVO> findPlanListByConOrAgreementNoAndPrePaymentNo(String conOrAgreementNo, String prePaymentNo) {
        JPAQuery<PurchasePaymentPlanVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQuerySelect.where(qdo.contractNo.eq(conOrAgreementNo)
                .and(qPurchasePaymentDO.paymentApplicationType.eq(PurchasePaymentEnum.PaymentType.ADVANCE_PAY.getCode()))
                .and(qPurchasePaymentDO.paymentNo.eq(prePaymentNo)));
        return jpaQuerySelect.fetch();
    }

    /**
     * 根据付款计划关联预付款付款计划id查询付款计划
     * @param prePaymentPlanId 预付款关联付款计划id
     * @return 付款计划
     */
    public List<PurchasePaymentPlanVO> queryListByPrePaymentPlanId(List<Long> prePaymentPlanId) {
        JPAQuery<PurchasePaymentPlanVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQuerySelect.where(qdo.prePaymentPlanId.in(prePaymentPlanId));
        return jpaQuerySelect.fetch();
    }

    /**
     * 查询付款计划已核销金额
     * @param prePaymentPlanIds 预付款核销付款计划关联预付款付款计划id
     * @return result
     */
    public List<Tuple> findWriteOffAmtByPrePaymentPlanIdIn(List<Long> prePaymentPlanIds) {
        JPAQuery<Tuple> jpaQuery = jpaQueryFactory.select(
                qdo.prePaymentPlanId,
                qdo.currentPaymentAmt.sum()
        ).from(qdo)
                .where(qdo.deleteFlag.eq(0).and(qdo.prePaymentPlanId.in(prePaymentPlanIds)))
                .groupBy(qdo.prePaymentPlanId);
        return jpaQuery.fetch();
    }

    /**
     * 根据预付款申请单编号查询已核销金额
     * @param prePaymentNo 预付款申请单编号
     * @return 已核销金额
     */
    public BigDecimal findWriteOffAmtByPrePaymentNo(String prePaymentNo) {
        JPAQuery<BigDecimal> jpaQuery = jpaQueryFactory.select(
                qdo.currentPaymentAmt.sum()
        ).from(qdo)
                .leftJoin(qPurchasePaymentDO).on(qdo.paymentApplyId.eq(qPurchasePaymentDO.id))
                .where(qdo.deleteFlag.eq(0)
                        .and(qPurchasePaymentDO.deleteFlag.eq(0))
                        .and(qPurchasePaymentDO.state.eq(PurchasePaymentEnum.PaymentStatus.APPROVED.getCode()))
                        .and(qPurchasePaymentDO.prePaymentNo.eq(prePaymentNo))
                );
        return jpaQuery.fetchOne();
    }
}

