package com.elitesland.tw.tw5.server.prd.qixin.service;

import cn.hutool.json.JSONUtil;
import com.elitesland.tw.tw5.server.common.util.RedisUtils;
import com.elitesland.tw.tw5.server.log.constant.ApiRequestLogTypeEnum;
import com.elitesland.tw.tw5.server.log.service.ApiRequestLogService;
import com.elitesland.tw.tw5.server.prd.qixin.config.QiXinProperties;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMethod;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * 启信宝 服务实现
 *
 * @author duwh
 * @date 2022/11/22
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class QiXinServiceImpl implements QiXinService {

    private final ApiRequestLogService apiRequestLogService;
    private final QiXinProperties qiXinProperties;
    private final RedisUtils redisUtils;

    private final static String QXB_CACHE_KEY = "QXB_CACHE_KEY";


    /**
     * 搜索 1.31 模糊搜索
     *
     * @param matchType 匹配类型（可选），keyword 需要匹配的 字段，可以输入以下一个或者几个类型， 多个类型使用|分隔： partner：股东 oper：法人 4 member：高管 contact：联系方式 scope：经营范围 name：公司名称 patent：专利 copyright：著作权作品名称 software：软件著作权名称 trademark：商标 domain：网址 product：产品
     * @param region    地区编码（可选），输入参数为空返回全 部区域匹配记录。可传入 省/直辖市，城市，区县的国家区划码。 只支持传入一个区划码。 省/直辖市：传入国家区划码前两位数字， 如搜索“江苏省”则传入 32，各省/直辖 市编码可参考本文档附录 城市：传入国家区划码前四位数字，如搜 索“苏州市”，则传入 3205 区县：传入国家区划码六位数字，如搜索 “苏州市吴中区”，则传入 320506 可以通过下面网页查询国家区划编码 http://www.mca.gov.cn/article/sj/tj bz/a/2017/2018
     * @param keyword   关键字（必填） 企业全名/注册号/统一社会信用代码
     * @param skip      跳过(可选) 跳过条目数（默认为0，单页返回10条数据）
     * @return {@link String}
     */
    @Override
    public String search(String matchType, String region, String keyword, String skip) {
        String uriSearch = qiXinProperties.getUriSearch();
        // 设置url参数
        Map<String, String> urlParams = new HashMap<String, String>();
        if (StringUtils.hasText(matchType)) {
            urlParams.put("matchType", matchType);
        }
        if (StringUtils.hasText(region)) {
            urlParams.put("region", region);
        }
        if (StringUtils.hasText(keyword)) {
            urlParams.put("keyword", keyword);
        }
        if (StringUtils.hasText(skip)) {
            urlParams.put("skip", skip);
        }
        List<String> concatParams = new ArrayList<String>();
        for (String s : urlParams.keySet()) {
            concatParams.add(s + "=" + URLEncoder.encode(urlParams.get(s), StandardCharsets.UTF_8));
        }

        uriSearch = uriSearch + "?" + String.join("&", concatParams);
        final String result = getMessage(uriSearch);
        return result;
    }

    /**
     * 1.2 高级搜索
     *
     * @param keyword       企业全名/注册号/统一社会信用代码
     * @param skip          跳过条目数（默认为0，单页返回20条数据）
     * @param method        搜索范围，多个使用逗号分隔，具体值参 考附录
     * @param org_type      组织类型，多个使用逗号分隔，具体值参 考附录
     * @param area_code     地区码
     * @param industry_code 行业码
     * @param econ_type     企业类型，多个使用逗号分隔，具体值参 考附录
     * @param capi_from     注册资本起
     * @param capi_to       注册资本止
     * @param date_from     成立日期起
     * @param date_to       成立日期止
     * @param status        企业状态，多个使用逗号分隔，具体值参 考附录
     * @param canbao_from   参保人数起
     * @param canbao_to     参保人数止
     * @return {@link String}
     */
    @Override
    public String advanceSearchNew(String keyword, String skip, String method, String org_type, String area_code, String industry_code,
                                   String econ_type, String capi_from, String capi_to, String date_from, String date_to,
                                   String status, String canbao_from, String canbao_to) {
        String uri = qiXinProperties.getUriAdvanceSearchNew();
        // 设置url参数
        Map<String, String> urlParams = new HashMap<String, String>();
        if (StringUtils.hasText(keyword)) {
            urlParams.put("keyword", keyword);
        }
        if (StringUtils.hasText(skip)) {
            urlParams.put("skip", skip);
        }
        if (StringUtils.hasText(method)) {
            urlParams.put("method", method);
        }
        if (StringUtils.hasText(org_type)) {
            urlParams.put("org_type", org_type);
        }
        if (StringUtils.hasText(area_code)) {
            urlParams.put("area_code", area_code);
        }
        if (StringUtils.hasText(industry_code)) {
            urlParams.put("industry_code", industry_code);
        }
        if (StringUtils.hasText(econ_type)) {
            urlParams.put("econ_type", econ_type);
        }
        if (StringUtils.hasText(capi_from)) {
            urlParams.put("capi_from", capi_from);
        }
        if (StringUtils.hasText(capi_to)) {
            urlParams.put("capi_to", capi_to);
        }
        if (StringUtils.hasText(date_from)) {
            urlParams.put("date_from", date_from);
        }
        if (StringUtils.hasText(date_to)) {
            urlParams.put("date_to", date_to);
        }
        if (StringUtils.hasText(status)) {
            urlParams.put("status", status);
        }
        if (StringUtils.hasText(canbao_from)) {
            urlParams.put("canbao_from", canbao_from);
        }
        if (StringUtils.hasText(canbao_to)) {
            urlParams.put("canbao_to", canbao_to);
        }
        List<String> concatParams = new ArrayList<String>();
        for (String s : urlParams.keySet()) {
            concatParams.add(s + "=" + URLEncoder.encode(urlParams.get(s), StandardCharsets.UTF_8));
        }

        uri = uri + "?" + String.join("&", concatParams);
        final String result = getMessage(uri);
        return result;
    }

    /**
     * 获取企业联系方式 1.51 企业联系方式
     *
     * @param keyword 关键字 企业全名/注册号/统一社会信用代码
     * @return {@link String}
     */
    @Override
    public String getContactInfo(String keyword) {
        String uri = qiXinProperties.getUriGetContactInfo();
        // 设置url参数
        Map<String, String> urlParams = new HashMap<String, String>();
        if (StringUtils.hasText(keyword)) {
            urlParams.put("keyword", keyword);
        }
        List<String> concatParams = new ArrayList<String>();
        for (String s : urlParams.keySet()) {
            concatParams.add(s + "=" + URLEncoder.encode(urlParams.get(s), StandardCharsets.UTF_8));
        }
        uri = uri + "?" + String.join("&", concatParams);
        final String result = getMessage(uri);
        return result;
    }

    /**
     * 10.1 新闻列表
     * 企业新闻列表，包括新闻来源、情感属性、舆情标签、负面指数等
     * 申请接口
     * 单价：0.1元/次
     *
     * @param name              必填 企业全名/注册号/统一社会信用代码
     * @param skip              跳过条目数（默认为 0）
     * @param tag_list          舆情分类，参考数据字典tag_list字段，示例：["131","132"]
     * @param sentiment         情感属性，参考数据字典sentiment字段，示例：["neg","pos"]
     * @param company_sentiment 企业的情感属性，参考数据字典sentiment字段，示例：["neg","pos"]
     * @param class_types       新闻类别，全部可不传，参考数据字典class_types字段，示例：["news"]
     * @param new_tag_list      事件分类，参考数据字典new_tag_list字段，示例：["101","1024"]
     * @param company_new_tag   企业的事件分类，参考数据字典new_tag_list字段，示例：["0102004","0102005"]
     * @param create_time_range 舆情发布时间范围，示例：["2019-01-01","2019-08-14"]
     * @return {@link String}
     */
    @Override
    public String getNewsListByName(String name, String skip, String tag_list, String sentiment,
                                    String company_sentiment, String class_types, String new_tag_list,
                                    String company_new_tag, String create_time_range) {
        String uri = qiXinProperties.getUriGetNewsListByName();
        // 设置url参数
        Map<String, String> urlParams = new HashMap<String, String>();
        if (StringUtils.hasText(name)) {
            urlParams.put("name", name);
        }
        if (StringUtils.hasText(skip)) {
            urlParams.put("skip", skip);
        }
        if (StringUtils.hasText(tag_list)) {
            urlParams.put("tag_list", tag_list);
        }
        if (StringUtils.hasText(sentiment)) {
            urlParams.put("sentiment", sentiment);
        }
        if (StringUtils.hasText(company_sentiment)) {
            urlParams.put("company_sentiment", company_sentiment);
        }
        if (StringUtils.hasText(class_types)) {
            urlParams.put("class_types", class_types);
        }
        if (StringUtils.hasText(new_tag_list)) {
            urlParams.put("new_tag_list", new_tag_list);
        }
        if (StringUtils.hasText(company_new_tag)) {
            urlParams.put("company_new_tag", company_new_tag);
        }
        if (StringUtils.hasText(create_time_range)) {
            //final String[] split = create_time_range.split(",");
            //urlParams.put("create_time_range", split.toString());
            urlParams.put("create_time_range", create_time_range);
        }
        List<String> concatParams = new ArrayList<String>();
        for (String s : urlParams.keySet()) {
            concatParams.add(s + "=" + URLEncoder.encode(urlParams.get(s), StandardCharsets.UTF_8));
        }
        uri = uri + "?" + String.join("&", concatParams);
        final String result = getMessage(uri);
        return result;
    }

    /**
     * 1.41 工商照面
     * 企业工商照面及相关信息，包括统一社会信用代码、注册资本、经营范围、企业法定代表人等
     *
     * @param keyword 企业全名/注册号/统一社会信用代码
     * @return
     */
    @Override
    public String getBasicInfo(String keyword) {
        String uri = qiXinProperties.getGetBasicInfoUri();
        // 设置url参数
        Map<String, String> urlParams = new HashMap<String, String>();
        if (StringUtils.hasText(keyword)) {
            urlParams.put("keyword", keyword);
        }
        List<String> concatParams = new ArrayList<String>();
        for (String s : urlParams.keySet()) {
            concatParams.add(s + "=" + URLEncoder.encode(urlParams.get(s), StandardCharsets.UTF_8));
        }
        uri = uri + "?" + String.join("&", concatParams);
        final String result = getMessage(uri);
        return result;
    }


    /**
     * 1.17 企业LOGO
     *
     * @param name 名字
     * @return {@link String}
     */
    @Override
    public String getEntLogoByName(String name) {
        String uri = qiXinProperties.getGetEntLogoByNameUri();
        // 设置url参数
        Map<String, String> urlParams = new HashMap<String, String>();
        if (StringUtils.hasText(name)) {
            urlParams.put("name", name);
        }
        List<String> concatParams = new ArrayList<String>();
        for (String s : urlParams.keySet()) {
            concatParams.add(s + "=" + URLEncoder.encode(urlParams.get(s), StandardCharsets.UTF_8));
        }
        uri = uri + "?" + String.join("&", concatParams);
        final String result = getMessage(uri);
        return result;
    }

    /**
     * 1.16 企业简介
     *
     * @param name 名字
     * @return {@link String}
     */
    @Override
    public String getEntBriefByName(String name) {
        String uri = qiXinProperties.getGetEntBriefByNameUri();
        // 设置url参数
        Map<String, String> urlParams = new HashMap<String, String>();
        if (StringUtils.hasText(name)) {
            urlParams.put("name", name);
        }
        List<String> concatParams = new ArrayList<String>();
        for (String s : urlParams.keySet()) {
            concatParams.add(s + "=" + URLEncoder.encode(urlParams.get(s), StandardCharsets.UTF_8));
        }
        uri = uri + "?" + String.join("&", concatParams);
        final String result = getMessage(uri);
        return result;
    }

    /**
     * 1.53 企业年报网址
     * 企业年报网址信息，包括网站名称、网址、网址类型、网址来源、审核时间等
     * <p>
     * 单价：0.1元/次
     *
     * @param keyword 企业全名/注册号/统一社会信用代码
     * @param skip    跳过条目数（默认为0，单页返回20条数据）
     * @return {@link String}
     */
    @Override
    public String getGetWebsites(String keyword, String skip) {
        String uri = qiXinProperties.getGetWebsitesUri();
        // 设置url参数
        Map<String, String> urlParams = new HashMap<String, String>();
        if (StringUtils.hasText(keyword)) {
            urlParams.put("keyword", keyword);
        }
        if (StringUtils.hasText(skip)) {
            urlParams.put("skip", skip);
        }
        List<String> concatParams = new ArrayList<String>();
        for (String s : urlParams.keySet()) {
            concatParams.add(s + "=" + URLEncoder.encode(urlParams.get(s), StandardCharsets.UTF_8));
        }
        uri = uri + "?" + String.join("&", concatParams);
        final String result = getMessage(uri);
        return result;
    }

    /**
     * 1.43 工商股东
     * 企业工商股东信息(不包含企业自行公示和年报股东)
     * <p>
     * 单价：0.1元/次
     *
     * @param keyword 企业全名/注册号/统一社会信用代码
     * @param skip    跳过条目数（默认为0，单页返回20条数据）
     * @return {@link String}
     */
    @Override
    public String getPartners(String keyword, String skip) {
        String uri = qiXinProperties.getGetPartnersUri();
        // 设置url参数
        Map<String, String> urlParams = new HashMap<String, String>();
        if (StringUtils.hasText(keyword)) {
            urlParams.put("keyword", keyword);
        }
        if (StringUtils.hasText(skip)) {
            urlParams.put("skip", skip);
        }
        List<String> concatParams = new ArrayList<String>();
        for (String s : urlParams.keySet()) {
            concatParams.add(s + "=" + URLEncoder.encode(urlParams.get(s), StandardCharsets.UTF_8));
        }
        uri = uri + "?" + String.join("&", concatParams);
        final String result = getMessage(uri);
        return result;
    }

    /**
     * 1.45 主要人员
     * 企业工商主要人员信息，包括主要人员姓名及职位
     * <p>
     * 单价：0.1元/次
     *
     * @param keyword 企业全名/注册号/统一社会信用代码
     * @param skip    跳过条目数（默认为0，单页返回20条数据）
     * @return {@link String}
     */
    @Override
    public String getGetEmployees(String keyword, String skip) {
        String uri = qiXinProperties.getGetEmployeesUri();
        // 设置url参数
        Map<String, String> urlParams = new HashMap<String, String>();
        if (StringUtils.hasText(keyword)) {
            urlParams.put("keyword", keyword);
        }
        if (StringUtils.hasText(skip)) {
            urlParams.put("skip", skip);
        }
        List<String> concatParams = new ArrayList<String>();
        for (String s : urlParams.keySet()) {
            concatParams.add(s + "=" + URLEncoder.encode(urlParams.get(s), StandardCharsets.UTF_8));
        }
        uri = uri + "?" + String.join("&", concatParams);
        final String result = getMessage(uri);
        return result;
    }

    /**
     * 发起请求
     *
     * @param path 路径
     * @return {@link String}
     */
    @Override
    public String getMessage(String path) {
        String result = "";
        if (!qiXinProperties.getEnabled()) {
            log.warn("启信宝未启用！");
            return "{\n" +
                    "  \"data\": null,\n" +
                    "  \"sign\": \"1234\",\n" +
                    "  \"status\": \"400\",\n" +
                    "  \"message\": \"未启用\"\n" +
                    "}";
        }
        String status = "";
        String appkey = qiXinProperties.getAppkey();
        String secretKey = qiXinProperties.getSecretKey();
        Calendar calendar = Calendar.getInstance();
        Long timestamp = calendar.getTime().getTime();
        String sign = getMD5Str(appkey + timestamp + secretKey);
        long start = System.currentTimeMillis();
        // 请求头
        Map<String, String> headers = new HashMap<String, String>();
        headers.put("Auth-version", "2.0");
        headers.put("appkey", "appkey");
        headers.put("timestamp", timestamp + "");
        headers.put("sign", sign);
        headers.put("Content-Type", "application/json");
        try {

            String key = QXB_CACHE_KEY + ":" + path;
            final String resultCache = (String) redisUtils.get(key);
            if (StringUtils.hasText(resultCache)) {
                return resultCache;
            }

            HttpGet request = new HttpGet(path);
            request.setHeader("Auth-version", "2.0");
            request.setHeader("appkey", appkey);
            request.setHeader("timestamp", timestamp + "");
            request.setHeader("sign", sign);
            request.setHeader("Content-Type", "application/json");
            HttpClient httpClient = HttpClientBuilder.create().build();
            org.apache.http.HttpResponse response = httpClient.execute(request);
            if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                result = EntityUtils.toString(response.getEntity(), "utf-8");
                redisUtils.set(key, result, qiXinProperties.getCacheDay(), TimeUnit.DAYS);
            } else {
                log.error("调用启信宝接口异常，path:{}; result:{};", path, result);
            }
            status = response.getStatusLine().getStatusCode() + "";
        } catch (Exception e) {
            result = e.getMessage();
            log.error("调用启信宝接口异常，path:{}; ", path, e);
        }
        long time = System.currentTimeMillis() - start;
        apiRequestLogService.saveOutLog(ApiRequestLogTypeEnum.QXB, path, RequestMethod.GET, null, JSONUtil.toJsonStr(headers), result, status, time);
        return result;
    }


    /**
     * 获取字符串的md5值
     *
     * @param str str
     * @return {@link String}
     */
    public static String getMD5Str(String str) {
        byte[] digest = null;
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(str.getBytes(StandardCharsets.UTF_8));
            digest = md.digest();

            StringBuilder sb = new StringBuilder();
            for (byte b : digest) {
                String s = String.format("%02x", b);
                sb.append(s);
            }
            return sb.toString();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

}
