package com.elitesland.tw.tw5.server.prd.acc.dao;


import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimSettingTitlePayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimSettingTitleQuery;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimSettingTitleVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccReimSettingTitleDO;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccReimSettingTitleDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimSettingTitleRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 报销抬头校验
 *
 * @author zoey
 * @date 2024-05-14
 */
@Repository
@RequiredArgsConstructor
public class AccReimSettingTitleDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AccReimSettingTitleRepo repo;
    private final QAccReimSettingTitleDO qdo = QAccReimSettingTitleDO.accReimSettingTitleDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimSettingTitleVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AccReimSettingTitleVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 报销配置主表id
                qdo.reimSettingId,
                // 发票类型OM:INPUT_INV_TYPE
                qdo.invType,
                // 校验内容（udc）
                qdo.checkContent,
                // 不参与校验发票
                qdo.exceptCommodityName
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimSettingTitleVO> getJpaQueryWhere(AccReimSettingTitleQuery query) {
        JPAQuery<AccReimSettingTitleVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AccReimSettingTitleQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AccReimSettingTitleQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 发票类型OM:INPUT_INV_TYPE 精确 */
        if (!ObjectUtils.isEmpty(query.getInvType())) {
            list.add(qdo.invType.eq(query.getInvType()));
        }
        /** 校验内容（udc） 精确 */
        if (!ObjectUtils.isEmpty(query.getCheckContent())) {
            list.add(qdo.checkContent.eq(query.getCheckContent()));
        }
        /** 不参与校验发票 精确 */
        if (!ObjectUtils.isEmpty(query.getExceptCommodityName())) {
            list.add(qdo.exceptCommodityName.eq(query.getExceptCommodityName()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AccReimSettingTitleVO queryByKey(Long id) {
        JPAQuery<AccReimSettingTitleVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimSettingTitleVO> queryListDynamic(AccReimSettingTitleQuery query) {
        JPAQuery<AccReimSettingTitleVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimSettingTitleVO> queryPaging(AccReimSettingTitleQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<AccReimSettingTitleVO> jpaQuery = getJpaQueryWhere(query);
        List<AccReimSettingTitleVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimSettingTitleVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AccReimSettingTitleDO save(AccReimSettingTitleDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AccReimSettingTitleDO> saveAll(List<AccReimSettingTitleDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AccReimSettingTitlePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 发票类型OM:INPUT_INV_TYPE
        if (payload.getInvType() != null) {
            update.set(qdo.invType, payload.getInvType());
        }
        // 校验内容（udc）
        if (payload.getCheckContent() != null) {
            update.set(qdo.checkContent, payload.getCheckContent());
        }
        // 不参与校验发票
        if (payload.getExceptCommodityName() != null) {
            update.set(qdo.exceptCommodityName, payload.getExceptCommodityName());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 报销配置主表id
            if (nullFields.contains("reimSettingId")) {
                update.setNull(qdo.reimSettingId);
            }
            // 发票类型OM:INPUT_INV_TYPE
            if (nullFields.contains("invType")) {
                update.setNull(qdo.invType);
            }
            // 校验内容（udc）
            if (nullFields.contains("professionalGrade")) {
                update.setNull(qdo.checkContent);
            }
            // 不参与校验发票
            if (nullFields.contains("exceptCommodityName")) {
                update.setNull(qdo.exceptCommodityName);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

