package com.elitesland.tw.tw5.server.prd.budget.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.budget.payload.BudgetAppropriationPayload;
import com.elitesland.tw.tw5.api.prd.budget.query.BudgetAppropriationQuery;
import com.elitesland.tw.tw5.api.prd.budget.vo.BudgetAppropriationVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.budget.entity.BudgetAppropriationDO;
import com.elitesland.tw.tw5.server.prd.budget.entity.QBudgetAppropriationDO;
import com.elitesland.tw.tw5.server.prd.budget.repo.BudgetAppropriationRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 预算拨付基本信息
 *
 * @author xxb
 * @date 2023-11-01
 */
@Repository
@RequiredArgsConstructor
public class BudgetAppropriationDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BudgetAppropriationRepo repo;
    private final QBudgetAppropriationDO qdo = QBudgetAppropriationDO.budgetAppropriationDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BudgetAppropriationVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BudgetAppropriationVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 预算ID
                qdo.budgetId,
                // 拨付单号
                qdo.appropriationNo,
                // 拨付状态
                qdo.appropriationStatus,
                // 申请人资源ID
                qdo.resId,
                // 申请费用金额
                qdo.applyFeeAmt,
                // 提现当量数
                qdo.applyEqva,
                // 提现当量金额
                qdo.applyEqvaAmt,
                // 申请总金额
                qdo.applyAmt,
                // 申请日期
                qdo.applyDate,
                // 流程实例ID
                qdo.procInstId,
                // 流程审批状态
                qdo.procInstStatus,
                // 审批时间
                qdo.approvedTime,
                // 拨款名称
                qdo.appropriationName,
                // 拨款日期
                qdo.appropriationDate,
                // 拨付附件
                qdo.appropriationFiles,
                // 流程定义的KEY
                qdo.procDefKey,
                // 拨付类型：科目、活动
                qdo.appropriationType,
                // 拨付当量
                qdo.allocateEqva,
                // 拨付费用
                qdo.allocateCost
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BudgetAppropriationVO> getJpaQueryWhere(BudgetAppropriationQuery query) {
        JPAQuery<BudgetAppropriationVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BudgetAppropriationQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BudgetAppropriationQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 预算ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBudgetId())) {
            list.add(qdo.budgetId.eq(query.getBudgetId()));
        }
        /** 拨付单号 精确 */
        if (!ObjectUtils.isEmpty(query.getAppropriationNo())) {
            list.add(qdo.appropriationNo.eq(query.getAppropriationNo()));
        }
        /** 拨付状态 精确 */
        if (!ObjectUtils.isEmpty(query.getAppropriationStatus())) {
            list.add(qdo.appropriationStatus.eq(query.getAppropriationStatus()));
        }
        /** 申请人资源ID 精确 */
        if (!ObjectUtils.isEmpty(query.getResId())) {
            list.add(qdo.resId.eq(query.getResId()));
        }
        /** 申请费用金额 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyFeeAmt())) {
            list.add(qdo.applyFeeAmt.eq(query.getApplyFeeAmt()));
        }
        /** 提现当量数 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyEqva())) {
            list.add(qdo.applyEqva.eq(query.getApplyEqva()));
        }
        /** 提现当量金额 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyEqvaAmt())) {
            list.add(qdo.applyEqvaAmt.eq(query.getApplyEqvaAmt()));
        }
        /** 申请总金额 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyAmt())) {
            list.add(qdo.applyAmt.eq(query.getApplyAmt()));
        }
        /** 申请日期 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyDate())) {
            list.add(qdo.applyDate.eq(query.getApplyDate()));
        }
        /** 流程实例ID 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstId())) {
            list.add(qdo.procInstId.eq(query.getProcInstId()));
        }
        /** 流程审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstStatus())) {
            list.add(qdo.procInstStatus.eq(query.getProcInstStatus()));
        }
        /** 审批时间 精确 */
        if (!ObjectUtils.isEmpty(query.getApprovedTime())) {
            list.add(qdo.approvedTime.eq(query.getApprovedTime()));
        }
        /** 拨款名称 精确 */
        if (!ObjectUtils.isEmpty(query.getAppropriationName())) {
            list.add(qdo.appropriationName.eq(query.getAppropriationName()));
        }
        /** 拨款日期 精确 */
        if (!ObjectUtils.isEmpty(query.getAppropriationDate())) {
            list.add(qdo.appropriationDate.eq(query.getAppropriationDate()));
        }
        /** 拨付附件 精确 */
        if (!ObjectUtils.isEmpty(query.getAppropriationFiles())) {
            list.add(qdo.appropriationFiles.eq(query.getAppropriationFiles()));
        }
        /** 流程定义的KEY 精确 */
        if (!ObjectUtils.isEmpty(query.getProcDefKey())) {
            list.add(qdo.procDefKey.eq(query.getProcDefKey()));
        }
        /** 拨付类型：科目、活动 精确 */
        if (!ObjectUtils.isEmpty(query.getAppropriationType())) {
            list.add(qdo.appropriationType.eq(query.getAppropriationType()));
        }
        /** 拨付当量 精确 */
        if (!ObjectUtils.isEmpty(query.getAllocateEqva())) {
            list.add(qdo.allocateEqva.eq(query.getAllocateEqva()));
        }
        /** 拨付费用 精确 */
        if (!ObjectUtils.isEmpty(query.getAllocateCost())) {
            list.add(qdo.allocateCost.eq(query.getAllocateCost()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BudgetAppropriationVO queryByKey(Long id) {
        JPAQuery<BudgetAppropriationVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BudgetAppropriationVO> queryListDynamic(BudgetAppropriationQuery query) {
        JPAQuery<BudgetAppropriationVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BudgetAppropriationVO> queryPaging(BudgetAppropriationQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<BudgetAppropriationVO> jpaQuery = getJpaQueryWhere(query);
        List<BudgetAppropriationVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BudgetAppropriationVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BudgetAppropriationDO save(BudgetAppropriationDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BudgetAppropriationDO> saveAll(List<BudgetAppropriationDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BudgetAppropriationPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 预算ID
        if (payload.getBudgetId() != null) {
            update.set(qdo.budgetId, payload.getBudgetId());
        }
        // 拨付单号
        if (payload.getAppropriationNo() != null) {
            update.set(qdo.appropriationNo, payload.getAppropriationNo());
        }
        // 拨付状态
        if (payload.getAppropriationStatus() != null) {
            update.set(qdo.appropriationStatus, payload.getAppropriationStatus());
        }
        // 申请人资源ID
        if (payload.getResId() != null) {
            update.set(qdo.resId, payload.getResId());
        }
        // 申请费用金额
        if (payload.getApplyFeeAmt() != null) {
            update.set(qdo.applyFeeAmt, payload.getApplyFeeAmt());
        }
        // 提现当量数
        if (payload.getApplyEqva() != null) {
            update.set(qdo.applyEqva, payload.getApplyEqva());
        }
        // 提现当量金额
        if (payload.getApplyEqvaAmt() != null) {
            update.set(qdo.applyEqvaAmt, payload.getApplyEqvaAmt());
        }
        // 申请总金额
        if (payload.getApplyAmt() != null) {
            update.set(qdo.applyAmt, payload.getApplyAmt());
        }
        // 申请日期
        if (payload.getApplyDate() != null) {
            update.set(qdo.applyDate, payload.getApplyDate());
        }
        // 流程实例ID
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 流程审批状态
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        // 审批时间
        if (payload.getApprovedTime() != null) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        // 拨款名称
        if (payload.getAppropriationName() != null) {
            update.set(qdo.appropriationName, payload.getAppropriationName());
        }
        // 拨款日期
        if (payload.getAppropriationDate() != null) {
            update.set(qdo.appropriationDate, payload.getAppropriationDate());
        }
        // 拨付附件
        if (payload.getAppropriationFiles() != null) {
            update.set(qdo.appropriationFiles, payload.getAppropriationFiles());
        }
        // 流程定义的KEY
        if (payload.getProcDefKey() != null) {
            update.set(qdo.procDefKey, payload.getProcDefKey());
        }
        // 拨付类型：科目、活动
        if (payload.getAppropriationType() != null) {
            update.set(qdo.appropriationType, payload.getAppropriationType());
        }
        // 拨付当量
        if (payload.getAllocateEqva() != null) {
            update.set(qdo.allocateEqva, payload.getAllocateEqva());
        }
        // 拨付费用
        if (payload.getAllocateCost() != null) {
            update.set(qdo.allocateCost, payload.getAllocateCost());
        }
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 预算ID
            if (nullFields.contains("budgetId")) {
                update.setNull(qdo.budgetId);
            }
            // 拨付单号
            if (nullFields.contains("appropriationNo")) {
                update.setNull(qdo.appropriationNo);
            }
            // 拨付状态
            if (nullFields.contains("appropriationStatus")) {
                update.setNull(qdo.appropriationStatus);
            }
            // 申请人资源ID
            if (nullFields.contains("resId")) {
                update.setNull(qdo.resId);
            }
            // 申请费用金额
            if (nullFields.contains("applyFeeAmt")) {
                update.setNull(qdo.applyFeeAmt);
            }
            // 提现当量数
            if (nullFields.contains("applyEqva")) {
                update.setNull(qdo.applyEqva);
            }
            // 提现当量金额
            if (nullFields.contains("applyEqvaAmt")) {
                update.setNull(qdo.applyEqvaAmt);
            }
            // 申请总金额
            if (nullFields.contains("applyAmt")) {
                update.setNull(qdo.applyAmt);
            }
            // 申请日期
            if (nullFields.contains("applyDate")) {
                update.setNull(qdo.applyDate);
            }
            // 流程实例ID
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 流程审批状态
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            // 审批时间
            if (nullFields.contains("approvedTime")) {
                update.setNull(qdo.approvedTime);
            }
            // 拨款名称
            if (nullFields.contains("appropriationName")) {
                update.setNull(qdo.appropriationName);
            }
            // 拨款日期
            if (nullFields.contains("appropriationDate")) {
                update.setNull(qdo.appropriationDate);
            }
            // 拨付附件
            if (nullFields.contains("appropriationFiles")) {
                update.setNull(qdo.appropriationFiles);
            }
            // 流程定义的KEY
            if (nullFields.contains("procDefKey")) {
                update.setNull(qdo.procDefKey);
            }
            // 拨付类型：科目、活动
            if (nullFields.contains("appropriationType")) {
                update.setNull(qdo.appropriationType);
            }
            // 拨付当量
            if (nullFields.contains("allocateEqva")) {
                update.setNull(qdo.allocateEqva);
            }
            // 拨付费用
            if (nullFields.contains("allocateCost")) {
                update.setNull(qdo.allocateCost);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

