package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.ResSetRatePayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.ResSetRateQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.ResSetRateVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QResSetRateDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.ResSetRateDO;
import com.elitesland.tw.tw5.server.prd.purchase.repo.ResSetRateRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 人力资源结算费率
 *
 * @author lemon
 * @date 2023-11-02
 */
@Repository
@RequiredArgsConstructor
public class ResSetRateDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ResSetRateRepo repo;
    private final QResSetRateDO qdo = QResSetRateDO.resSetRateDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ResSetRateVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ResSetRateVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 采购协议单据ID
                qdo.documentId,
                // 采购协议单据编号
                qdo.documentNo,
                // 服务费率
                qdo.serviceRate,
                // 开始金额
                qdo.startAtm,
                // 结束金额
                qdo.endAtm
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ResSetRateVO> getJpaQueryWhere(ResSetRateQuery query) {
        JPAQuery<ResSetRateVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
//        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ResSetRateQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ResSetRateQuery query){
        List<Predicate> list = new ArrayList<>();
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 采购协议单据ID 精确 */
        if (!ObjectUtils.isEmpty(query.getDocumentId())) {
            list.add(qdo.documentId.eq(query.getDocumentId()));
        }
        /** 采购协议单据编号 精确 */
        if (!ObjectUtils.isEmpty(query.getDocumentNo())) {
            list.add(qdo.documentNo.eq(query.getDocumentNo()));
        }
        /** 服务费率 精确 */
        if (!ObjectUtils.isEmpty(query.getServiceRate())) {
            list.add(qdo.serviceRate.eq(query.getServiceRate()));
        }
        /** 开始金额 精确 */
        if (!ObjectUtils.isEmpty(query.getStartAtm())) {
            list.add(qdo.startAtm.eq(query.getStartAtm()));
        }
        /** 结束金额 精确 */
        if (!ObjectUtils.isEmpty(query.getEndAtm())) {
            list.add(qdo.endAtm.eq(query.getEndAtm()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ResSetRateVO queryByKey(Long id) {
        JPAQuery<ResSetRateVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ResSetRateVO> queryListDynamic(ResSetRateQuery query) {
        JPAQuery<ResSetRateVO> jpaQuery = getJpaQueryWhere(query);
        // 按照开始金额升序排序
        jpaQuery.orderBy(qdo.startAtm.asc());
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ResSetRateVO> queryPaging(ResSetRateQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<ResSetRateVO> jpaQuery = getJpaQueryWhere(query);
        List<ResSetRateVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ResSetRateVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ResSetRateDO save(ResSetRateDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ResSetRateDO> saveAll(List<ResSetRateDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ResSetRatePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 采购协议单据ID
        if (payload.getDocumentId() != null) {
            update.set(qdo.documentId, payload.getDocumentId());
        }
        // 采购协议单据编号
        if (payload.getDocumentNo() != null) {
            update.set(qdo.documentNo, payload.getDocumentNo());
        }
        // 服务费率
        if (payload.getServiceRate() != null) {
            update.set(qdo.serviceRate, payload.getServiceRate());
        }
        // 开始金额
        if (payload.getStartAtm() != null) {
            update.set(qdo.startAtm, payload.getStartAtm());
        }
        // 结束金额
        if (payload.getEndAtm() != null) {
            update.set(qdo.endAtm, payload.getEndAtm());
        }
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        if (payload.getDeleteFlag() != null) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 采购协议单据ID
            if (nullFields.contains("documentId")) {
                update.setNull(qdo.documentId);
            }
            // 采购协议单据编号
            if (nullFields.contains("documentNo")) {
                update.setNull(qdo.documentNo);
            }
            // 服务费率
            if (nullFields.contains("serviceRate")) {
                update.setNull(qdo.serviceRate);
            }
            // 开始金额
            if (nullFields.contains("startAtm")) {
                update.setNull(qdo.startAtm);
            }
            // 结束金额
            if (nullFields.contains("endAtm")) {
                update.setNull(qdo.endAtm);
            }
            // 删除标志
            if (nullFields.contains("deleteFlag")) {
                update.setNull(qdo.deleteFlag);
            }
            // 备注
            if (nullFields.contains("remark")) {
                update.setNull(qdo.remark);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 删除人力资源结算费
     * @param documentIdList 协议IdList
     */
    public void deleteSoftByDocumentIdList(List<Long> documentIdList) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.documentId.in(documentIdList));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        update.execute();
    }

    /**
     * 查看人力资源结算费
     * @param agreementId 协议Id
     * @return 人力资源结算费结束金额列表
     */
    public List<ResSetRateVO> findByAgreementId(Long agreementId) {
        JPAQuery<ResSetRateVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.deleteFlag.eq(0)
                .and(qdo.documentId.eq(agreementId)));
        return jpaQuery.fetch();
    }
}

