package com.elitesland.tw.tw5.server.prd.salecon.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.salecon.payload.ConTemplatePayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConTemplateQuery;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConTemplateVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.salecon.entity.ConTemplateDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConTemplateDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConTemplateRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 合同模板
 *
 * @author likunpeng
 * @date 2023-04-17
 */
@Repository
@RequiredArgsConstructor
public class ConTemplateDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ConTemplateRepo repo;
    private final QConTemplateDO qdo = QConTemplateDO.conTemplateDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ConTemplateVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ConTemplateVO.class,
                qdo.id,
//                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 模板名称
                qdo.name,
                // 合同类型
                qdo.conType,
                // 是否标准模板
                qdo.standardTemp,
                // 模板描述
                qdo.tempDesc,
                // 附件
                qdo.fileCodes
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ConTemplateVO> getJpaQueryWhere(ConTemplateQuery query) {
        JPAQuery<ConTemplateVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ConTemplateQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ConTemplateQuery query){
        List<Predicate> list = new ArrayList<>();
        list.add(qdo.deleteFlag.eq(0));
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 模板名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getName())) {
             list.add(qdo.name.like(SqlUtil.toSqlLikeString(query.getName())));
        }
        /** 合同类型 精确 */
        if (!ObjectUtils.isEmpty(query.getConType())) {
            list.add(qdo.conType.eq(query.getConType()));
        }
        /** 是否标准模板 精确 */
        if (!ObjectUtils.isEmpty(query.getStandardTemp())) {
            list.add(qdo.standardTemp.eq(query.getStandardTemp()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ConTemplateVO queryByKey(Long id) {
        JPAQuery<ConTemplateVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ConTemplateVO> queryListDynamic(ConTemplateQuery query) {
        JPAQuery<ConTemplateVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ConTemplateVO> queryPaging(ConTemplateQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ConTemplateVO> jpaQuery = getJpaQueryWhere(query);
        List<ConTemplateVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ConTemplateVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ConTemplateDO save(ConTemplateDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ConTemplateDO> saveAll(List<ConTemplateDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ConTemplatePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 模板名称
        if (payload.getName() != null) {
            update.set(qdo.name, payload.getName());
        }
        // 合同类型
        if (payload.getConType() != null) {
            update.set(qdo.conType, payload.getConType());
        }
        // 是否标准模板
        if (payload.getStandardTemp() != null) {
            update.set(qdo.standardTemp, payload.getStandardTemp());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 模板名称
            if (nullFields.contains("name")) {
                update.setNull(qdo.name);
            }
            // 合同类型
            if (nullFields.contains("conType")) {
                update.setNull(qdo.conType);
            }
            // 是否标准模板
            if (nullFields.contains("standardTemp")) {
                update.setNull(qdo.standardTemp);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
    /**
     * 查询数据库是否存在该VO
     * @param templateName
     * @return ConTemplateVO
     */
    public ConTemplateVO selectInspect(String templateName) {
        JPAQuery<ConTemplateVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.name.eq(templateName));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

}

