package com.elitesland.tw.tw5.server.prd.system.dao;


import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemSitePayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemSiteQuery;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemSiteVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemSiteDO;
import com.elitesland.tw.tw5.server.prd.system.entity.QPrdSystemSiteDO;
import com.elitesland.tw.tw5.server.prd.system.entity.QPrdSystemSiteRefDO;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemSiteRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 公共地点表
 *
 * @author zoey
 * @date 2024-04-22
 */
@Repository
@RequiredArgsConstructor
public class PrdSystemSiteDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PrdSystemSiteRepo repo;
    private final QPrdSystemSiteDO qdo = QPrdSystemSiteDO.prdSystemSiteDO;
    private final QPrdSystemSiteRefDO qdoRef = QPrdSystemSiteRefDO.prdSystemSiteRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemSiteVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdSystemSiteVO.class,
                qdo.id,
                //qdo.remark,
                qdo.createUserId,
                //qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
//                qdo.updater,
                qdo.modifyTime,
                // 地点名称
                qdo.siteDesc,
                // 经度
                qdo.siteLongitude,
                // 纬度
                qdo.siteLatitude,
                // 关系表id
                qdoRef.id.as("refId")
        )).from(qdo)
                .leftJoin(qdoRef).on(qdo.id.longValue().eq(qdoRef.siteId.longValue()).and(qdoRef.deleteFlag.eq(0)));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemSiteVO> getJpaQueryWhere(PrdSystemSiteQuery query) {
        JPAQuery<PrdSystemSiteVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PrdSystemSiteQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo)
                .leftJoin(qdoRef).on(qdo.id.longValue().eq(qdoRef.siteId.longValue()).and(qdoRef.deleteFlag.eq(0)));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PrdSystemSiteQuery query){
        List<Predicate> list = new ArrayList<>();
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 地点名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getSiteDesc())) {
             list.add(qdo.siteDesc.like(SqlUtil.toSqlLikeString(query.getSiteDesc())));
        }
        /** 经度 精确 */
        if (!ObjectUtils.isEmpty(query.getSiteLongitude())) {
            list.add(qdo.siteLongitude.eq(query.getSiteLongitude()));
        }
        /** 纬度 精确 */
        if (!ObjectUtils.isEmpty(query.getSiteLatitude())) {
            list.add(qdo.siteLatitude.eq(query.getSiteLatitude()));
        }
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 单据ID 精确 */
        if (!ObjectUtils.isEmpty(query.getDocId())) {
            list.add(qdoRef.docId.eq(query.getDocId()));
        }
        /** 单据类型 精确 */
        if (!ObjectUtils.isEmpty(query.getDocType())) {
            list.add(qdoRef.docType.eq(query.getDocType()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdSystemSiteVO queryByKey(Long id) {
        JPAQuery<PrdSystemSiteVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrdSystemSiteVO> queryListDynamic(PrdSystemSiteQuery query) {
        JPAQuery<PrdSystemSiteVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdSystemSiteVO> queryPaging(PrdSystemSiteQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PrdSystemSiteVO> jpaQuery = getJpaQueryWhere(query);
        List<PrdSystemSiteVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PrdSystemSiteVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrdSystemSiteDO save(PrdSystemSiteDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PrdSystemSiteDO> saveAll(List<PrdSystemSiteDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PrdSystemSitePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 地点名称
        if (payload.getSiteDesc() != null) {
            update.set(qdo.siteDesc, payload.getSiteDesc());
        }
        // 经度
        if (payload.getSiteLongitude() != null) {
            update.set(qdo.siteLongitude, payload.getSiteLongitude());
        }
        // 纬度
        if (payload.getSiteLatitude() != null) {
            update.set(qdo.siteLatitude, payload.getSiteLatitude());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 地点名称
            if (nullFields.contains("siteDesc")) {
                update.setNull(qdo.siteDesc);
            }
            // 经度
            if (nullFields.contains("siteLongitude")) {
                update.setNull(qdo.siteLongitude);
            }
            // 纬度
            if (nullFields.contains("siteLatitude")) {
                update.setNull(qdo.siteLatitude);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

