package com.elitesland.tw.tw5.server.prd.budget.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.budget.payload.PmsBudgetOccupyPayload;
import com.elitesland.tw.tw5.api.prd.budget.query.PmsBudgetOccupyQuery;
import com.elitesland.tw.tw5.api.prd.budget.vo.PmsBudgetOccupyVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.budget.convert.PmsBudgetOccupyConvert;
import com.elitesland.tw.tw5.server.prd.budget.entity.PmsBudgetOccupyDO;
import com.elitesland.tw.tw5.server.prd.budget.entity.QPmsBudgetOccupyDO;
import com.elitesland.tw.tw5.server.prd.budget.repo.PmsBudgetOccupyRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 预算占用管理
 *
 * @author ben
 * @date 2023-12-05
 */
@Repository
@RequiredArgsConstructor
public class PmsBudgetOccupyDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsBudgetOccupyRepo repo;
    private final QPmsBudgetOccupyDO qdo = QPmsBudgetOccupyDO.pmsBudgetOccupyDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetOccupyDO> getJpaQuerySelect() {
        return jpaQueryFactory.select(qdo).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetOccupyDO> getJpaQueryWhere(PmsBudgetOccupyQuery query) {
        JPAQuery<PmsBudgetOccupyDO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsBudgetOccupyQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsBudgetOccupyQuery query) {
        List<Predicate> list = new ArrayList<>();
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 单据id 精确 */
        if (!ObjectUtils.isEmpty(query.getDocId())) {
            list.add(qdo.docId.eq(query.getDocId()));
        }
        /** 费用类型：当量｜费用 精确 */
        if (!ObjectUtils.isEmpty(query.getCostType())) {
            list.add(qdo.costType.eq(query.getCostType()));
        }
        /** 来源类型：任务｜报销 精确 */
        if (!ObjectUtils.isEmpty(query.getDocType())) {
            list.add(qdo.docType.eq(query.getDocType()));
        }
        /** 单据编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getDocCode())) {
            list.add(qdo.docCode.like(SqlUtil.toSqlLikeString(query.getDocCode())));
        }
        /** 单据名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getDocName())) {
            list.add(qdo.docName.like(SqlUtil.toSqlLikeString(query.getDocName())));
        }
        /** 预算id 精确 */
        if (!ObjectUtils.isEmpty(query.getBudgetId())) {
            list.add(qdo.budgetId.eq(query.getBudgetId()));
        }
        /** 归属资源（当量预算|费用预算）id 精确 */
        if (!ObjectUtils.isEmpty(query.getBelongToId())) {
            list.add(qdo.belongToId.eq(query.getBelongToId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsBudgetOccupyVO queryByKey(Long id) {
        JPAQuery<PmsBudgetOccupyDO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return PmsBudgetOccupyConvert.INSTANCE.toVo(jpaQuery.fetchFirst());
    }

    /**
     * 根据业务Key查询
     *
     * @param budgetId
     * @param docType
     * @param belongToId
     * @param costType
     * @param docId
     * @return 结果
     */
    public PmsBudgetOccupyVO queryByBizKey(Long budgetId, String docType, Long belongToId,
                                           String costType, Long docId) {
        JPAQuery<PmsBudgetOccupyDO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.budgetId.eq(budgetId));
        jpaQuery.where(qdo.docType.eq(docType));
        jpaQuery.where(qdo.belongToId.eq(belongToId));
        jpaQuery.where(qdo.costType.eq(costType));
        jpaQuery.where(qdo.docId.eq(docId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return PmsBudgetOccupyConvert.INSTANCE.toVo(jpaQuery.fetchFirst());
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsBudgetOccupyVO> queryListDynamic(PmsBudgetOccupyQuery query) {
        JPAQuery<PmsBudgetOccupyDO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch().stream().map(PmsBudgetOccupyConvert.INSTANCE::toVo).toList();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsBudgetOccupyVO> queryPaging(PmsBudgetOccupyQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsBudgetOccupyDO> jpaQuery = getJpaQueryWhere(query);
        List<PmsBudgetOccupyDO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        List<PmsBudgetOccupyVO> vos = result.stream().map(PmsBudgetOccupyConvert.INSTANCE::toVo).toList();
        return PagingVO.<PmsBudgetOccupyVO>builder().records(vos).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsBudgetOccupyDO save(PmsBudgetOccupyDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsBudgetOccupyDO> saveAll(List<PmsBudgetOccupyDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsBudgetOccupyPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 单据id
        if (payload.getDocId() != null) {
            update.set(qdo.docId, payload.getDocId());
        }
        // 费用类型：当量｜费用
        if (payload.getCostType() != null) {
            update.set(qdo.costType, payload.getCostType());
        }
        // 来源类型：任务｜报销
        if (payload.getDocType() != null) {
            update.set(qdo.docType, payload.getDocType());
        }
        // 单据编号
        if (payload.getDocCode() != null) {
            update.set(qdo.docCode, payload.getDocCode());
        }
        // 单据名称
        if (payload.getDocName() != null) {
            update.set(qdo.docName, payload.getDocName());
        }
        // 预算id
        if (payload.getBudgetId() != null) {
            update.set(qdo.budgetId, payload.getBudgetId());
        }
        // 归属资源（当量预算|费用预算）id
        if (payload.getBelongToId() != null) {
            update.set(qdo.belongToId, payload.getBelongToId());
        }
        // 占用数
        if (payload.getOccupyAmt() != null) {
            update.set(qdo.occupyAmt, payload.getOccupyAmt());
        }
        // 已使用数
        if (payload.getUsedAmt() != null) {
            update.set(qdo.usedAmt, payload.getUsedAmt());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 单据id
            if (nullFields.contains("docId")) {
                update.setNull(qdo.docId);
            }
            // 费用类型：当量｜费用
            if (nullFields.contains("costType")) {
                update.setNull(qdo.costType);
            }
            // 来源类型：任务｜报销
            if (nullFields.contains("docType")) {
                update.setNull(qdo.docType);
            }
            // 单据编号
            if (nullFields.contains("docCode")) {
                update.setNull(qdo.docCode);
            }
            // 单据名称
            if (nullFields.contains("docName")) {
                update.setNull(qdo.docName);
            }
            // 预算id
            if (nullFields.contains("budgetId")) {
                update.setNull(qdo.budgetId);
            }
            // 归属资源（当量预算|费用预算）id
            if (nullFields.contains("belongToId")) {
                update.setNull(qdo.belongToId);
            }
            // 占用数
            if (nullFields.contains("occupyAmt")) {
                update.setNull(qdo.occupyAmt);
            }
            // 已使用数
            if (nullFields.contains("usedAmt")) {
                update.setNull(qdo.usedAmt);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

