package com.elitesland.tw.tw5.server.prd.budget.service;


import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.budget.payload.BudgetAppropriationDetailPayload;
import com.elitesland.tw.tw5.api.prd.budget.payload.BudgetAppropriationPayload;
import com.elitesland.tw.tw5.api.prd.budget.query.BudgetAppropriationQuery;
import com.elitesland.tw.tw5.api.prd.budget.service.BudgetAppropriationDetailService;
import com.elitesland.tw.tw5.api.prd.budget.service.BudgetAppropriationService;
import com.elitesland.tw.tw5.api.prd.budget.service.BudgetCommonService;
import com.elitesland.tw.tw5.api.prd.budget.vo.BudgetAppropriationVO;
import com.elitesland.tw.tw5.api.prd.budget.vo.BudgetCommonVO;
import com.elitesland.tw.tw5.api.prd.budget.vo.BudgetVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgOrganizationVO;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsProjectActivityService;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsProjectService;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectVO;
import com.elitesland.tw.tw5.api.prd.salecon.service.SaleConContractService;
import com.elitesland.tw.tw5.api.prd.salecon.vo.SaleConContractVO;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemRoleService;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.budget.convert.BudgetAppropriationConvert;
import com.elitesland.tw.tw5.server.prd.budget.dao.BudgetAppropriationDAO;
import com.elitesland.tw.tw5.server.prd.budget.dao.BudgetDAO;
import com.elitesland.tw.tw5.server.prd.budget.entity.BudgetAppropriationDO;
import com.elitesland.tw.tw5.server.prd.budget.repo.BudgetAppropriationRepo;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.SaleConEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.SaleConStatusEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.PmsProcDefKey;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.PmsReasonTypeEnum;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.ProjActivityAllocateStateEnum;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.ProjActivityAllocateTypeEnum;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.elitesland.workflow.payload.ProcessStatusChangePayload;
import com.elitesland.workflow.payload.StartProcessPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

/**
 * 预算拨付基本信息
 *
 * @author xxb
 * @date 2023-11-01
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class BudgetAppropriationServiceImpl extends BaseServiceImpl implements BudgetAppropriationService {

    private final BudgetAppropriationRepo budgetAppropriationRepo;
    private final BudgetAppropriationDAO budgetAppropriationDAO;

    private final BudgetDAO budgetDAO;

    private final PmsProjectService pmsProjectService;

    private final BudgetAppropriationDetailService appropriationDetailService;

    private final PrdSystemRoleService roleService;

    private final WorkflowUtil workflowUtil;

    private final CacheUtil cacheUtil;

    private final BudgetCommonService budgetCommonService;

    private final PmsProjectActivityService pmsProjectActivityService;

    private final SaleConContractService saleConContractService;

    @Override
    public PagingVO<BudgetAppropriationVO> queryPaging(BudgetAppropriationQuery query) {
        PagingVO<BudgetAppropriationVO> pagingVO = budgetAppropriationDAO.queryPaging(query);
        if (!ObjectUtils.isEmpty(pagingVO)) {
            pagingVO.getRecords().forEach(vo -> vo.setCreator(cacheUtil.getUserName(vo.getCreateUserId())));
        }
        return pagingVO;
    }

    @Override
    public List<BudgetAppropriationVO> queryListDynamic(BudgetAppropriationQuery query) {
        return budgetAppropriationDAO.queryListDynamic(query);
    }

    @Override
    public BudgetAppropriationVO queryByKey(Long key) {
        BudgetAppropriationDO entity = budgetAppropriationRepo.findById(key).orElseGet(BudgetAppropriationDO::new);
        Assert.notNull(entity.getId(), "不存在");
        BudgetAppropriationVO vo = BudgetAppropriationConvert.INSTANCE.toVo(entity);
        vo.setCreator(cacheUtil.getUserName(vo.getCreateUserId()));

        // 添加预算信息
        BudgetVO budgetVO = budgetCommonService.queryBudgetForAppropriation(entity.getBudgetId());
        vo.setBudgetCode(budgetVO.getBudgetCode());
        vo.setSourceType(budgetVO.getSourceType());
        vo.setSourceId(budgetVO.getSourceId());
        vo.setSourceName(budgetVO.getSourceName());
        vo.setBudgetName(budgetVO.getBudgetName());
        vo.setRemainingEqva(budgetVO.getRemainingEqva());
        vo.setRemainingAmt(budgetVO.getRemainingAmt());
        vo.setAllocatedAppropriation(budgetVO.getAllocatedAppropriation().multiply(new BigDecimal(100)));
        vo.setActivityVOS(budgetVO.getActivityVOS());
        vo.setContractId(budgetVO.getContractId());
        vo.setWorkType(budgetVO.getWorkType());
        vo.setPlatType(budgetVO.getPlatType());


        //审批页面 需要加上 查看当量使用情况、费用使用情况
        if (!WorkFlowStatusEnum.APPROVING_WORK.getCode().equals(vo.getAppropriationStatus())) {
            //  拨付相关金额相关 信息
            budgetCommonService.getAppropriationInfo(budgetVO);
            //获取占用及使用数据
            BudgetCommonVO commonVO = budgetCommonService.queryBudgetEqvaAndAmt(budgetVO.getSourceType(), budgetVO.getSourceId());
            vo.setAllocatedAmt(budgetVO.getAllocatedAmt());
            vo.setAllocatedEqva(budgetVO.getAllocatedEqva());
            vo.setAllocatedEqvaAmt(budgetVO.getAllocatedEqvaAmt());
            vo.setAllocatedTotalAmt(budgetVO.getAllocatedTotalAmt());

            vo.setOccupyEqva(commonVO.getOccupyEqva());
            vo.setUsedEqva(commonVO.getUsedEqva());
            vo.setPlanEqva(budgetVO.getPlanEqva());

            vo.setOccupyAmt(commonVO.getOccupyAmt());
            vo.setUsedAmt(commonVO.getUsedAmt());
            vo.setPlanAmt(budgetVO.getPlanAmt());
        }
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(BudgetAppropriationPayload payload) {
        //数据验证
        Long budgetId = payload.getBudgetId();
        if (ObjectUtils.isEmpty(budgetId)) {
            throw TwException.error("", "预算ID不能为空");
        }
        BudgetVO budgetVO = budgetDAO.queryByKey(budgetId);
        if (budgetVO == null) {
            throw TwException.error("", "预算不存在");
        }
        BigDecimal eqvaPrice = budgetVO.getEqvaPrice();
        if (eqvaPrice == null || eqvaPrice.compareTo(BigDecimal.ZERO) <= 0) {
            eqvaPrice = BigDecimal.valueOf(2000);
        }
        payload.setEqvaPrice(eqvaPrice);
        payload.setProjName(budgetVO.getSourceName());

        if (ObjectUtils.isEmpty(payload.getId())) {
            // 设置拨付编号
            // payload.setAppropriationNo(generateSeqNum("PMS_BUDGET_APPROPRIATE"));
            payload.setResId(GlobalUtil.getLoginUserId());
        } else {
            BudgetAppropriationVO appropriationVO = budgetAppropriationDAO.queryByKey(payload.getId());
            payload.setProcInstId(appropriationVO.getProcInstId());
        }

        if (payload.getSubmitFlag() && ObjectUtils.isEmpty(payload.getProcInstId())) {
            // 确认当前是否是只有一个流程
            budgetCommonService.checkOnlyOneProc(budgetVO);
        }
        if (budgetVO.getSourceType().equals("PROJ_CONTRACT")) {
            Long projId = budgetVO.getSourceId();
            PmsProjectVO pmsProjectVO = pmsProjectService.queryByKey(projId);
            Long contractId = pmsProjectVO.getContractId();
            payload.setContractId(contractId);
        }
        // 拨付验证 及金额计算
        checkAppropriation(payload, budgetVO);

        // 保存 拨付相关信息
        saveBudgetAppropriation(payload);

        // 保存流程相关信息
        if (payload.getSubmitFlag()) {
            if (ObjectUtils.isEmpty(payload.getProcInstId())) {
                // 获取项目信息
                getProjectInfo(payload);
                submitProc(payload);
            } else {
                // 只单纯更改状态
                BudgetAppropriationPayload payload0 = new BudgetAppropriationPayload();
                payload0.setId(payload.getId());
                payload0.setProcInstStatus(ProcInstStatus.APPROVING.ordinal());
                payload0.setAppropriationStatus(WorkFlowStatusEnum.APPROVING_WORK.getCode());
                budgetAppropriationDAO.updateByKeyDynamic(payload0);
            }
        }
    }

    //checkAppropriation(){
//
//}
    @Override
    public void saveBudgetAppropriation(BudgetAppropriationPayload payload) {
        //数据验证
        if (ObjectUtils.isEmpty(payload.getBudgetId())) {
            throw TwException.error("", "预算ID不能为空");
        }

        if (ObjectUtils.isEmpty(payload.getId())) {
            // 设置拨付编号
            payload.setAppropriationNo(generateSeqNum("PMS_BUDGET_APPROPRIATE"));
        }
        // 保存 拨付基本信息表
        BudgetAppropriationDO entityDo = BudgetAppropriationConvert.INSTANCE.toDo(payload);
        entityDo = budgetAppropriationRepo.save(entityDo);
        payload.setId(entityDo.getId());

        // 保存拨付明细
        appropriationDetailService.batchSave(entityDo.getId(), payload.getAppropriationDetailPayloads());

        // 保存活动信息
        // 如果拨付类型是活动 还需要更新 项目活动表
        if (ProjActivityAllocateTypeEnum.ACTIVITY.getType().equals(payload.getAppropriationType()) && !ObjectUtils.isEmpty(payload.getAppropriationDetailPayloads())) {
            // 由于活动一次拨付只拨付一条记录  所以更新一条活动即可
            BudgetAppropriationDetailPayload appropriationDetailPayload = payload.getAppropriationDetailPayloads().get(0);
            pmsProjectActivityService.updateAllocate(appropriationDetailPayload.getActivityId(), appropriationDetailPayload.getAllocateEqva(), appropriationDetailPayload.getAllocateCost(), ProjActivityAllocateStateEnum.ING.getType());
        }
    }


    /**
     * 发起审批流程
     */
    private void submitProc(BudgetAppropriationPayload payload) {
        boolean outerProjectFlag = budgetCommonService.isExternalProject(payload.getPlatType());

        HashMap<String, Object> variables = new HashMap<>();
        String procInstName = "";
        String processDefinitionKey = "";

        // 判断项目类型 【交付项目、贸易含服务项目、运维项目」T&M项目】是否 走外部项目预算
        if (outerProjectFlag) {
            //  P04.外部项目预算拨付流程-项目名称
            procInstName = "P04.外部项目预算拨付流程" + "-" + payload.getProjName();
            processDefinitionKey = PmsProcDefKey.PMS_BUDGET_ALLOCAT_EXTERNAL.name();

            //交付负责人
            variables.put("Activity_00x3shq", CollUtil.newArrayList(payload.getDeliUserId()));

            //交付BU负责人
            PrdOrgOrganizationVO deliBu = cacheUtil.getOrg(payload.getDeliBuId());
            variables.put("Activity_1inir1s", CollUtil.newArrayList(deliBu.getManageId()));

//            // 平台财务负责人 PLAT_FIN_PIC
//            variables.put("Activity_0uyx6zx", roleService.queryUserIdByRoleCode(RoleEnum.PLAT_FIN_PIC.getCode()));

            // 平台首席财务官 PLAT_CFO
            variables.put("Activity_0uyx6zx", roleService.queryUserIdByRoleCode(RoleEnum.PLAT_CFO.getCode()));

            // 运营总裁
            variables.put("Activity_05xypoo", roleService.queryUserIdByRoleCode(RoleEnum.OPERATION_PRESIDENT.getCode()));

            //获取 验证参数
            variables.putAll(budgetCommonService.getCheckParam(payload, payload.getBudgetId(), payload.getContractId()));
        } else {
            //  P10.内部项目预算拨付流程-项目名称
            procInstName = "P10.内部项目预算拨付流程" + "-" + payload.getProjName();
            processDefinitionKey = PmsProcDefKey.PMS_BUDGET_ALLOCAT_INTERNAL.name();

            //交付负责人
            variables.put("Activity_01iwzn2", CollUtil.newArrayList(payload.getDeliUserId()));
        }

        //发起流程审批
        ProcessInfo processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                processDefinitionKey,
                procInstName,
                payload.getId() + "",
                variables)
        );

        //流程启动成功后，回写业务表数据
        budgetCommonService.updateStatus(payload.getBudgetId(), payload.getId(), processInfo);
    }


    @Override
    public void checkAppropriation(BudgetAppropriationPayload payload, BudgetVO budgetVO) {
        if (ObjectUtils.isEmpty(payload.getAppropriationType())) {
            throw TwException.error("", "拨付类型不能为空");
        }
        List<BudgetAppropriationDetailPayload> appropriationDetailPayloads = payload.getAppropriationDetailPayloads();
        if (ProjActivityAllocateTypeEnum.ACTIVITY.getType().equals(payload.getAppropriationType())) {
            // 若为外部项目 一定要带活动明细
            if (budgetCommonService.isExternalProject(payload.getPlatType())) {
                if (appropriationDetailPayloads.size() == 0) {
                    throw TwException.error("", "活动明细不能为空");
                }
                if (appropriationDetailPayloads.size() > 1) {
                    throw TwException.error("", "如果是活动拨付，单次拨付只能拨付一条活动");
                }
                long count = appropriationDetailPayloads.stream().filter(v -> ObjectUtils.isEmpty(v.getActivityId())).count();
                if (count > 0) {
                    throw TwException.error("", "活动id不能为空");
                }
            }
        }
        if (budgetCommonService.isExternalProject(payload.getPlatType())) {
            //外部项目拨付要重新赋值拨付数据
            // 根据活动计算 当量和金额
            countAmountByActivity(payload);
        }
        //已经拨付当量
        BigDecimal allocatedEqva = budgetVO.getAllocatedEqva() == null ? BigDecimal.ZERO : budgetVO.getAllocatedEqva();
        //已经拨付费用
        BigDecimal allocatedAmt = budgetVO.getAllocatedAmt() == null ? BigDecimal.ZERO : budgetVO.getAllocatedAmt();
        //可拨付当量
        BigDecimal avalEqva = budgetVO.getPlanEqva().subtract(allocatedEqva);
        //可拨付金额
        BigDecimal avalAmt = budgetVO.getPlanAmt().subtract(allocatedAmt);
        //判断合同及合同状态,待激活状态的无合同入场虚拟合同单独做拨付校验
        Long contractId = payload.getContractId();
        SaleConContractVO saleConContractVO = saleConContractService.queryByKey(contractId);
        String status = saleConContractVO.getStatus();
        String platType = saleConContractVO.getPlatType();
        if (platType.equals(SaleConEnum.NO_CONTRACT_VIRTUAL_CONTRACT.getCode()) && (status.equals(SaleConStatusEnum.ACTIVE_WAITING.getCode()) || status.equals(SaleConStatusEnum.APPROVING.getCode()))) {
            avalAmt = BigDecimal.ZERO;
            avalEqva = budgetVO.getPlanEqva().multiply(new BigDecimal("0.15")).setScale(4, RoundingMode.HALF_UP);
            if ((allocatedEqva.add(payload.getApplyEqva())).compareTo(avalEqva) > 0) {
                throw TwException.error("", "无合同入场的项目，总拨付当量不能超过总当量的15%");
            }
            if (payload.getApplyFeeAmt().compareTo(avalAmt) > 0) {
                throw TwException.error("", "无合同入场的项目,总拨付金额不能大于0");
            }
        }
        if (payload.getApplyEqva().compareTo(avalEqva) > 0) {
            throw TwException.error("", "拨付当量不能大于可拨付当量【" + avalEqva + "】");
        }
        if (payload.getApplyFeeAmt().compareTo(avalAmt) > 0) {
            throw TwException.error("", "拨付金额不能大于可拨付金额【" + avalAmt + "】");
        }

    }


    @Override
    public void processStatusChange(ProcessStatusChangePayload processStatusChangePayload) {
        log.info("流程状态变化回调参数:{}", processStatusChangePayload);
        String businessKey = processStatusChangePayload.getBusinessKey();
        //根据业务key查询当前业务对象
        BudgetAppropriationVO vo = budgetAppropriationDAO.queryByKey(Long.valueOf(businessKey));
        if (vo != null) {
            ProcInstStatus procInstStatus = processStatusChangePayload.getProcInstStatus();
            if (procInstStatus.name().equals(ProcInstStatus.INVALID.name())) {
                deleteSoft(vo.getId());
            } else {
                ProcessInfo processInfo = new ProcessInfo();
                processInfo.setProcInstStatus(procInstStatus);
                budgetCommonService.updateStatus(vo.getBudgetId(), vo.getId(), processInfo);
            }
        }
    }

    public void deleteSoft(Long key) {
        budgetAppropriationDAO.deleteSoft(Arrays.asList(key));
        appropriationDetailService.deleteSoftByAppId(key);
    }


    /**
     * 根据活动计算本次拨付金额和当量
     */
    void countAmountByActivity(BudgetAppropriationPayload payload) {
        List<BudgetAppropriationDetailPayload> appropriationDetailVOS = payload.getAppropriationDetailPayloads();
        if (!ObjectUtils.isEmpty(appropriationDetailVOS)) {
            // 总当量
            BigDecimal applyEqva = appropriationDetailVOS.stream().filter(v -> !ObjectUtils.isEmpty(v.getAllocateEqva()))
                    .map(BudgetAppropriationDetailPayload::getAllocateEqva)
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
            // 总当量金额
            BigDecimal applyEqvaAmt = applyEqva.multiply(payload.getEqvaPrice());
            payload.setApplyEqva(applyEqva);
            payload.setApplyEqvaAmt(applyEqvaAmt);
            // 总费用金额
            BigDecimal applyFeeAmt = appropriationDetailVOS.stream().filter(v -> !ObjectUtils.isEmpty(v.getAllocateCost()))
                    .map(BudgetAppropriationDetailPayload::getAllocateCost)
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
            payload.setApplyFeeAmt(applyFeeAmt);
            // 总金额 =费用金额+当量金额
            payload.setApplyAmt(applyFeeAmt.add(applyEqvaAmt));
        } else {
            // 总当量金额
            BigDecimal applyEqvaAmt = payload.getApplyEqva().multiply(payload.getEqvaPrice());
            payload.setApplyEqvaAmt(applyEqvaAmt);
            // 总金额 =费用金额+当量金额
            payload.setApplyAmt(payload.getApplyFeeAmt().add(applyEqvaAmt));
        }
    }

    /**
     * 添加项目相关信息验证操作权限
     *
     * @param payload
     */
    public void getProjectInfo(BudgetAppropriationPayload payload) {
        BudgetVO budgetVO = budgetDAO.queryByKey(payload.getBudgetId());
        if (!ObjectUtils.isEmpty(budgetVO) && PmsReasonTypeEnum.PROJ_CONTRACT.getCode().equals(budgetVO.getSourceType())) {
            PmsProjectVO projectVO = pmsProjectService.queryByKey(budgetVO.getSourceId());
            // 只有项目经理：进行预算编制、变更、拨付等的操作
            if (!ObjectUtils.isEmpty(projectVO)) {
                if (!GlobalUtil.getLoginUserId().equals(projectVO.getPmResId())) {
                    throw TwException.error("", " 只有项目经理可以进行预算编制、变更、拨付等的操作 !");
                }
                // 当量预估单价
                payload.setEqvaPrice(projectVO.getEqvaPrice());
                // 工作类型
                payload.setWorkType(projectVO.getWorkType());
                // 平台合同类型
                payload.setPlatType(projectVO.getPlatType());
                // pmo资源负责人
                payload.setPmoResId(projectVO.getPmoResId());
                // 交付BU_ID
                payload.setDeliBuId(projectVO.getDeliBuId());
                // 交付用户id
                payload.setDeliUserId(projectVO.getDeliUserId());
                // 项目名称
                payload.setProjName(projectVO.getProjName());
                // 合同id
                payload.setContractId(projectVO.getContractId());
            }
        }

    }


}
