package com.elitesland.tw.tw5.server.prd.inv.dao;

import com.elitesland.tw.tw5.api.prd.inv.payload.InvItemPayload;
import com.elitesland.tw.tw5.api.prd.inv.query.InvItemQuery;
import com.elitesland.tw.tw5.api.prd.inv.vo.InvItemVO;
import com.elitesland.tw.tw5.server.prd.inv.entity.InvItemDO;
import com.elitesland.tw.tw5.server.prd.inv.entity.QInvItemDO;
import com.elitesland.tw.tw5.server.prd.inv.repo.InvItemRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 发票商品对象（百望）
 *
 * @author zoey
 * @date 2023-07-31
 */
@Repository
@RequiredArgsConstructor
public class InvItemDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final InvItemRepo repo;
    private final QInvItemDO qdo = QInvItemDO.invItemDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<InvItemVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(InvItemVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 省份
                qdo.province,
                // 商品编码（税务局）
                qdo.goodsCode,
                // 商品编码名称（税务局）
                qdo.goodsCodeName,
                // 商品编码（TW）
                qdo.twGoodsCode,
                // 商品编码名称（TW）
                qdo.twGoodsCodeName,
                // 商品编号
                qdo.goodsNo,
                // 商品名称
                qdo.goodsName,
                // 商品税率
                qdo.goodsTaxRate,
                // 含税标志 N：不含税 Y：含税
                qdo.priceTaxMark,
                // 是否使用优惠政策 0:未使用，1:使用
                qdo.preferentialMark,
                // 优惠政策类型 preferentialMark=Y 必填 preferentialMark =N 不必填
                qdo.preferentialType,
                // 说商品说明
                qdo.goodsDescribe,
                // 商品简码
                qdo.goodsShorthand,
                // 规格型号
                qdo.goodsSpecification,
                // 计量单位
                qdo.goodsUnit,
                // 商品单价
                qdo.goodsPrice,
                // 免税类型 0:正常税率；1:出口免税或其他免税优惠政策；2不征增值税；3普通零税率，默认为0
                qdo.freeTaxType
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<InvItemVO> getJpaQueryWhere(InvItemQuery query) {
        JPAQuery<InvItemVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(InvItemQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(InvItemQuery query){
        List<Predicate> list = new ArrayList<>();
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 省份 精确 */
        if (!ObjectUtils.isEmpty(query.getProvince())) {
            list.add(qdo.province.eq(query.getProvince()));
        }
        /** 商品编码（税务局） 精确 */
        if (!ObjectUtils.isEmpty(query.getGoodsCode())) {
            list.add(qdo.goodsCode.eq(query.getGoodsCode()));
        }
        /** 商品编码名称（税务局） 模糊 */
        if (!ObjectUtils.isEmpty(query.getGoodsCodeName())) {
             list.add(qdo.goodsCodeName.like(SqlUtil.toSqlLikeString(query.getGoodsCodeName())));
        }
        /** 商品编码（TW） 精确 */
        if (!ObjectUtils.isEmpty(query.getTwGoodsCode())) {
            list.add(qdo.twGoodsCode.eq(query.getTwGoodsCode()));
        }
        /** 商品编码名称（TW） 模糊 */
        if (!ObjectUtils.isEmpty(query.getTwGoodsCodeName())) {
             list.add(qdo.twGoodsCodeName.like(SqlUtil.toSqlLikeString(query.getTwGoodsCodeName())));
        }
        /** 商品编号 精确 */
        if (!ObjectUtils.isEmpty(query.getGoodsNo())) {
            list.add(qdo.goodsNo.eq(query.getGoodsNo()));
        }
        /** 商品名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getGoodsName())) {
             list.add(qdo.goodsName.like(SqlUtil.toSqlLikeString(query.getGoodsName())));
        }
        /** 商品税率 精确 */
        if (!ObjectUtils.isEmpty(query.getGoodsTaxRate())) {
            list.add(qdo.goodsTaxRate.eq(query.getGoodsTaxRate()));
        }
        /** 含税标志 N：不含税 Y：含税 精确 */
        if (!ObjectUtils.isEmpty(query.getPriceTaxMark())) {
            list.add(qdo.priceTaxMark.eq(query.getPriceTaxMark()));
        }
        /** 是否使用优惠政策 0:未使用，1:使用 精确 */
        if (!ObjectUtils.isEmpty(query.getPreferentialMark())) {
            list.add(qdo.preferentialMark.eq(query.getPreferentialMark()));
        }
        /** 优惠政策类型 preferentialMark=Y 必填 preferentialMark =N 不必填 精确 */
        if (!ObjectUtils.isEmpty(query.getPreferentialType())) {
            list.add(qdo.preferentialType.eq(query.getPreferentialType()));
        }
        /** 说商品说明 模糊 */
        if (!ObjectUtils.isEmpty(query.getGoodsDescribe())) {
             list.add(qdo.goodsDescribe.like(SqlUtil.toSqlLikeString(query.getGoodsDescribe())));
        }
        /** 商品简码 精确 */
        if (!ObjectUtils.isEmpty(query.getGoodsShorthand())) {
            list.add(qdo.goodsShorthand.eq(query.getGoodsShorthand()));
        }
        /** 规格型号 精确 */
        if (!ObjectUtils.isEmpty(query.getGoodsSpecification())) {
            list.add(qdo.goodsSpecification.eq(query.getGoodsSpecification()));
        }
        /** 计量单位 精确 */
        if (!ObjectUtils.isEmpty(query.getGoodsUnit())) {
            list.add(qdo.goodsUnit.eq(query.getGoodsUnit()));
        }
        /** 商品单价 精确 */
        if (!ObjectUtils.isEmpty(query.getGoodsPrice())) {
            list.add(qdo.goodsPrice.eq(query.getGoodsPrice()));
        }
        /** 免税类型 0:正常税率；1:出口免税或其他免税优惠政策；2不征增值税；3普通零税率，默认为0 精确 */
        if (!ObjectUtils.isEmpty(query.getFreeTaxType())) {
            list.add(qdo.freeTaxType.eq(query.getFreeTaxType()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public InvItemVO queryByKey(Long id) {
        JPAQuery<InvItemVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<InvItemVO> queryListDynamic(InvItemQuery query) {
        JPAQuery<InvItemVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<InvItemVO> queryPaging(InvItemQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<InvItemVO> jpaQuery = getJpaQueryWhere(query);
        List<InvItemVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<InvItemVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public InvItemDO save(InvItemDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<InvItemDO> saveAll(List<InvItemDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(InvItemPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 省份
        if (payload.getProvince() != null) {
            update.set(qdo.province, payload.getProvince());
        }
        // 商品编码（税务局）
        if (payload.getGoodsCode() != null) {
            update.set(qdo.goodsCode, payload.getGoodsCode());
        }
        // 商品编码名称（税务局）
        if (payload.getGoodsCodeName() != null) {
            update.set(qdo.goodsCodeName, payload.getGoodsCodeName());
        }
        // 商品编码（TW）
        if (payload.getTwGoodsCode() != null) {
            update.set(qdo.twGoodsCode, payload.getTwGoodsCode());
        }
        // 商品编码名称（TW）
        if (payload.getTwGoodsCodeName() != null) {
            update.set(qdo.twGoodsCodeName, payload.getTwGoodsCodeName());
        }
        // 商品编号
        if (payload.getGoodsNo() != null) {
            update.set(qdo.goodsNo, payload.getGoodsNo());
        }
        // 商品名称
        if (payload.getGoodsName() != null) {
            update.set(qdo.goodsName, payload.getGoodsName());
        }
        // 商品税率
        if (payload.getGoodsTaxRate() != null) {
            update.set(qdo.goodsTaxRate, payload.getGoodsTaxRate());
        }
        // 含税标志 N：不含税 Y：含税
        if (payload.getPriceTaxMark() != null) {
            update.set(qdo.priceTaxMark, payload.getPriceTaxMark());
        }
        // 是否使用优惠政策 0:未使用，1:使用
        if (payload.getPreferentialMark() != null) {
            update.set(qdo.preferentialMark, payload.getPreferentialMark());
        }
        // 优惠政策类型 preferentialMark=Y 必填 preferentialMark =N 不必填
        if (payload.getPreferentialType() != null) {
            update.set(qdo.preferentialType, payload.getPreferentialType());
        }
        // 说商品说明
        if (payload.getGoodsDescribe() != null) {
            update.set(qdo.goodsDescribe, payload.getGoodsDescribe());
        }
        // 商品简码
        if (payload.getGoodsShorthand() != null) {
            update.set(qdo.goodsShorthand, payload.getGoodsShorthand());
        }
        // 规格型号
        if (payload.getGoodsSpecification() != null) {
            update.set(qdo.goodsSpecification, payload.getGoodsSpecification());
        }
        // 计量单位
        if (payload.getGoodsUnit() != null) {
            update.set(qdo.goodsUnit, payload.getGoodsUnit());
        }
        // 商品单价
        if (payload.getGoodsPrice() != null) {
            update.set(qdo.goodsPrice, payload.getGoodsPrice());
        }
        // 免税类型 0:正常税率；1:出口免税或其他免税优惠政策；2不征增值税；3普通零税率，默认为0
        if (payload.getFreeTaxType() != null) {
            update.set(qdo.freeTaxType, payload.getFreeTaxType());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 省份
            if (nullFields.contains("province")) {
                update.setNull(qdo.province);
            }
            // 商品编码（税务局）
            if (nullFields.contains("goodsCode")) {
                update.setNull(qdo.goodsCode);
            }
            // 商品编码名称（税务局）
            if (nullFields.contains("goodsCodeName")) {
                update.setNull(qdo.goodsCodeName);
            }
            // 商品编码（TW）
            if (nullFields.contains("twGoodsCode")) {
                update.setNull(qdo.twGoodsCode);
            }
            // 商品编码名称（TW）
            if (nullFields.contains("twGoodsCodeName")) {
                update.setNull(qdo.twGoodsCodeName);
            }
            // 商品编号
            if (nullFields.contains("goodsNo")) {
                update.setNull(qdo.goodsNo);
            }
            // 商品名称
            if (nullFields.contains("goodsName")) {
                update.setNull(qdo.goodsName);
            }
            // 商品税率
            if (nullFields.contains("goodsTaxRate")) {
                update.setNull(qdo.goodsTaxRate);
            }
            // 含税标志 N：不含税 Y：含税
            if (nullFields.contains("priceTaxMark")) {
                update.setNull(qdo.priceTaxMark);
            }
            // 是否使用优惠政策 0:未使用，1:使用
            if (nullFields.contains("preferentialMark")) {
                update.setNull(qdo.preferentialMark);
            }
            // 优惠政策类型 preferentialMark=Y 必填 preferentialMark =N 不必填
            if (nullFields.contains("preferentialType")) {
                update.setNull(qdo.preferentialType);
            }
            // 说商品说明
            if (nullFields.contains("goodsDescribe")) {
                update.setNull(qdo.goodsDescribe);
            }
            // 商品简码
            if (nullFields.contains("goodsShorthand")) {
                update.setNull(qdo.goodsShorthand);
            }
            // 规格型号
            if (nullFields.contains("goodsSpecification")) {
                update.setNull(qdo.goodsSpecification);
            }
            // 计量单位
            if (nullFields.contains("goodsUnit")) {
                update.setNull(qdo.goodsUnit);
            }
            // 商品单价
            if (nullFields.contains("goodsPrice")) {
                update.setNull(qdo.goodsPrice);
            }
            // 免税类型 0:正常税率；1:出口免税或其他免税优惠政策；2不征增值税；3普通零税率，默认为0
            if (nullFields.contains("freeTaxType")) {
                update.setNull(qdo.freeTaxType);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

