package com.elitesland.tw.tw5.server.prd.pms.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgOrganizationService;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectEventLogPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectEventLogQuery;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsProjectEventLogService;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectEventLogVO;
import com.elitesland.tw.tw5.server.common.ExcelUtil;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.util.DataUtil;
import com.elitesland.tw.tw5.server.common.util.DateUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.pms.convert.PmsProjectEventLogConvert;
import com.elitesland.tw.tw5.server.prd.pms.dao.PmsProjectEventLogDAO;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectEventLogDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectEventLogRepo;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.InputStream;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 项目事件日志功能管理
 *
 * @author carl
 * @date 2023-09-18
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsProjectEventLogServiceImpl extends BaseServiceImpl implements PmsProjectEventLogService {

    private final PmsProjectEventLogRepo pmsProjectEventLogRepo;
    private final PmsProjectEventLogDAO pmsProjectEventLogDAO;
    private final CacheUtil cacheUtil;
    private final TransactionUtilService transactionUtilService;
    private final PrdOrgOrganizationService service;
    private final ExcelUtil excelUtil;
    private final UdcUtil udcUtil;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsProjectEventLogVO insertOrUpdate(PmsProjectEventLogPayload payload) {

        //数据长度校验
        DataUtil.checkDataLength(payload, null);

        if (payload.getId() != null) {
            PmsProjectEventLogVO projectEventLogVO = pmsProjectEventLogDAO.queryByKey(payload.getId());
            if (!ObjectUtils.isEmpty(projectEventLogVO)) {
                if (projectEventLogVO.getStatus().equals("COMPLETED")) {
                    throw TwException.error("", "已完成状态不可修改");
                }
            } else {
                throw TwException.error("", "修改数据不存在");
            }
        }
        if (payload.getStatus().equals("COMPLETED")) {
            payload.setFinishTime(LocalDateTime.now());
        }

        PmsProjectEventLogDO entityDo = PmsProjectEventLogConvert.INSTANCE.toDo(payload);
        return PmsProjectEventLogConvert.INSTANCE.toVo(pmsProjectEventLogRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateStatus(Long id) {
        PmsProjectEventLogPayload payload = new PmsProjectEventLogPayload();
        payload.setId(id);
        payload.setStatus("COMPLETED");
        payload.setFinishTime(LocalDateTime.now());
        return pmsProjectEventLogDAO.updateByKeyDynamic(payload);
    }

    @Override
    public PagingVO<PmsProjectEventLogVO> queryPaging(PmsProjectEventLogQuery query) {
        PagingVO<PmsProjectEventLogVO> pmsProjectEventLogVOPagingVO = pmsProjectEventLogDAO.queryPaging(query);
        if (pmsProjectEventLogVOPagingVO.getTotal() > 0) {
            pmsProjectEventLogVOPagingVO.getRecords().forEach(this::tranData);
        }
        return pmsProjectEventLogVOPagingVO;
    }

    void tranData(PmsProjectEventLogVO vo) {
        if (vo != null) {
            vo.setCreator(cacheUtil.getUserName(vo.getCreateUserId()));
        }
    }

    @Override
    public PagingVO<PmsProjectEventLogVO> queryPushPaging(PmsProjectEventLogQuery query) {
        Long loginUserId = GlobalUtil.getLoginUserId();
        //查询作为bu负责人的所有组织id
        List<Long> orgIds = service.queryByManageIdOrgIds(loginUserId);
        query.setLoginUserId(loginUserId);
        query.setHeadOrgIds(orgIds);
        PagingVO<PmsProjectEventLogVO> pmsProjectEventLogVOPagingVO = pmsProjectEventLogDAO.queryPushPaging(query);
        if (pmsProjectEventLogVOPagingVO.getTotal() > 0) {
            List<PmsProjectEventLogVO> records = pmsProjectEventLogVOPagingVO.getRecords();
            /**
             * 1.先处理实际延期状态没改的
             * 2.处理当天延期已超过十二点的
             * 3.统一赋值备注字段
             */
            LocalDate localDate = LocalDate.now();
            List<Long> ids = new ArrayList<>();
            List<PmsProjectEventLogVO> identified = records.stream().filter(record -> record.getStatus().equals("IDENTIFIED")
                    && (record.getExpectedEndDate() != null && localDate.toEpochDay() - record.getExpectedEndDate().toEpochDay() > 0)).collect(Collectors.toList());
            int hour = LocalDateTime.now().getHour();
            if (hour >= 12) {
                List<PmsProjectEventLogVO> identified0 = records.stream().filter(record -> record.getStatus().equals("IDENTIFIED")
                        && (record.getExpectedEndDate() != null && localDate.toEpochDay() - record.getExpectedEndDate().toEpochDay() == 0)).collect(Collectors.toList());
                if (!ObjectUtils.isEmpty(identified0)) {
                    if (ObjectUtils.isEmpty(identified)) {
                        identified = new ArrayList<>();
                    }
                    identified.addAll(identified0);
                }
            }
            identified.forEach(eventLogVO -> {
                eventLogVO.setStatus("EXTENDED");
                eventLogVO.setRiskLevel("9");
                ids.add(eventLogVO.getId());
            });
            records.forEach(eventLogVO -> {
                String remark = "已识别待处理";
                long delayDay = 0;
                if (eventLogVO.getStatus().equals("EXTENDED")) {
                    delayDay = localDate.toEpochDay() - eventLogVO.getExpectedEndDate().toEpochDay();
                    if (delayDay <= 7) {
                        remark = "已延期" + delayDay + "天";
                    } else {
                        remark = "严重延期未处理";
                    }
                }
                eventLogVO.setRemark(remark);
                eventLogVO.setDelayDay(delayDay);
            });

            if (ids.size() > 0) {
                //开启事务执行修改，主要是修改审批状态
                transactionUtilService.executeWithRunnable(() -> {
                    pmsProjectEventLogDAO.updateStatus(ids, "EXTENDED", "9");
                });
            }
        }

        return pmsProjectEventLogVOPagingVO;
    }


    @Override
    public PmsProjectEventLogVO queryByKey(Long key) {
        PmsProjectEventLogVO pmsProjectEventLogVO = pmsProjectEventLogDAO.queryByKey(key);
        tranData(pmsProjectEventLogVO);
        return pmsProjectEventLogVO;
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            List<PmsProjectEventLogVO> pmsProjectEventLogVOs = pmsProjectEventLogDAO.queryByKeys(keys);
            if (!ObjectUtils.isEmpty(pmsProjectEventLogVOs)) {
                List<PmsProjectEventLogVO> identified = pmsProjectEventLogVOs.stream().filter(vo -> !vo.getStatus().equals("IDENTIFIED")).collect(Collectors.toList());
                if (!ObjectUtils.isEmpty(identified)) {
                    throw TwException.error("", "只有已识别状态的事件日志可以删除");
                }
            }
            pmsProjectEventLogDAO.deleteSoft(keys);
        }
    }

    @Override
    public void downloadPlus(HttpServletResponse response) {
        ClassPathResource classPathResource = new ClassPathResource("template/projectEventLog.xlsx");
        try {
            InputStream inputStream = classPathResource.getInputStream();
            Workbook workbook = WorkbookFactory.create(inputStream);
            String fileName = "项目事件日志导入模板-" + LocalDate.now();
            ExcelUtil.writeResponse(response, fileName, workbook);
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    @Override
    public void exportExcel(PmsProjectEventLogQuery query, HttpServletResponse response) {
        ClassPathResource classPathResource = new ClassPathResource("template/projectEventLogExport.xlsx");
        try {
            InputStream inputStream = classPathResource.getInputStream();
            Workbook workbook = WorkbookFactory.create(inputStream);
            XSSFSheet batchProjectSheet = (XSSFSheet) workbook.getSheet("项目事件日志");
            String fileName = "项目事件日志-" + LocalDate.now();
            List<PmsProjectEventLogVO> pmsProjectEventLogVOS = pmsProjectEventLogDAO.queryListDynamic(query);
            if (!ObjectUtils.isEmpty(pmsProjectEventLogVOS)) {
                pmsProjectEventLogVOS = udcUtil.translateList(pmsProjectEventLogVOS);
                int nextRow = 1;
                for (PmsProjectEventLogVO eventLogVO : pmsProjectEventLogVOS) {
                    tranData(eventLogVO);
                    Row row = batchProjectSheet.createRow(nextRow);
                    excelUtil.setCellValue(row, 0, nextRow); // 序号
                    excelUtil.setCellValue(row, 1, eventLogVO.getProjNo()); //
                    excelUtil.setCellValue(row, 2, eventLogVO.getProjName()); //
                    excelUtil.setCellValue(row, 3, eventLogVO.getDeliBuName()); //
                    excelUtil.setCellValue(row, 4, eventLogVO.getPmResName()); //
                    excelUtil.setCellValue(row, 5, eventLogVO.getEventContentSource()); //
                    excelUtil.setCellValue(row, 6, eventLogVO.getPossibleImpacts()); //
                    excelUtil.setCellValue(row, 7, eventLogVO.getHeadResName()); //负责人
                    excelUtil.setCellValue(row, 8, eventLogVO.getTypeDesc()); //类型
                    excelUtil.setCellValue(row, 9, eventLogVO.getSeverityDesc()); //严重程度
                    excelUtil.setCellValue(row, 10, eventLogVO.getProbabilityOfOccurrenceDesc()); //发生几率
                    excelUtil.setCellValue(row, 11, eventLogVO.getRiskLevelDesc()); //
                    excelUtil.setCellValue(row, 12, eventLogVO.getCopingStrategy()); //
                    excelUtil.setCellValue(row, 13, eventLogVO.getExpectedEndDate()); //
                    excelUtil.setCellValue(row, 14, eventLogVO.getStatusDesc()); //
                    excelUtil.setCellValue(row, 15, eventLogVO.getCreator()); //
                    excelUtil.setCellValue(row, 16, eventLogVO.getCreateTime()); //
                    excelUtil.setCellValue(row, 17, eventLogVO.getFinishTime()); //

                    nextRow++;
                }
            }

            ExcelUtil.writeResponse(response, fileName, workbook);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean importEmployees(MultipartFile file, Long projectId) {
        if (file == null) {
            throw TwException.error("", "上传文件异常");
        }
        Workbook workbook = null;
        try {
            workbook = WorkbookFactory.create(file.getInputStream());
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.getMessage());
            throw TwException.error("", "文件解析异常");
        }
        Sheet sheet = workbook.getSheet("项目事件日志");
        if (sheet == null) {
            throw TwException.error("", "表结构错误");
        }
        int dataStartRow = 2;

        List<PmsProjectEventLogDO> eventLogDos = new ArrayList<>();
        for (int i = dataStartRow; i <= sheet.getLastRowNum(); i++) {
            Row row = sheet.getRow(i);
            String eventContentSource = ExcelUtil.getCellFormatValue(row.getCell(1));
            if (!StringUtils.hasText(eventContentSource)) {
                throw TwException.error("", "事件内容及来源不可为空");
            }
            String possibleImpacts = ExcelUtil.getCellFormatValue(row.getCell(2));
            String type = ExcelUtil.getCellFormatValue(row.getCell(3));
            if (StringUtils.hasText(type)) {
                type = cacheUtil.getSystemSelectionValueByName("PMS_PROJECT_EVENT_TYPE", type);
                if (!StringUtils.hasText(type)) {
                    throw TwException.error("", "所属类型不存在");
                }
            } else {
                throw TwException.error("", "所属类型不可为空");
            }
            String severity = ExcelUtil.getCellFormatValue(row.getCell(4));
            if (StringUtils.hasText(severity)) {
                severity = cacheUtil.getSystemSelectionValueByName("PMS:PROJECT:EVENT:SEVERITY", severity);
                if (!StringUtils.hasText(severity)) {
                    throw TwException.error("", "严重程度不存在");
                }
            } else {
                throw TwException.error("", "严重程度不可为空");
            }
            String probabilityOfOccurrence = ExcelUtil.getCellFormatValue(row.getCell(5));
            if (StringUtils.hasText(probabilityOfOccurrence)) {
                probabilityOfOccurrence = cacheUtil.getSystemSelectionValueByName("PMS:PROJECT:EVENT:ODDS", probabilityOfOccurrence);
            }
            String riskLevel = ExcelUtil.getCellFormatValue(row.getCell(6));
            if (StringUtils.hasText(riskLevel)) {
                riskLevel = cacheUtil.getSystemSelectionValueByName("PMS:PROJECT:RISK", riskLevel);
            }
            String copingStrategy = ExcelUtil.getCellFormatValue(row.getCell(7));

            String expectedStr = ExcelUtil.getCellFormatValue(row.getCell(8));
            LocalDate expectedEndDate = null;
            if (StringUtils.hasText(expectedStr)) {
                expectedEndDate = DateUtil.strToLocalDate(expectedStr);
            }
            String status = ExcelUtil.getCellFormatValue(row.getCell(9));
            if (StringUtils.hasText(status)) {
                status = cacheUtil.getSystemSelectionValueByName("PMS_PROJECT_EVENT_STATUS", status);
                if (!StringUtils.hasText(status)) {
                    throw TwException.error("", "状态不存在");
                }
            } else {
                throw TwException.error("", "状态不可为空");
            }
            PmsProjectEventLogDO eventLogDO = new PmsProjectEventLogDO();
            eventLogDO.setCopingStrategy(copingStrategy);
            eventLogDO.setEventContentSource(eventContentSource);
            eventLogDO.setExpectedEndDate(expectedEndDate);
            eventLogDO.setPossibleImpacts(possibleImpacts);
            eventLogDO.setProbabilityOfOccurrence(probabilityOfOccurrence);
            eventLogDO.setRiskLevel(riskLevel);
            eventLogDO.setStatus(status);
            eventLogDO.setType(type);
            eventLogDO.setProjectId(projectId);
            eventLogDO.setSeverity(severity);
            if (status.equals("COMPLETED")) {
                eventLogDO.setFinishTime(LocalDateTime.now());
            }
            eventLogDos.add(eventLogDO);

        }
        //数据长度校验
        DataUtil.checkDataLength(eventLogDos, null);
        pmsProjectEventLogDAO.saveAll(eventLogDos);
        return true;
    }

}
