package com.elitesland.tw.tw5.server.prd.salecon.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.salecon.payload.ConPurchaseDemandPayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConPurchaseDemandQuery;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConPurchaseDemandVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.crm.entity.QCrmOpportunityDO;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBusinessPartnerDO;
import com.elitesland.tw.tw5.server.prd.prj.entity.QPrjProjectDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.ConPurchaseDemandDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConPurchaseDemandDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QSaleConContractDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConPurchaseDemandRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 采购需求处理
 *
 * @author likunpeng
 * @date 2023-03-29
 */
@Repository
@RequiredArgsConstructor
public class ConPurchaseDemandDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ConPurchaseDemandRepo repo;
    private final QConPurchaseDemandDO qdo = QConPurchaseDemandDO.conPurchaseDemandDO;

    private final QSaleConContractDO qdoSaleCon = QSaleConContractDO.saleConContractDO;

    private final QCrmOpportunityDO qdoOppo = QCrmOpportunityDO.crmOpportunityDO;

    private final QPrjProjectDO qdoProj = QPrjProjectDO.prjProjectDO;
    //客户表
   // private final QCrmCustomerDO qdoCrmCust = QCrmCustomerDO.crmCustomerDO;
    private final QBusinessPartnerDO qBusinessPartnerDO = QBusinessPartnerDO.businessPartnerDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ConPurchaseDemandVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ConPurchaseDemandVO.class,
                        qdo.id,
                        qdo.remark,
                        qdo.createUserId,
                        qdo.creator,
                        qdo.createTime,
                        qdo.modifyUserId,
                        qdo.updater,
                        qdo.modifyTime,
                        qdo.purDemandIdV4,
                        // 需求编号
                        qdo.code,
                        // 需求总金额
                        qdo.demandTotalAmt,
                        // 销售合同id
                        qdo.saleConId,
                        // 销售合同号
                        qdoSaleCon.code.as("saleConCode"),
                        //客户Id
                        qdoSaleCon.custId,
                        // 客户名称
                       // qdoCrmCust.customerName.as("custName"),
                        qBusinessPartnerDO.partnerName.as("custName"),
                        qdoSaleCon.demandType,
                        qdoOppo.id.as("oppoId"),
                        qdoProj.projectName.as("oppoName")
                )).from(qdo)
                .leftJoin(qdoSaleCon).on(qdoSaleCon.id.longValue().eq(qdo.saleConId.longValue()))
                .leftJoin(qdoOppo).on(qdoSaleCon.oppoId.longValue().eq(qdoOppo.id.longValue()))
                .leftJoin(qdoProj).on(qdoOppo.projectId.longValue().eq(qdoProj.id.longValue()))
               // .leftJoin(qdoCrmCust).on(qdoSaleCon.custId.longValue().eq(qdoCrmCust.bookId.longValue()));
                .leftJoin(qBusinessPartnerDO).on(qdoSaleCon.custId.longValue().eq(qBusinessPartnerDO.bookId.longValue()));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ConPurchaseDemandVO> getJpaQueryWhere(ConPurchaseDemandQuery query) {
        JPAQuery<ConPurchaseDemandVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ConPurchaseDemandQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ConPurchaseDemandQuery query){
        List<Predicate> list = new ArrayList<>();
        list.add(qdo.deleteFlag.eq(0));
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 需求编号 精确 */
        if (!ObjectUtils.isEmpty(query.getCode())) {
            list.add(qdo.code.eq(query.getCode()));
        }
        /** 需求总金额 精确 */
        if (!ObjectUtils.isEmpty(query.getDemandTotalAmt())) {
            list.add(qdo.demandTotalAmt.eq(query.getDemandTotalAmt()));
        }
        /** 销售合同id 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleConId())) {
            list.add(qdo.saleConId.eq(query.getSaleConId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ConPurchaseDemandVO queryByKey(Long id) {
        JPAQuery<ConPurchaseDemandVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ConPurchaseDemandVO> queryListDynamic(ConPurchaseDemandQuery query) {
        JPAQuery<ConPurchaseDemandVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ConPurchaseDemandVO> queryPaging(ConPurchaseDemandQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<ConPurchaseDemandVO> jpaQuery = getJpaQueryWhere(query);
        List<ConPurchaseDemandVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ConPurchaseDemandVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ConPurchaseDemandDO save(ConPurchaseDemandDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ConPurchaseDemandDO> saveAll(List<ConPurchaseDemandDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ConPurchaseDemandPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 需求编号
        if (payload.getCode() != null) {
            update.set(qdo.code, payload.getCode());
        }
        // 需求总金额
        if (payload.getDemandTotalAmt() != null) {
            update.set(qdo.demandTotalAmt, payload.getDemandTotalAmt());
        }
        // 销售合同id
        if (payload.getSaleConId() != null) {
            update.set(qdo.saleConId, payload.getSaleConId());
        }
        // 销售合同id
        if (payload.getPurDemandIdV4() != null) {
            update.set(qdo.purDemandIdV4, payload.getPurDemandIdV4());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 需求编号
            if (nullFields.contains("code")) {
                update.setNull(qdo.code);
            }
            // 需求总金额
            if (nullFields.contains("demandTotalAmt")) {
                update.setNull(qdo.demandTotalAmt);
            }
            // 销售合同id
            if (nullFields.contains("saleConId")) {
                update.setNull(qdo.saleConId);
            }
            // 销售合同id
            if (nullFields.contains("purDemandIdV4")) {
                update.setNull(qdo.purDemandIdV4);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据合同ID查询采购需求
     * @param saleConId
     * @return ConPurchaseDemandVO
     */
    public ConPurchaseDemandVO queryBySaleConId(Long saleConId) {
        JPAQuery<ConPurchaseDemandVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.saleConId.eq(saleConId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.groupBy(qdo.id);
        return jpaQuery.fetchOne();
    }

    /**
     * 修改合同的有效合同金额
     * @param newEffectiveAmt
     * @param conId
     */
    public void updateSubConEffectiveAmt(BigDecimal newEffectiveAmt, Long conId,BigDecimal sysGross) {
        JPAUpdateClause update = jpaQueryFactory.update(qdoSaleCon);
        update.set(qdoSaleCon.effectiveAmt,newEffectiveAmt);
        if (!ObjectUtils.isEmpty(sysGross)){
            update.set(qdoSaleCon.sysGross,sysGross);
        }
        update.where(qdoSaleCon.id.eq(conId)).execute();
    }
}

