package com.elitesland.tw.tw5.server.prd.acc.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccBudgetItemPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccBudgetItemQuery;
import com.elitesland.tw.tw5.api.prd.acc.service.AccBudgetItemService;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccBudgetItemVO;
import com.elitesland.tw.tw5.server.common.ExcelUtil;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.prd.acc.convert.AccBudgetItemConvert;
import com.elitesland.tw.tw5.server.prd.acc.dao.AccBudgetItemDAO;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccBudgetItemDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccBudgetItemRepo;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.InputStream;
import java.time.LocalDate;
import java.util.*;

/**
 * 预算项目管理
 *
 * @author carl
 * @date 2023-09-25
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class AccBudgetItemServiceImpl extends BaseServiceImpl implements AccBudgetItemService {

    private final AccBudgetItemRepo accBudgetItemRepo;
    private final AccBudgetItemDAO accBudgetItemDAO;
    private final ExcelUtil excelUtil;
    private final UdcUtil udcUtil;
    private final TransactionUtilService transactionUtilService;

    @Override
    public PagingVO<AccBudgetItemVO> queryPaging(AccBudgetItemQuery query) {
        return accBudgetItemDAO.queryPaging(query);
    }

    @Override
    public List<AccBudgetItemVO> queryListDynamic(AccBudgetItemQuery query) {
        return accBudgetItemDAO.queryListDynamic(query);
    }


    @Override
    public AccBudgetItemVO queryByKey(Long key) {

        return accBudgetItemDAO.queryByKey(key);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public AccBudgetItemVO insert(AccBudgetItemPayload payload) {
        checkData(payload);
        List<Long> ids = accBudgetItemDAO.queryByCodeAndName(payload.getBudgetCode(), payload.getBudgetName());
        if (ObjectUtils.isEmpty(ids)) {
            if (!StringUtils.hasText(payload.getBudgetStatus())) {
                payload.setBudgetStatus("ACTIVE");
            }
            payload.setBudgetLevel(1);
            //如果是,要判断上级
            if (!ObjectUtils.isEmpty(payload.getParentId())) {
                AccBudgetItemVO accBudgetItemVO = accBudgetItemDAO.queryByKey(payload.getParentId());
                payload.setBudgetLevel(accBudgetItemVO.getBudgetLevel() + 1);
            }
            AccBudgetItemDO entityDo = AccBudgetItemConvert.INSTANCE.toDo(payload);
            return AccBudgetItemConvert.INSTANCE.toVo(accBudgetItemRepo.save(entityDo));
        } else {
            throw TwException.error("", "科目编号和名称不可重复");
        }
    }

    /**
     * 数据校验
     *
     * @param payload
     */
    void checkData(AccBudgetItemPayload payload) {
        if (!StringUtils.hasText(payload.getBudgetCode()) || !StringUtils.hasText(payload.getBudgetName())) {
            throw TwException.error("", "编号和名称不可为空");
        }
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(AccBudgetItemPayload payload) {
        List<Long> ids = accBudgetItemDAO.queryByCodeAndName(payload.getBudgetCode(), payload.getBudgetName());
        if (ObjectUtils.isEmpty(ids) || (ids.size() == 1 && ids.get(0).equals(payload.getId()))) {
            payload.setBudgetLevel(1);
            //如果是,要判断上级
            if (!ObjectUtils.isEmpty(payload.getParentId())) {
                AccBudgetItemVO accBudgetItemVO = accBudgetItemDAO.queryByKey(payload.getParentId());
                payload.setBudgetLevel(accBudgetItemVO.getBudgetLevel() + 1);
            }
            long result = accBudgetItemDAO.updateByKeyDynamic(payload);
            return result;
        } else {
            throw TwException.error("", "科目编号和名称不可重复");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            accBudgetItemDAO.deleteSoft(keys);
        }
    }

    @Override
    public void downloadPlus(HttpServletResponse response, AccBudgetItemQuery query) {
        ClassPathResource classPathResource = new ClassPathResource("template/accBudgetItem.xlsx");
        try {
            InputStream inputStream = classPathResource.getInputStream();
            Workbook workbook = WorkbookFactory.create(inputStream);
            XSSFSheet batchProjectSheet = (XSSFSheet) workbook.getSheet("科目数据");
            excelUtil.generateRangeList(batchProjectSheet, 4, 1, "LOV", 2, "C"); // 科目状态 数据验证
            excelUtil.generateRangeList(batchProjectSheet, 9, 1, "LOV", 2, "A"); // 汇总 数据验证
            excelUtil.generateRangeList(batchProjectSheet, 3, 1, "科目数据", 2, "C"); // 上级会计科目 数据验证
            String fileName = "预算项目数据-" + LocalDate.now();
            Boolean isHave = false;
            if (!StringUtils.hasText(query.getDownloadType()) || query.getDownloadType().equals("data")) {
                //数据导出
                List<AccBudgetItemVO> accBudgetItemVOS = accBudgetItemDAO.queryListDynamic(query);
                if (!ObjectUtils.isEmpty(accBudgetItemVOS)) {
                    accBudgetItemVOS = udcUtil.translateList(accBudgetItemVOS);
                    isHave = true;
                    int nextRow = 1;
                    for (AccBudgetItemVO accBudgetItemVO : accBudgetItemVOS) {
                        Row row = batchProjectSheet.createRow(nextRow);

                        Optional<AccBudgetItemVO> first = accBudgetItemVOS.stream().filter(vo -> vo.getId().equals(accBudgetItemVO.getParentId())).findFirst();
                        if (first.isPresent()) {
                            excelUtil.setCellValue(row, 3, first.get().getBudgetName());// 上级名称
                        }
                        excelUtil.setCellValue(row, 0, nextRow); // 序号
                        excelUtil.setCellValue(row, 1, accBudgetItemVO.getBudgetCode());// 编号
                        excelUtil.setCellValue(row, 2, accBudgetItemVO.getBudgetName());// 名称
                        excelUtil.setCellValue(row, 4, accBudgetItemVO.getBudgetStatusDesc());// 科目状态
                        excelUtil.setCellValue(row, 5, accBudgetItemVO.getBudgetType1());// 大类
                        excelUtil.setCellValue(row, 6, accBudgetItemVO.getBudgetType2());// 明细类1
                        excelUtil.setCellValue(row, 7, accBudgetItemVO.getBudgetType3());// 明细类2
                        excelUtil.setCellValue(row, 8, accBudgetItemVO.getDtlAcc());// 明细账

                        String sumFlag = "是";
                        if (accBudgetItemVO.getSumFlag() == null || accBudgetItemVO.getSumFlag() == 0) {
                            sumFlag = "否";
                        }
                        excelUtil.setCellValue(row, 9, sumFlag);// 汇总
                        excelUtil.setCellValue(row, 10, accBudgetItemVO.getLedgerType());// 子账类型
                        excelUtil.setCellValue(row, 11, accBudgetItemVO.getRemark());// 备注
                        nextRow++;
                    }
                }
            } else {
                //下载模板
                fileName = "预算项目导入模板-" + LocalDate.now();
            }

            if (!isHave) {
                excelUtil.setColumnFormulas(batchProjectSheet, 12, "VLOOKUP(D" + ExcelUtil.formulasReplace + ",B:C,1,FALSE)");//公海列表
                excelUtil.setColumnFormulas(batchProjectSheet, 13, "VLOOKUP(E" + ExcelUtil.formulasReplace + ",LOV!C:D,2,FALSE)");//市场活动
                excelUtil.setColumnFormulas(batchProjectSheet, 14, "VLOOKUP(J" + ExcelUtil.formulasReplace + ",LOV!A:B,2,FALSE)");
            }
            batchProjectSheet.setColumnHidden(12, true);
            batchProjectSheet.setColumnHidden(13, true);
            batchProjectSheet.setColumnHidden(14, true);
            ExcelUtil.writeResponse(response, fileName, workbook);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean batchImport(MultipartFile file) {

        if (file == null) {
            throw TwException.error("", "上传文件异常");
        }
        Workbook workbook = null;
        try {
            workbook = WorkbookFactory.create(file.getInputStream());
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.getMessage());
            throw TwException.error("", "文件解析异常");
        }
        Sheet sheet = workbook.getSheet("科目数据");
        if (sheet == null) {
            throw TwException.error("", "表结构错误");
        }
        int dataStartRow = 1;

        List<String> codeNames = new ArrayList<>();
        Map<String, String> map = new HashMap<>();
        List<AccBudgetItemPayload> accBudgetItemPayloads = new ArrayList<>();
        for (int i = dataStartRow; i <= sheet.getLastRowNum(); i++) {
            Row row = sheet.getRow(i);
            String accCode = ExcelUtil.getCellFormatValue(row.getCell(1));
            String accName = ExcelUtil.getCellFormatValue(row.getCell(2));
            if (!StringUtils.hasText(accCode) || !StringUtils.hasText(accName)) {
                break;
            }
            map.put(accName, accCode);
            String codeStr = "," + accCode + ",";
            String nameStr = "," + accName + ",";
            if (codeNames.contains(codeStr) || codeNames.contains(nameStr)) {
                throw TwException.error("", "科目编号或名称不可重复:【" + accCode + "-" + accName + "】");
            } else {
                codeNames.add(codeStr);
                codeNames.add(nameStr);
            }
            String upperName = ExcelUtil.getCellFormatValue(row.getCell(3));
            String accType1 = ExcelUtil.getCellFormatValue(row.getCell(5));
            String accType2 = ExcelUtil.getCellFormatValue(row.getCell(6));
            String accType3 = ExcelUtil.getCellFormatValue(row.getCell(7));
            String dtlAcc = ExcelUtil.getCellFormatValue(row.getCell(8));//明细账
            String ledgerType = ExcelUtil.getCellFormatValue(row.getCell(10));//子账类型
            String remark = ExcelUtil.getCellFormatValue(row.getCell(11));//备注
            String upperCode = ExcelUtil.getCellFormatValue(row.getCell(12));//上级科目编号
            String accStatus = ExcelUtil.getCellFormatValue(row.getCell(13));//状态
            if (!StringUtils.hasText(accStatus)) {
                accStatus = "ACTIVE";
            }

            String sumFlag = ExcelUtil.getCellFormatValue(row.getCell(14));
            if (!StringUtils.hasText(sumFlag)) {
                sumFlag = "0";
            }
            AccBudgetItemPayload accBudgetItemPayload = new AccBudgetItemPayload();
            accBudgetItemPayload.setBudgetName(accName);
            accBudgetItemPayload.setBudgetCode(accCode);
            accBudgetItemPayload.setBudgetStatus(accStatus);
            accBudgetItemPayload.setParentCode(upperCode);
            accBudgetItemPayload.setParentName(upperName);
            accBudgetItemPayload.setBudgetType1(accType1);
            accBudgetItemPayload.setBudgetType2(accType2);
            accBudgetItemPayload.setBudgetType3(accType3);
            accBudgetItemPayload.setLedgerType(ledgerType);
            accBudgetItemPayload.setDtlAcc(dtlAcc);
            accBudgetItemPayload.setSumFlag(Integer.valueOf(sumFlag));
            accBudgetItemPayload.setRemark(remark);
            if (!StringUtils.hasText(upperName) && !StringUtils.hasText(upperCode)) {
                accBudgetItemPayload.setBudgetLevel(1);
            }

            accBudgetItemPayloads.add(accBudgetItemPayload);
        }
        List<AccBudgetItemDO> accBudgetItemDOs = new ArrayList<>();
        accBudgetItemPayloads.forEach(accBudgetItemPayload -> {
            String upperName = accBudgetItemPayload.getParentName();
            String upperCode = accBudgetItemPayload.getParentCode();
            if (StringUtils.hasText(upperName)) {
                if (!StringUtils.hasText(upperCode)) {
                    accBudgetItemPayload.setParentCode(map.get(upperName));
                }
                //设置科目级别
                setLevel(accBudgetItemPayloads, accBudgetItemPayload);
            }

            accBudgetItemDOs.add(AccBudgetItemConvert.INSTANCE.toDo(accBudgetItemPayload));
        });

        List<AccBudgetItemDO> accBudgetItemDOS = accBudgetItemDAO.saveAll(accBudgetItemDOs);
        List<AccBudgetItemDO> updataDOs = new ArrayList<>();
        accBudgetItemDOS.forEach(accBudgetItemDO -> {
            if (StringUtils.hasText(accBudgetItemDO.getParentCode())) {
                AccBudgetItemDO updataDO = accBudgetItemDOS.stream().filter(masPayload -> masPayload.getBudgetCode().equals(accBudgetItemDO.getParentCode())).findFirst().get();
                accBudgetItemDO.setParentId(updataDO.getId());
                updataDOs.add(accBudgetItemDO);
            }
        });
        //开启事务执行修改
        transactionUtilService.executeWithRunnable(() -> {
            accBudgetItemDAO.saveAll(updataDOs);
        });
        return true;
    }

    /**
     * 为科目设计级别
     *
     * @param accBudgetItemPayloads
     * @param accBudgetItemPayload
     */
    void setLevel(List<AccBudgetItemPayload> accBudgetItemPayloads, AccBudgetItemPayload accBudgetItemPayload) {
        Optional<AccBudgetItemPayload> optional = accBudgetItemPayloads.stream().filter(masPayload -> masPayload.getBudgetCode().equals(accBudgetItemPayload.getParentCode())).findFirst();
        if (optional.isPresent()) {
            AccBudgetItemPayload payload = optional.get();
            if (payload.getBudgetLevel() == null) {
                setLevel(accBudgetItemPayloads, payload);
            }
            accBudgetItemPayload.setBudgetLevel(payload.getBudgetLevel() + 1);
        } else {
            accBudgetItemPayload.setBudgetLevel(1);
        }
    }
}
