package com.elitesland.tw.tw5.server.prd.inv.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.inv.payload.InvInvoicePayload;
import com.elitesland.tw.tw5.api.prd.inv.query.InvInvoiceQuery;
import com.elitesland.tw.tw5.api.prd.inv.vo.InvInvoiceVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.inv.common.InvoiceReimStatusEnum;
import com.elitesland.tw.tw5.server.prd.inv.entity.InvInvoiceDO;
import com.elitesland.tw.tw5.server.prd.inv.entity.QInvInvoiceDO;
import com.elitesland.tw.tw5.server.prd.inv.entity.QInvInvoiceDocRefDO;
import com.elitesland.tw.tw5.server.prd.inv.repo.InvInvoiceRepo;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 发票主表
 *
 * @author zoey
 * @date 2023-12-05
 */
@Repository
@RequiredArgsConstructor
public class InvInvoiceDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final InvInvoiceRepo repo;
    private final QInvInvoiceDO qdo = QInvInvoiceDO.invInvoiceDO;
    private final QInvInvoiceDocRefDO qdoDocRefDo = QInvInvoiceDocRefDO.invInvoiceDocRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<InvInvoiceVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(InvInvoiceVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 租户ID
                qdo.tenantId,
                // 发票类型COM:INPUT_INV_TYPE
                qdo.invType,
                // 发票状态ACC:INV_STATUS
                qdo.invState,
                // 发票报销状态ACC:INV_REIMB_STATUS
                qdo.invReimStatus,
                // 归属人
                qdo.invOwnerId,
                // 发票内容根据百望识别的发票内容，例如 餐饮、住宿等，需确认识别方式
                qdo.invLtype,
                // 百望系统发票ID
                qdo.baiwangInvId,
                // 单据主键id:关联的单据主键id
                qdo.billId,
                // 文件主键id:票据信息关联的票据影像文件主键id
                qdo.fileId,
                // 是否可抵扣:是否可抵扣，0：否，1：是
                qdo.isDeduct,
                // 抵扣税额:可抵扣税额
                qdo.deductTax,
                // 百望发票类型:01-增值税专用发票，02-货物运输业增值税专用发票，03-机动车销售统一发票，04-增值税普通发票，10-增值税电子普通发票，11-增值税普通发票(卷式)，14-增值税电子普通发票（通行费），15-二手车销售统一发票，1002-火车票，1003-飞机票，1004-出租车票，1005-定额发票，1006-客运发票，99-其他发票
                qdo.invoiceType,
                // 是否删除:是否删除，0-否，1-是
                qdo.isDel,
                // 有无影像:有无影像，0-无影像，1-有影像
                qdo.hasPicture,
                // 有无坐标:有无坐标，0-无，1-有
                qdo.hasPosition,
                // 创建时间:创建时间：yyyy-MM-dd HH:mm:ss
                qdo.baiwangCreateTime,
                // 修改时间:修改时间：yyyy-MM-dd HH:mm:ss
                qdo.baiwangUpdateTime,
                // 查验时间:查验时间：yyyy-MM-dd HH:mm:ss
                qdo.inspectionTime,
                // 查验状态ACC:INV_INSPECTION_STATUS:0-待查验，1-查验一致，2-查无此票，3-不一致，4-查验失败,5-无需查验
                qdo.inspectionStatus,
                // 查验失败描述:查验失败描述
                qdo.inspectionErrorDesc,
                // 零税率标识:1:税率栏位显示“免税”， 2:税率栏位显示“不征收”， 3:零税率
                qdo.zeroTaxRateSign,
                // 百望发票状态:0：正常, 1：失控, 2：作废, 3：红冲, 4：异常
                qdo.invoiceState,
                // 校验状态:票据数据校验状态，0-校验通过，1-校验失败，2-重复发票
                qdo.verifyStatus,
                // 票据报销状态:票据使用状态 ，0-未关联，2-已关联
                qdo.reimburseStatus,
                // 归集用途类型:0：普通归集，1：审单归集，2：仅采集
                qdo.collectUseType,
                // 创建用户账号、采集人:采集人账号
                qdo.userAccount,
                // 创建用户id、采集人:采集人id
                qdo.userId,
                // 用户名:采集人姓名
                qdo.userName,
                // 组织机构id:组织结构id
                qdo.orgId,
                // 组织机构名称:组织机构名称
                qdo.orgName,
                // 所属行政区编码:所属行政区编码，增值税通用
                qdo.administrativeDivisionNo,
                // 所属行政区名称:所属行政区名称，增值税通用
                qdo.administrativeDivisionName,
                // 发票代码:发票代码，通用；火车票、行程单无代码，部分其他发票也无代码
                qdo.invoiceCode,
                // 发票号码:发票号码，通用；火车票为左上角票号，行程单为电子客票号码
                qdo.invoiceNo,
                // 开票日期:开票日期，通用，火车票对应的是乘车日期YYYY-MM-DD
                qdo.invoiceDate,
                // 购方名称:购方名称-增值税通用
                qdo.purchaserName,
                // 购方税号:“购方税号，通用；机动车-购方名称及身份证号码/组织机构代码”
                qdo.purchaserTaxNo,
                // 购方开户行账户:购方开户行账户，通用，卷票无
                qdo.purchaserBank,
                // 购方地址电话:购方地址电话，通用，卷票无
                qdo.purchaserAddressPhone,
                // 销方名称:销方名称，通用
                qdo.saleName,
                // 销方税号:销方税号，通用
                qdo.saleTaxNo,
                // 销方地址电话:销方地址电话，增值税通用，卷票无
                qdo.saleAddressPhone,
                // 销方开户行账户:销方开户行账户，增值税通用，卷票无
                qdo.saleBank,
                // 不含税金额:不含税金额：1.增值税专/普/电：不含税总金额合计；2.机动车：不含税价
                qdo.totalAmount,
                // 合计税额:合计税额：1.增值税专/普/电：税额合计；2.机动车：增值税税额；3.行程单：燃油附加费
                qdo.totalTax,
                // 价税合计:价税合计：1.增值税专/普/电/机动车：价税合计；2.二手车：车价合计；3.卷票：合计金额；4.火车票/出租车/定额/客运发票/其他发票：金额；5.行程单：合计（票价+民航发展基金+燃油附加+其他税费）
                qdo.amountTax,
                // 价税合计_中文:价税合计_中文，增值税通用
                qdo.amountTaxCn,
                // 其他税费:其他税费（行程单专有）
                qdo.otherTax,
                // 民航发展基金:民航发展基金（行程单专有）
                qdo.civilAviationFund,
                // 校验码:校验码，增值税普/电/卷票、行程单含有
                qdo.checkCode,
                // 税控盘号、机器码:税控盘号、机器码，增值税通用
                qdo.machineCode,
                // 发票密文:增值税专/普/电-密码区，机动车/二手车-税控码
                qdo.ciphertext,
                // 备注:发票备注，增值税通用
                qdo.baiwangRemark,
                // 乘车人:乘车/机人姓名-火车票/行程单/客运发票
                qdo.drawer,
                // 出发地:出发地：火车票、客运发票
                qdo.leaveCity,
                // 到站地:到达地：火车票、客运发票
                qdo.arriveCity,
                // 出发时间:出发时间：yyyy-MM-dd HH:mm，火车票的乘车时间，出租车票的上车时间
                qdo.leaveTime,
                // 到站时间:到达时间：yyyy-MM-dd HH:mm，出租车票的下车时间
                qdo.arriveTime,
                // 车次:火车票车次
                qdo.trainNumber,
                // 座位等级:火车票座位类型
                qdo.trainSeat,
                // 身份证号:身份证号（行程单）
                qdo.idNum,
                // 里程:里程（出租车）
                qdo.mileage,
                // 税率(明细最大最小税率，格式：6%~13%、13%)
                qdo.taxRate,
                // 买方单位代码,个人身份证号
                qdo.buyerIdentification,
                // 二手车市场税号
                qdo.marketTaxNo,
                // 卖方单位代码,个人身份证号
                qdo.sellerId,
                // 发票在影像中的旋转角度（顺时针）
                qdo.orientation,
                //  是否有公司印章，0：无；1：有
                qdo.hasSeal,
                // 车牌号(出租车)
                qdo.carNo,
                // 车架号/车辆识别代码(机动车、二手车)
                qdo.carCode,
                // 发动机号码(机动车)
                qdo.carEngineCode,
                // 机打号码
                qdo.machineInvoiceNo,
                // 机打代码
                qdo.machineInvoiceCode,
                // 异步查询码
                qdo.asyncCode,
                // 发票坐标对象 该发票在文件中的位置
                qdo.invoicePositionX1,
                // 发票坐标对象 该发票在文件中的位置
                qdo.invoicePositionY1,
                // 发票坐标对象 该发票在文件中的位置
                qdo.invoicePositionX2,
                // 发票坐标对象 该发票在文件中的位置
                qdo.invoicePositionY2,
                // 发票联次
                qdo.invoiceTemplateType,
                // 发票联次名称
                qdo.invoiceTemplateName,
                // 发票密文
                qdo.invoiceCiphertext,
                // 主管税务机关代码
                qdo.taxAuthoritiesCode,
                // 主管税务机关名称
                qdo.taxAuthoritiesName,
                // 厂牌型号
                qdo.carModel,
                // 合格证号
                qdo.certificateNo,
                // 二手车市场
                qdo.marketName,
                // 登记证号
                qdo.registrationNo,
                // 序列号
                qdo.serialNum,
                // 保险费
                qdo.premium,
                // 印刷序号
                qdo.printNumber,
                // 开票时间
                qdo.invoiceTime,
                // 入口
                qdo.entrance,
                // 出口
                qdo.roadExit,
                // 高速标识
                qdo.isHighway,
                // 发票明细表ID
                qdo.reimId,
                qdo.reimDId,
                // 已核销金额
                qdo.writtenOffAmt,
                // 发票照片
                qdo.imgContent,
                // 发票凭证字段
                qdo.invoiceVoucher,
                //JDE单据公司
                qdo.jdeCompany,
                //JDE单据号
                qdo.jdeDocumentNo,
                //JDE单据类型
                qdo.jdeDocumentType,
                //JDE付款项
                qdo.jdePaymentItem,
                qdo.jdeInvoiceFailReason
        )).from(qdo).leftJoin(qdoDocRefDo).on(qdo.id.eq(qdoDocRefDo.invId).and(qdoDocRefDo.deleteFlag.eq(0)));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<InvInvoiceVO> getJpaQueryWhere(InvInvoiceQuery query) {
        JPAQuery<InvInvoiceVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(InvInvoiceQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo).leftJoin(qdoDocRefDo).on(qdo.id.eq(qdoDocRefDo.invId).and(qdoDocRefDo.deleteFlag.eq(0)));
        ;
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(InvInvoiceQuery query) {
        List<Predicate> list = new ArrayList<>();
        // 已核销金额要小于发票金额
        if (query.getWrittenOffFlag() == 1) {
//            if(query.getWrittenOffFlag()==1 && !ObjectUtils.isEmpty(query.getWrittenOffAmt())){
            list.add(qdo.writtenOffAmt.isNull().or(qdo.writtenOffAmt.lt(qdo.amountTax)));
        } else {
            list.add(qdo.writtenOffAmt.isNull().or(qdo.writtenOffAmt.eq(BigDecimal.ZERO)));
        }
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 发票类型COM:INPUT_INV_TYPE 精确 */
        if (!ObjectUtils.isEmpty(query.getInvType())) {
            list.add(qdo.invType.eq(query.getInvType()));
        }
        /** 发票状态ACC:INV_STATUS 精确 */
        if (!ObjectUtils.isEmpty(query.getInvState())) {
            list.add(qdo.invState.eq(query.getInvState()));
        }
        /** 发票状态ACC:INV_STATUS 精确 */
        if (!ObjectUtils.isEmpty(query.getInvStateNot())) {
            list.add(qdo.invState.ne(query.getInvStateNot()));
        }

        /** 发票报销状态ACC:INV_REIMB_STATUS 精确 */
        if (!ObjectUtils.isEmpty(query.getInvReimStatus())) {
            list.add(qdo.invReimStatus.eq(query.getInvReimStatus()));
        }
        /** 归属人 精确 */
        if (!ObjectUtils.isEmpty(query.getInvOwnerId())) {
            list.add(qdo.invOwnerId.eq(query.getInvOwnerId()).or(qdo.createUserId.eq(query.getInvOwnerId())));
        }
        /** 发票内容根据百望识别的发票内容，例如 餐饮、住宿等，需确认识别方式 精确 */
        if (!ObjectUtils.isEmpty(query.getInvLtype())) {
            list.add(qdo.invLtype.eq(query.getInvLtype()));
        }
        /** 百望系统发票ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBaiwangInvId())) {
            list.add(qdo.baiwangInvId.eq(query.getBaiwangInvId()));
        }
        /** 单据主键id:关联的单据主键id 精确 */
        if (!ObjectUtils.isEmpty(query.getBillId())) {
            list.add(qdo.billId.eq(query.getBillId()));
        }
        /** 文件主键id:票据信息关联的票据影像文件主键id 精确 */
        if (!ObjectUtils.isEmpty(query.getFileId())) {
            list.add(qdo.fileId.eq(query.getFileId()));
        }
        /** 是否可抵扣:是否可抵扣，0：否，1：是 精确 */
        if (!ObjectUtils.isEmpty(query.getIsDeduct())) {
            list.add(qdo.isDeduct.eq(query.getIsDeduct()));
        }
        /** 抵扣税额:可抵扣税额 精确 */
        if (!ObjectUtils.isEmpty(query.getDeductTax())) {
            list.add(qdo.deductTax.eq(query.getDeductTax()));
        }
        /** 百望发票类型:01-增值税专用发票，02-货物运输业增值税专用发票，03-机动车销售统一发票，04-增值税普通发票，10-增值税电子普通发票，11-增值税普通发票(卷式)，14-增值税电子普通发票（通行费），15-二手车销售统一发票，1002-火车票，1003-飞机票，1004-出租车票，1005-定额发票，1006-客运发票，99-其他发票 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceType())) {
            list.add(qdo.invoiceType.eq(query.getInvoiceType()));
        }
        /** 是否删除:是否删除，0-否，1-是 精确 */
        if (!ObjectUtils.isEmpty(query.getIsDel())) {
            list.add(qdo.isDel.eq(query.getIsDel()));
        }
        /** 有无影像:有无影像，0-无影像，1-有影像 精确 */
        if (!ObjectUtils.isEmpty(query.getHasPicture())) {
            list.add(qdo.hasPicture.eq(query.getHasPicture()));
        }
        /** 有无坐标:有无坐标，0-无，1-有 精确 */
        if (!ObjectUtils.isEmpty(query.getHasPosition())) {
            list.add(qdo.hasPosition.eq(query.getHasPosition()));
        }
        /** 创建时间:创建时间：yyyy-MM-dd HH:mm:ss 精确 */
        if (!ObjectUtils.isEmpty(query.getBaiwangCreateTime())) {
            list.add(qdo.baiwangCreateTime.eq(query.getBaiwangCreateTime()));
        }
        /** 修改时间:修改时间：yyyy-MM-dd HH:mm:ss 精确 */
        if (!ObjectUtils.isEmpty(query.getBaiwangUpdateTime())) {
            list.add(qdo.baiwangUpdateTime.eq(query.getBaiwangUpdateTime()));
        }
        /** 查验时间:查验时间：yyyy-MM-dd HH:mm:ss 精确 */
        if (!ObjectUtils.isEmpty(query.getInspectionTime())) {
            list.add(qdo.inspectionTime.eq(query.getInspectionTime()));
        }
        /** 查验状态ACC:INV_INSPECTION_STATUS:0-待查验，1-查验一致，2-查无此票，3-不一致，4-查验失败,5-无需查验 精确 */
        if (!ObjectUtils.isEmpty(query.getInspectionStatus())) {
            list.add(qdo.inspectionStatus.eq(query.getInspectionStatus()));
        }
        /** 查验失败描述:查验失败描述 精确 */
        if (!ObjectUtils.isEmpty(query.getInspectionErrorDesc())) {
            list.add(qdo.inspectionErrorDesc.eq(query.getInspectionErrorDesc()));
        }
        /** 零税率标识:1:税率栏位显示“免税”， 2:税率栏位显示“不征收”， 3:零税率 精确 */
        if (!ObjectUtils.isEmpty(query.getZeroTaxRateSign())) {
            list.add(qdo.zeroTaxRateSign.eq(query.getZeroTaxRateSign()));
        }
        /** 百望发票状态:0：正常, 1：失控, 2：作废, 3：红冲, 4：异常 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceState())) {
            list.add(qdo.invoiceState.eq(query.getInvoiceState()));
        }
        /** 校验状态:票据数据校验状态，0-校验通过，1-校验失败，2-重复发票 精确 */
        if (!ObjectUtils.isEmpty(query.getVerifyStatus())) {
            list.add(qdo.verifyStatus.eq(query.getVerifyStatus()));
        }
        /** 票据报销状态:票据使用状态 ，0-未关联，2-已关联 精确 */
        if (!ObjectUtils.isEmpty(query.getReimburseStatus())) {
            list.add(qdo.reimburseStatus.eq(query.getReimburseStatus()));
        }
        /** 归集用途类型:0：普通归集，1：审单归集，2：仅采集 精确 */
        if (!ObjectUtils.isEmpty(query.getCollectUseType())) {
            list.add(qdo.collectUseType.eq(query.getCollectUseType()));
        }
        /** 创建用户账号、采集人:采集人账号 精确 */
        if (!ObjectUtils.isEmpty(query.getUserAccount())) {
            list.add(qdo.userAccount.eq(query.getUserAccount()));
        }
        /** 创建用户id、采集人:采集人id 精确 */
        if (!ObjectUtils.isEmpty(query.getUserId())) {
            list.add(qdo.userId.eq(query.getUserId()));
        }
        /** 创建用户id、采集人:采集人id 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            list.add(qdo.createUserId.eq(query.getCreateUserId()));
        }
        /** 用户名:采集人姓名 精确 */
        if (!ObjectUtils.isEmpty(query.getUserName())) {
            list.add(qdo.userName.eq(query.getUserName()));
        }
        /** 组织机构id:组织结构id 精确 */
        if (!ObjectUtils.isEmpty(query.getOrgId())) {
            list.add(qdo.orgId.eq(query.getOrgId()));
        }
        /** 组织机构名称:组织机构名称 精确 */
        if (!ObjectUtils.isEmpty(query.getOrgName())) {
            list.add(qdo.orgName.eq(query.getOrgName()));
        }
        /** 所属行政区编码:所属行政区编码，增值税通用 精确 */
        if (!ObjectUtils.isEmpty(query.getAdministrativeDivisionNo())) {
            list.add(qdo.administrativeDivisionNo.eq(query.getAdministrativeDivisionNo()));
        }
        /** 所属行政区名称:所属行政区名称，增值税通用 精确 */
        if (!ObjectUtils.isEmpty(query.getAdministrativeDivisionName())) {
            list.add(qdo.administrativeDivisionName.eq(query.getAdministrativeDivisionName()));
        }
        /** 发票代码:发票代码，通用；火车票、行程单无代码，部分其他发票也无代码 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceCode())) {
            list.add(qdo.invoiceCode.eq(query.getInvoiceCode()));
        }
        /** 发票号码:发票号码，通用；火车票为左上角票号，行程单为电子客票号码 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceNo())) {
            list.add(qdo.invoiceNo.like(SqlUtil.toSqlLikeString(query.getInvoiceNo())));
        }

        /** 开票日期:开票日期，通用，火车票对应的是乘车日期YYYY-MM-DD 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceDate())) {
            list.add(qdo.invoiceDate.between(query.getInvoiceDate().get(0), query.getInvoiceDate().get(1)));
        }

        /** 购方名称:购方名称-增值税通用 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaserName())) {
            list.add(qdo.purchaserName.like(SqlUtil.toSqlLikeString(query.getPurchaserName())));
        }
        /** 购方税号:“购方税号，通用；机动车-购方名称及身份证号码/组织机构代码” 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaserTaxNo())) {
            list.add(qdo.purchaserTaxNo.eq(query.getPurchaserTaxNo()));
        }
        /** 购方开户行账户:购方开户行账户，通用，卷票无 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaserBank())) {
            list.add(qdo.purchaserBank.eq(query.getPurchaserBank()));
        }
        /** 购方地址电话:购方地址电话，通用，卷票无 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaserAddressPhone())) {
            list.add(qdo.purchaserAddressPhone.eq(query.getPurchaserAddressPhone()));
        }
        /** 销方名称:销方名称，通用 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleName())) {
            list.add(qdo.saleName.like(SqlUtil.toSqlLikeString(query.getSaleName())));
        }
        /** 销方税号:销方税号，通用 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleTaxNo())) {
            list.add(qdo.saleTaxNo.eq(query.getSaleTaxNo()));
        }
        /** 销方地址电话:销方地址电话，增值税通用，卷票无 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleAddressPhone())) {
            list.add(qdo.saleAddressPhone.eq(query.getSaleAddressPhone()));
        }
        /** 销方开户行账户:销方开户行账户，增值税通用，卷票无 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleBank())) {
            list.add(qdo.saleBank.eq(query.getSaleBank()));
        }
        /** 不含税金额:不含税金额：1.增值税专/普/电：不含税总金额合计；2.机动车：不含税价 精确 */
        if (!ObjectUtils.isEmpty(query.getTotalAmount())) {
            list.add(qdo.totalAmount.eq(query.getTotalAmount()));
        }
        /** 合计税额:合计税额：1.增值税专/普/电：税额合计；2.机动车：增值税税额；3.行程单：燃油附加费 精确 */
        if (!ObjectUtils.isEmpty(query.getTotalTax())) {
            list.add(qdo.totalTax.eq(query.getTotalTax()));
        }
        /** 价税合计:价税合计：1.增值税专/普/电/机动车：价税合计；2.二手车：车价合计；3.卷票：合计金额；4.火车票/出租车/定额/客运发票/其他发票：金额；5.行程单：合计（票价+民航发展基金+燃油附加+其他税费） 精确 */
        if (!ObjectUtils.isEmpty(query.getAmountTax())) {
            list.add(qdo.amountTax.eq(query.getAmountTax()));
        }
        /** 价税合计_中文:价税合计_中文，增值税通用 精确 */
        if (!ObjectUtils.isEmpty(query.getAmountTaxCn())) {
            list.add(qdo.amountTaxCn.eq(query.getAmountTaxCn()));
        }
        /** 其他税费:其他税费（行程单专有） 精确 */
        if (!ObjectUtils.isEmpty(query.getOtherTax())) {
            list.add(qdo.otherTax.eq(query.getOtherTax()));
        }
        /** 民航发展基金:民航发展基金（行程单专有） 精确 */
        if (!ObjectUtils.isEmpty(query.getCivilAviationFund())) {
            list.add(qdo.civilAviationFund.eq(query.getCivilAviationFund()));
        }
        /** 校验码:校验码，增值税普/电/卷票、行程单含有 精确 */
        if (!ObjectUtils.isEmpty(query.getCheckCode())) {
            list.add(qdo.checkCode.eq(query.getCheckCode()));
        }
        /** 税控盘号、机器码:税控盘号、机器码，增值税通用 精确 */
        if (!ObjectUtils.isEmpty(query.getMachineCode())) {
            list.add(qdo.machineCode.eq(query.getMachineCode()));
        }
        /** 发票密文:增值税专/普/电-密码区，机动车/二手车-税控码 精确 */
        if (!ObjectUtils.isEmpty(query.getCiphertext())) {
            list.add(qdo.ciphertext.eq(query.getCiphertext()));
        }
        /** 备注:发票备注，增值税通用 精确 */
        if (!ObjectUtils.isEmpty(query.getBaiwangRemark())) {
            list.add(qdo.baiwangRemark.eq(query.getBaiwangRemark()));
        }
        /** 乘车人:乘车/机人姓名-火车票/行程单/客运发票 精确 */
        if (!ObjectUtils.isEmpty(query.getDrawer())) {
            list.add(qdo.drawer.eq(query.getDrawer()));
        }
        /** 出发地:出发地：火车票、客运发票 精确 */
        if (!ObjectUtils.isEmpty(query.getLeaveCity())) {
            list.add(qdo.leaveCity.eq(query.getLeaveCity()));
        }
        /** 到站地:到达地：火车票、客运发票 精确 */
        if (!ObjectUtils.isEmpty(query.getArriveCity())) {
            list.add(qdo.arriveCity.eq(query.getArriveCity()));
        }
        /** 出发时间:出发时间：yyyy-MM-dd HH:mm，火车票的乘车时间，出租车票的上车时间 精确 */
        if (!ObjectUtils.isEmpty(query.getLeaveTime())) {
            list.add(qdo.leaveTime.eq(query.getLeaveTime()));
        }
        /** 到站时间:到达时间：yyyy-MM-dd HH:mm，出租车票的下车时间 精确 */
        if (!ObjectUtils.isEmpty(query.getArriveTime())) {
            list.add(qdo.arriveTime.eq(query.getArriveTime()));
        }
        /** 车次:火车票车次 精确 */
        if (!ObjectUtils.isEmpty(query.getTrainNumber())) {
            list.add(qdo.trainNumber.eq(query.getTrainNumber()));
        }
        /** 座位等级:火车票座位类型 精确 */
        if (!ObjectUtils.isEmpty(query.getTrainSeat())) {
            list.add(qdo.trainSeat.eq(query.getTrainSeat()));
        }
        /** 身份证号:身份证号（行程单） 精确 */
        if (!ObjectUtils.isEmpty(query.getIdNum())) {
            list.add(qdo.idNum.eq(query.getIdNum()));
        }
        /** 里程:里程（出租车） 精确 */
        if (!ObjectUtils.isEmpty(query.getMileage())) {
            list.add(qdo.mileage.eq(query.getMileage()));
        }
        /** 税率(明细最大最小税率，格式：6%~13%、13%) 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxRate())) {
            list.add(qdo.taxRate.eq(query.getTaxRate()));
        }
        /** 买方单位代码,个人身份证号 精确 */
        if (!ObjectUtils.isEmpty(query.getBuyerIdentification())) {
            list.add(qdo.buyerIdentification.eq(query.getBuyerIdentification()));
        }
        /** 二手车市场税号 精确 */
        if (!ObjectUtils.isEmpty(query.getMarketTaxNo())) {
            list.add(qdo.marketTaxNo.eq(query.getMarketTaxNo()));
        }
        /** 卖方单位代码,个人身份证号 精确 */
        if (!ObjectUtils.isEmpty(query.getSellerId())) {
            list.add(qdo.sellerId.eq(query.getSellerId()));
        }
        /** 发票在影像中的旋转角度（顺时针） 精确 */
        if (!ObjectUtils.isEmpty(query.getOrientation())) {
            list.add(qdo.orientation.eq(query.getOrientation()));
        }
        /**  是否有公司印章，0：无；1：有 精确 */
        if (!ObjectUtils.isEmpty(query.getHasSeal())) {
            list.add(qdo.hasSeal.eq(query.getHasSeal()));
        }
        /** 车牌号(出租车) 精确 */
        if (!ObjectUtils.isEmpty(query.getCarNo())) {
            list.add(qdo.carNo.eq(query.getCarNo()));
        }
        /** 车架号/车辆识别代码(机动车、二手车) 精确 */
        if (!ObjectUtils.isEmpty(query.getCarCode())) {
            list.add(qdo.carCode.eq(query.getCarCode()));
        }
        /** 发动机号码(机动车) 精确 */
        if (!ObjectUtils.isEmpty(query.getCarEngineCode())) {
            list.add(qdo.carEngineCode.eq(query.getCarEngineCode()));
        }
        /** 机打号码 精确 */
        if (!ObjectUtils.isEmpty(query.getMachineInvoiceNo())) {
            list.add(qdo.machineInvoiceNo.eq(query.getMachineInvoiceNo()));
        }
        /** 机打代码 精确 */
        if (!ObjectUtils.isEmpty(query.getMachineInvoiceCode())) {
            list.add(qdo.machineInvoiceCode.eq(query.getMachineInvoiceCode()));
        }
        /** 异步查询码 精确 */
        if (!ObjectUtils.isEmpty(query.getAsyncCode())) {
            list.add(qdo.asyncCode.eq(query.getAsyncCode()));
        }
        /** 发票坐标对象 该发票在文件中的位置 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoicePositionX1())) {
            list.add(qdo.invoicePositionX1.eq(query.getInvoicePositionX1()));
        }
        /** 发票坐标对象 该发票在文件中的位置 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoicePositionY1())) {
            list.add(qdo.invoicePositionY1.eq(query.getInvoicePositionY1()));
        }
        /** 发票坐标对象 该发票在文件中的位置 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoicePositionX2())) {
            list.add(qdo.invoicePositionX2.eq(query.getInvoicePositionX2()));
        }
        /** 发票坐标对象 该发票在文件中的位置 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoicePositionY2())) {
            list.add(qdo.invoicePositionY2.eq(query.getInvoicePositionY2()));
        }
        /** 发票联次 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceTemplateType())) {
            list.add(qdo.invoiceTemplateType.eq(query.getInvoiceTemplateType()));
        }
        /** 发票联次名称 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceTemplateName())) {
            list.add(qdo.invoiceTemplateName.eq(query.getInvoiceTemplateName()));
        }
        /** 发票密文 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceCiphertext())) {
            list.add(qdo.invoiceCiphertext.eq(query.getInvoiceCiphertext()));
        }
        /** 主管税务机关代码 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxAuthoritiesCode())) {
            list.add(qdo.taxAuthoritiesCode.eq(query.getTaxAuthoritiesCode()));
        }
        /** 主管税务机关名称 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxAuthoritiesName())) {
            list.add(qdo.taxAuthoritiesName.eq(query.getTaxAuthoritiesName()));
        }
        /** 厂牌型号 精确 */
        if (!ObjectUtils.isEmpty(query.getCarModel())) {
            list.add(qdo.carModel.eq(query.getCarModel()));
        }
        /** 合格证号 精确 */
        if (!ObjectUtils.isEmpty(query.getCertificateNo())) {
            list.add(qdo.certificateNo.eq(query.getCertificateNo()));
        }
        /** 二手车市场 精确 */
        if (!ObjectUtils.isEmpty(query.getMarketName())) {
            list.add(qdo.marketName.eq(query.getMarketName()));
        }
        /** 登记证号 精确 */
        if (!ObjectUtils.isEmpty(query.getRegistrationNo())) {
            list.add(qdo.registrationNo.eq(query.getRegistrationNo()));
        }
        /** 序列号 精确 */
        if (!ObjectUtils.isEmpty(query.getSerialNum())) {
            list.add(qdo.serialNum.eq(query.getSerialNum()));
        }
        /** 保险费 精确 */
        if (!ObjectUtils.isEmpty(query.getPremium())) {
            list.add(qdo.premium.eq(query.getPremium()));
        }
        /** 印刷序号 精确 */
        if (!ObjectUtils.isEmpty(query.getPrintNumber())) {
            list.add(qdo.printNumber.eq(query.getPrintNumber()));
        }
        /** 开票时间 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceTime())) {
            list.add(qdo.invoiceTime.eq(query.getInvoiceTime()));
        }
        /** 入口 精确 */
        if (!ObjectUtils.isEmpty(query.getEntrance())) {
            list.add(qdo.entrance.eq(query.getEntrance()));
        }
        /** 出口 精确 */
        if (!ObjectUtils.isEmpty(query.getRoadExit())) {
            list.add(qdo.roadExit.eq(query.getRoadExit()));
        }
        /** 高速标识 精确 */
        if (!ObjectUtils.isEmpty(query.getIsHighway())) {
            list.add(qdo.isHighway.eq(query.getIsHighway()));
        }
        if (!ObjectUtils.isEmpty(query.getReimId())) {
            list.add(qdo.reimId.eq(query.getReimId()));
        }
        /** 发票明细表ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReimDId())) {
            list.add(qdo.reimDId.eq(query.getReimDId()));
        }
        /** 报销明细ID集合 */
        if (!ObjectUtils.isEmpty(query.getReimDIdList())) {
            list.add(qdo.reimDId.in(query.getReimDIdList()));
        }
        /** 已核销金额 精确 */
        if (!ObjectUtils.isEmpty(query.getWrittenOffAmt())) {
            list.add(qdo.writtenOffAmt.eq(query.getWrittenOffAmt()));
        }
        /** 发票照片 精确 */
        if (!ObjectUtils.isEmpty(query.getImgContent())) {
            list.add(qdo.imgContent.eq(query.getImgContent()));
        }
        //ID集合
        if (!ObjectUtils.isEmpty(query.getIdList())) {
            list.add(qdo.id.in(query.getIdList()));
        }

        if (!CollectionUtils.isEmpty(query.getInvoiceNoIn())) {
            list.add(qdo.invoiceNo.in(query.getInvoiceNoIn()));
        }

        if (!ObjectUtils.isEmpty(query.getInvoiceDateGt())) {
            list.add(qdo.invoiceDate.gt(query.getInvoiceDateGt().toString()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public InvInvoiceVO queryByKey(Long id) {
        JPAQuery<InvInvoiceVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<InvInvoiceVO> queryListDynamic(InvInvoiceQuery query) {
        JPAQuery<InvInvoiceVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<InvInvoiceVO> queryPaging(InvInvoiceQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<InvInvoiceVO> jpaQuery = getJpaQueryWhere(query);
        List<InvInvoiceVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<InvInvoiceVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public InvInvoiceDO save(InvInvoiceDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<InvInvoiceDO> saveAll(List<InvInvoiceDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(InvInvoicePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 发票类型COM:INPUT_INV_TYPE
        if (payload.getInvType() != null) {
            update.set(qdo.invType, payload.getInvType());
        }
        // 发票状态ACC:INV_STATUS
        if (payload.getInvState() != null) {
            update.set(qdo.invState, payload.getInvState());
        }
        // 发票报销状态ACC:INV_REIMB_STATUS
        if (payload.getInvReimStatus() != null) {
            update.set(qdo.invReimStatus, payload.getInvReimStatus());
        }
        // 归属人
        if (payload.getInvOwnerId() != null) {
            update.set(qdo.invOwnerId, payload.getInvOwnerId());
        }
        // 发票内容根据百望识别的发票内容，例如 餐饮、住宿等，需确认识别方式
        if (payload.getInvLtype() != null) {
            update.set(qdo.invLtype, payload.getInvLtype());
        }
        // 百望系统发票ID
        if (payload.getBaiwangInvId() != null) {
            update.set(qdo.baiwangInvId, payload.getBaiwangInvId());
        }
        // 单据主键id:关联的单据主键id
        if (payload.getBillId() != null) {
            update.set(qdo.billId, payload.getBillId());
        }
        // 文件主键id:票据信息关联的票据影像文件主键id
        if (payload.getFileId() != null) {
            update.set(qdo.fileId, payload.getFileId());
        }
        // 是否可抵扣:是否可抵扣，0：否，1：是
        if (payload.getIsDeduct() != null) {
            update.set(qdo.isDeduct, payload.getIsDeduct());
        }
        // 抵扣税额:可抵扣税额
        if (payload.getDeductTax() != null) {
            update.set(qdo.deductTax, payload.getDeductTax());
        }
        // 百望发票类型:01-增值税专用发票，02-货物运输业增值税专用发票，03-机动车销售统一发票，04-增值税普通发票，10-增值税电子普通发票，11-增值税普通发票(卷式)，14-增值税电子普通发票（通行费），15-二手车销售统一发票，1002-火车票，1003-飞机票，1004-出租车票，1005-定额发票，1006-客运发票，99-其他发票
        if (payload.getInvoiceType() != null) {
            update.set(qdo.invoiceType, payload.getInvoiceType());
        }
        // 是否删除:是否删除，0-否，1-是
        if (payload.getIsDel() != null) {
            update.set(qdo.isDel, payload.getIsDel());
        }
        // 有无影像:有无影像，0-无影像，1-有影像
        if (payload.getHasPicture() != null) {
            update.set(qdo.hasPicture, payload.getHasPicture());
        }
        // 有无坐标:有无坐标，0-无，1-有
        if (payload.getHasPosition() != null) {
            update.set(qdo.hasPosition, payload.getHasPosition());
        }
        // 创建时间:创建时间：yyyy-MM-dd HH:mm:ss
        if (payload.getBaiwangCreateTime() != null) {
            update.set(qdo.baiwangCreateTime, payload.getBaiwangCreateTime());
        }
        // 修改时间:修改时间：yyyy-MM-dd HH:mm:ss
        if (payload.getBaiwangUpdateTime() != null) {
            update.set(qdo.baiwangUpdateTime, payload.getBaiwangUpdateTime());
        }
        // 查验时间:查验时间：yyyy-MM-dd HH:mm:ss
        if (payload.getInspectionTime() != null) {
            update.set(qdo.inspectionTime, payload.getInspectionTime());
        }
        // 查验状态ACC:INV_INSPECTION_STATUS:0-待查验，1-查验一致，2-查无此票，3-不一致，4-查验失败,5-无需查验
        if (payload.getInspectionStatus() != null) {
            update.set(qdo.inspectionStatus, payload.getInspectionStatus());
        }
        // 查验失败描述:查验失败描述
        if (payload.getInspectionErrorDesc() != null) {
            update.set(qdo.inspectionErrorDesc, payload.getInspectionErrorDesc());
        }
        // 零税率标识:1:税率栏位显示“免税”， 2:税率栏位显示“不征收”， 3:零税率
        if (payload.getZeroTaxRateSign() != null) {
            update.set(qdo.zeroTaxRateSign, payload.getZeroTaxRateSign());
        }
        // 百望发票状态:0：正常, 1：失控, 2：作废, 3：红冲, 4：异常
        if (payload.getInvoiceState() != null) {
            update.set(qdo.invoiceState, payload.getInvoiceState());
        }
        // 校验状态:票据数据校验状态，0-校验通过，1-校验失败，2-重复发票
        if (payload.getVerifyStatus() != null) {
            update.set(qdo.verifyStatus, payload.getVerifyStatus());
        }
        // 票据报销状态:票据使用状态 ，0-未关联，2-已关联
        if (payload.getReimburseStatus() != null) {
            update.set(qdo.reimburseStatus, payload.getReimburseStatus());
        }
        // 归集用途类型:0：普通归集，1：审单归集，2：仅采集
        if (payload.getCollectUseType() != null) {
            update.set(qdo.collectUseType, payload.getCollectUseType());
        }
        // 创建用户账号、采集人:采集人账号
        if (payload.getUserAccount() != null) {
            update.set(qdo.userAccount, payload.getUserAccount());
        }
        // 创建用户id、采集人:采集人id
        if (payload.getUserId() != null) {
            update.set(qdo.userId, payload.getUserId());
        }
        // 用户名:采集人姓名
        if (payload.getUserName() != null) {
            update.set(qdo.userName, payload.getUserName());
        }
        // 组织机构id:组织结构id
        if (payload.getOrgId() != null) {
            update.set(qdo.orgId, payload.getOrgId());
        }
        // 组织机构名称:组织机构名称
        if (payload.getOrgName() != null) {
            update.set(qdo.orgName, payload.getOrgName());
        }
        // 所属行政区编码:所属行政区编码，增值税通用
        if (payload.getAdministrativeDivisionNo() != null) {
            update.set(qdo.administrativeDivisionNo, payload.getAdministrativeDivisionNo());
        }
        // 所属行政区名称:所属行政区名称，增值税通用
        if (payload.getAdministrativeDivisionName() != null) {
            update.set(qdo.administrativeDivisionName, payload.getAdministrativeDivisionName());
        }
        // 发票代码:发票代码，通用；火车票、行程单无代码，部分其他发票也无代码
        if (payload.getInvoiceCode() != null) {
            update.set(qdo.invoiceCode, payload.getInvoiceCode());
        }
        // 发票号码:发票号码，通用；火车票为左上角票号，行程单为电子客票号码
        if (payload.getInvoiceNo() != null) {
            update.set(qdo.invoiceNo, payload.getInvoiceNo());
        }
        // 开票日期:开票日期，通用，火车票对应的是乘车日期YYYY-MM-DD
        if (payload.getInvoiceDate() != null) {
            update.set(qdo.invoiceDate, payload.getInvoiceDate());
        }
        // 购方名称:购方名称-增值税通用
        if (payload.getPurchaserName() != null) {
            update.set(qdo.purchaserName, payload.getPurchaserName());
        }
        // 购方税号:“购方税号，通用；机动车-购方名称及身份证号码/组织机构代码”
        if (payload.getPurchaserTaxNo() != null) {
            update.set(qdo.purchaserTaxNo, payload.getPurchaserTaxNo());
        }
        // 购方开户行账户:购方开户行账户，通用，卷票无
        if (payload.getPurchaserBank() != null) {
            update.set(qdo.purchaserBank, payload.getPurchaserBank());
        }
        // 购方地址电话:购方地址电话，通用，卷票无
        if (payload.getPurchaserAddressPhone() != null) {
            update.set(qdo.purchaserAddressPhone, payload.getPurchaserAddressPhone());
        }
        // 销方名称:销方名称，通用
        if (payload.getSaleName() != null) {
            update.set(qdo.saleName, payload.getSaleName());
        }
        // 销方税号:销方税号，通用
        if (payload.getSaleTaxNo() != null) {
            update.set(qdo.saleTaxNo, payload.getSaleTaxNo());
        }
        // 销方地址电话:销方地址电话，增值税通用，卷票无
        if (payload.getSaleAddressPhone() != null) {
            update.set(qdo.saleAddressPhone, payload.getSaleAddressPhone());
        }
        // 销方开户行账户:销方开户行账户，增值税通用，卷票无
        if (payload.getSaleBank() != null) {
            update.set(qdo.saleBank, payload.getSaleBank());
        }
        // 不含税金额:不含税金额：1.增值税专/普/电：不含税总金额合计；2.机动车：不含税价
        if (payload.getTotalAmount() != null) {
            update.set(qdo.totalAmount, payload.getTotalAmount());
        }
        // 合计税额:合计税额：1.增值税专/普/电：税额合计；2.机动车：增值税税额；3.行程单：燃油附加费
        if (payload.getTotalTax() != null) {
            update.set(qdo.totalTax, payload.getTotalTax());
        }
        // 价税合计:价税合计：1.增值税专/普/电/机动车：价税合计；2.二手车：车价合计；3.卷票：合计金额；4.火车票/出租车/定额/客运发票/其他发票：金额；5.行程单：合计（票价+民航发展基金+燃油附加+其他税费）
        if (payload.getAmountTax() != null) {
            update.set(qdo.amountTax, payload.getAmountTax());
        }
        // 价税合计_中文:价税合计_中文，增值税通用
        if (payload.getAmountTaxCn() != null) {
            update.set(qdo.amountTaxCn, payload.getAmountTaxCn());
        }
        // 其他税费:其他税费（行程单专有）
        if (payload.getOtherTax() != null) {
            update.set(qdo.otherTax, payload.getOtherTax());
        }
        // 民航发展基金:民航发展基金（行程单专有）
        if (payload.getCivilAviationFund() != null) {
            update.set(qdo.civilAviationFund, payload.getCivilAviationFund());
        }
        // 校验码:校验码，增值税普/电/卷票、行程单含有
        if (payload.getCheckCode() != null) {
            update.set(qdo.checkCode, payload.getCheckCode());
        }
        // 税控盘号、机器码:税控盘号、机器码，增值税通用
        if (payload.getMachineCode() != null) {
            update.set(qdo.machineCode, payload.getMachineCode());
        }
        // 发票密文:增值税专/普/电-密码区，机动车/二手车-税控码
        if (payload.getCiphertext() != null) {
            update.set(qdo.ciphertext, payload.getCiphertext());
        }
        // 备注:发票备注，增值税通用
        if (payload.getBaiwangRemark() != null) {
            update.set(qdo.baiwangRemark, payload.getBaiwangRemark());
        }
        // 乘车人:乘车/机人姓名-火车票/行程单/客运发票
        if (payload.getDrawer() != null) {
            update.set(qdo.drawer, payload.getDrawer());
        }
        // 出发地:出发地：火车票、客运发票
        if (payload.getLeaveCity() != null) {
            update.set(qdo.leaveCity, payload.getLeaveCity());
        }
        // 到站地:到达地：火车票、客运发票
        if (payload.getArriveCity() != null) {
            update.set(qdo.arriveCity, payload.getArriveCity());
        }
        // 出发时间:出发时间：yyyy-MM-dd HH:mm，火车票的乘车时间，出租车票的上车时间
        if (payload.getLeaveTime() != null) {
            update.set(qdo.leaveTime, payload.getLeaveTime());
        }
        // 到站时间:到达时间：yyyy-MM-dd HH:mm，出租车票的下车时间
        if (payload.getArriveTime() != null) {
            update.set(qdo.arriveTime, payload.getArriveTime());
        }
        // 车次:火车票车次
        if (payload.getTrainNumber() != null) {
            update.set(qdo.trainNumber, payload.getTrainNumber());
        }
        // 座位等级:火车票座位类型
        if (payload.getTrainSeat() != null) {
            update.set(qdo.trainSeat, payload.getTrainSeat());
        }
        // 身份证号:身份证号（行程单）
        if (payload.getIdNum() != null) {
            update.set(qdo.idNum, payload.getIdNum());
        }
        // 里程:里程（出租车）
        if (payload.getMileage() != null) {
            update.set(qdo.mileage, payload.getMileage());
        }
        // 税率(明细最大最小税率，格式：6%~13%、13%)
        if (payload.getTaxRate() != null) {
            update.set(qdo.taxRate, payload.getTaxRate());
        }
        // 买方单位代码,个人身份证号
        if (payload.getBuyerIdentification() != null) {
            update.set(qdo.buyerIdentification, payload.getBuyerIdentification());
        }
        // 二手车市场税号
        if (payload.getMarketTaxNo() != null) {
            update.set(qdo.marketTaxNo, payload.getMarketTaxNo());
        }
        // 卖方单位代码,个人身份证号
        if (payload.getSellerId() != null) {
            update.set(qdo.sellerId, payload.getSellerId());
        }
        // 发票在影像中的旋转角度（顺时针）
        if (payload.getOrientation() != null) {
            update.set(qdo.orientation, payload.getOrientation());
        }
        //  是否有公司印章，0：无；1：有
        if (payload.getHasSeal() != null) {
            update.set(qdo.hasSeal, payload.getHasSeal());
        }
        // 车牌号(出租车)
        if (payload.getCarNo() != null) {
            update.set(qdo.carNo, payload.getCarNo());
        }
        // 车架号/车辆识别代码(机动车、二手车)
        if (payload.getCarCode() != null) {
            update.set(qdo.carCode, payload.getCarCode());
        }
        // 发动机号码(机动车)
        if (payload.getCarEngineCode() != null) {
            update.set(qdo.carEngineCode, payload.getCarEngineCode());
        }
        // 机打号码
        if (payload.getMachineInvoiceNo() != null) {
            update.set(qdo.machineInvoiceNo, payload.getMachineInvoiceNo());
        }
        // 机打代码
        if (payload.getMachineInvoiceCode() != null) {
            update.set(qdo.machineInvoiceCode, payload.getMachineInvoiceCode());
        }
        // 异步查询码
        if (payload.getAsyncCode() != null) {
            update.set(qdo.asyncCode, payload.getAsyncCode());
        }
        // 发票坐标对象 该发票在文件中的位置
        if (payload.getInvoicePositionX1() != null) {
            update.set(qdo.invoicePositionX1, payload.getInvoicePositionX1());
        }
        // 发票坐标对象 该发票在文件中的位置
        if (payload.getInvoicePositionY1() != null) {
            update.set(qdo.invoicePositionY1, payload.getInvoicePositionY1());
        }
        // 发票坐标对象 该发票在文件中的位置
        if (payload.getInvoicePositionX2() != null) {
            update.set(qdo.invoicePositionX2, payload.getInvoicePositionX2());
        }
        // 发票坐标对象 该发票在文件中的位置
        if (payload.getInvoicePositionY2() != null) {
            update.set(qdo.invoicePositionY2, payload.getInvoicePositionY2());
        }
        // 发票联次
        if (payload.getInvoiceTemplateType() != null) {
            update.set(qdo.invoiceTemplateType, payload.getInvoiceTemplateType());
        }
        // 发票联次名称
        if (payload.getInvoiceTemplateName() != null) {
            update.set(qdo.invoiceTemplateName, payload.getInvoiceTemplateName());
        }
        // 发票密文
        if (payload.getInvoiceCiphertext() != null) {
            update.set(qdo.invoiceCiphertext, payload.getInvoiceCiphertext());
        }
        // 主管税务机关代码
        if (payload.getTaxAuthoritiesCode() != null) {
            update.set(qdo.taxAuthoritiesCode, payload.getTaxAuthoritiesCode());
        }
        // 主管税务机关名称
        if (payload.getTaxAuthoritiesName() != null) {
            update.set(qdo.taxAuthoritiesName, payload.getTaxAuthoritiesName());
        }
        // 厂牌型号
        if (payload.getCarModel() != null) {
            update.set(qdo.carModel, payload.getCarModel());
        }
        // 合格证号
        if (payload.getCertificateNo() != null) {
            update.set(qdo.certificateNo, payload.getCertificateNo());
        }
        // 二手车市场
        if (payload.getMarketName() != null) {
            update.set(qdo.marketName, payload.getMarketName());
        }
        // 登记证号
        if (payload.getRegistrationNo() != null) {
            update.set(qdo.registrationNo, payload.getRegistrationNo());
        }
        // 序列号
        if (payload.getSerialNum() != null) {
            update.set(qdo.serialNum, payload.getSerialNum());
        }
        // 保险费
        if (payload.getPremium() != null) {
            update.set(qdo.premium, payload.getPremium());
        }
        // 印刷序号
        if (payload.getPrintNumber() != null) {
            update.set(qdo.printNumber, payload.getPrintNumber());
        }
        // 开票时间
        if (payload.getInvoiceTime() != null) {
            update.set(qdo.invoiceTime, payload.getInvoiceTime());
        }
        // 入口
        if (payload.getEntrance() != null) {
            update.set(qdo.entrance, payload.getEntrance());
        }
        // 出口
        if (payload.getRoadExit() != null) {
            update.set(qdo.roadExit, payload.getRoadExit());
        }
        // 高速标识
        if (payload.getIsHighway() != null) {
            update.set(qdo.isHighway, payload.getIsHighway());
        }
        if (payload.getReimId() != null) {
            update.set(qdo.reimId, payload.getReimId());
        }
        // 发票明细表ID
        if (payload.getReimDId() != null) {
            update.set(qdo.reimDId, payload.getReimDId());
        }
        // 已核销金额
        if (payload.getWrittenOffAmt() != null) {
            update.set(qdo.writtenOffAmt, payload.getWrittenOffAmt());
        }
        // 发票照片
        if (payload.getImgContent() != null) {
            update.set(qdo.imgContent, payload.getImgContent());
        }
        // JDE批号
        if (payload.getInvoiceVoucher() != null) {
            update.set(qdo.invoiceVoucher, payload.getInvoiceVoucher());
        }
        // jde单据公司
        if (payload.getJdeCompany() != null) {
            update.set(qdo.jdeCompany, payload.getJdeCompany());
        }
        // jde单据号
        if (payload.getJdeDocumentNo() != null) {
            update.set(qdo.jdeDocumentNo, payload.getJdeDocumentNo());
        }
        // jde单据类型
        if (payload.getJdeDocumentType() != null) {
            update.set(qdo.jdeDocumentType, payload.getJdeDocumentType());
        }
        // jde付款项
        if (payload.getJdePaymentItem() != null) {
            update.set(qdo.jdePaymentItem, payload.getJdePaymentItem());
        }
        // jde发票凭证失败原因
        if (payload.getJdeInvoiceFailReason() != null) {
            update.set(qdo.jdeInvoiceFailReason, payload.getJdeInvoiceFailReason());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 租户ID
            if (nullFields.contains("tenantId")) {
                update.setNull(qdo.tenantId);
            }
            // 发票类型COM:INPUT_INV_TYPE
            if (nullFields.contains("invType")) {
                update.setNull(qdo.invType);
            }
            // 发票状态ACC:INV_STATUS
            if (nullFields.contains("invState")) {
                update.setNull(qdo.invState);
            }
            // 发票报销状态ACC:INV_REIMB_STATUS
            if (nullFields.contains("invReimStatus")) {
                update.setNull(qdo.invReimStatus);
            }
            // 归属人
            if (nullFields.contains("invOwnerId")) {
                update.setNull(qdo.invOwnerId);
            }
            // 发票内容根据百望识别的发票内容，例如 餐饮、住宿等，需确认识别方式
            if (nullFields.contains("invLtype")) {
                update.setNull(qdo.invLtype);
            }
            // 百望系统发票ID
            if (nullFields.contains("baiwangInvId")) {
                update.setNull(qdo.baiwangInvId);
            }
            // 单据主键id:关联的单据主键id
            if (nullFields.contains("billId")) {
                update.setNull(qdo.billId);
            }
            // 文件主键id:票据信息关联的票据影像文件主键id
            if (nullFields.contains("fileId")) {
                update.setNull(qdo.fileId);
            }
            // 是否可抵扣:是否可抵扣，0：否，1：是
            if (nullFields.contains("isDeduct")) {
                update.setNull(qdo.isDeduct);
            }
            // 抵扣税额:可抵扣税额
            if (nullFields.contains("deductTax")) {
                update.setNull(qdo.deductTax);
            }
            // 百望发票类型:01-增值税专用发票，02-货物运输业增值税专用发票，03-机动车销售统一发票，04-增值税普通发票，10-增值税电子普通发票，11-增值税普通发票(卷式)，14-增值税电子普通发票（通行费），15-二手车销售统一发票，1002-火车票，1003-飞机票，1004-出租车票，1005-定额发票，1006-客运发票，99-其他发票
            if (nullFields.contains("invoiceType")) {
                update.setNull(qdo.invoiceType);
            }
            // 是否删除:是否删除，0-否，1-是
            if (nullFields.contains("isDel")) {
                update.setNull(qdo.isDel);
            }
            // 有无影像:有无影像，0-无影像，1-有影像
            if (nullFields.contains("hasPicture")) {
                update.setNull(qdo.hasPicture);
            }
            // 有无坐标:有无坐标，0-无，1-有
            if (nullFields.contains("hasPosition")) {
                update.setNull(qdo.hasPosition);
            }
            // 创建时间:创建时间：yyyy-MM-dd HH:mm:ss
            if (nullFields.contains("baiwangCreateTime")) {
                update.setNull(qdo.baiwangCreateTime);
            }
            // 修改时间:修改时间：yyyy-MM-dd HH:mm:ss
            if (nullFields.contains("baiwangUpdateTime")) {
                update.setNull(qdo.baiwangUpdateTime);
            }
            // 查验时间:查验时间：yyyy-MM-dd HH:mm:ss
            if (nullFields.contains("inspectionTime")) {
                update.setNull(qdo.inspectionTime);
            }
            // 查验状态ACC:INV_INSPECTION_STATUS:0-待查验，1-查验一致，2-查无此票，3-不一致，4-查验失败,5-无需查验
            if (nullFields.contains("inspectionStatus")) {
                update.setNull(qdo.inspectionStatus);
            }
            // 查验失败描述:查验失败描述
            if (nullFields.contains("inspectionErrorDesc")) {
                update.setNull(qdo.inspectionErrorDesc);
            }
            // 零税率标识:1:税率栏位显示“免税”， 2:税率栏位显示“不征收”， 3:零税率
            if (nullFields.contains("zeroTaxRateSign")) {
                update.setNull(qdo.zeroTaxRateSign);
            }
            // 百望发票状态:0：正常, 1：失控, 2：作废, 3：红冲, 4：异常
            if (nullFields.contains("invoiceState")) {
                update.setNull(qdo.invoiceState);
            }
            // 校验状态:票据数据校验状态，0-校验通过，1-校验失败，2-重复发票
            if (nullFields.contains("verifyStatus")) {
                update.setNull(qdo.verifyStatus);
            }
            // 票据报销状态:票据使用状态 ，0-未关联，2-已关联
            if (nullFields.contains("reimburseStatus")) {
                update.setNull(qdo.reimburseStatus);
            }
            // 归集用途类型:0：普通归集，1：审单归集，2：仅采集
            if (nullFields.contains("collectUseType")) {
                update.setNull(qdo.collectUseType);
            }
            // 创建用户账号、采集人:采集人账号
            if (nullFields.contains("userAccount")) {
                update.setNull(qdo.userAccount);
            }
            // 创建用户id、采集人:采集人id
            if (nullFields.contains("userId")) {
                update.setNull(qdo.userId);
            }
            // 用户名:采集人姓名
            if (nullFields.contains("userName")) {
                update.setNull(qdo.userName);
            }
            // 组织机构id:组织结构id
            if (nullFields.contains("orgId")) {
                update.setNull(qdo.orgId);
            }
            // 组织机构名称:组织机构名称
            if (nullFields.contains("orgName")) {
                update.setNull(qdo.orgName);
            }
            // 所属行政区编码:所属行政区编码，增值税通用
            if (nullFields.contains("administrativeDivisionNo")) {
                update.setNull(qdo.administrativeDivisionNo);
            }
            // 所属行政区名称:所属行政区名称，增值税通用
            if (nullFields.contains("administrativeDivisionName")) {
                update.setNull(qdo.administrativeDivisionName);
            }
            // 发票代码:发票代码，通用；火车票、行程单无代码，部分其他发票也无代码
            if (nullFields.contains("invoiceCode")) {
                update.setNull(qdo.invoiceCode);
            }
            // 发票号码:发票号码，通用；火车票为左上角票号，行程单为电子客票号码
            if (nullFields.contains("invoiceNo")) {
                update.setNull(qdo.invoiceNo);
            }
            // 开票日期:开票日期，通用，火车票对应的是乘车日期YYYY-MM-DD
            if (nullFields.contains("invoiceDate")) {
                update.setNull(qdo.invoiceDate);
            }
            // 购方名称:购方名称-增值税通用
            if (nullFields.contains("purchaserName")) {
                update.setNull(qdo.purchaserName);
            }
            // 购方税号:“购方税号，通用；机动车-购方名称及身份证号码/组织机构代码”
            if (nullFields.contains("purchaserTaxNo")) {
                update.setNull(qdo.purchaserTaxNo);
            }
            // 购方开户行账户:购方开户行账户，通用，卷票无
            if (nullFields.contains("purchaserBank")) {
                update.setNull(qdo.purchaserBank);
            }
            // 购方地址电话:购方地址电话，通用，卷票无
            if (nullFields.contains("purchaserAddressPhone")) {
                update.setNull(qdo.purchaserAddressPhone);
            }
            // 销方名称:销方名称，通用
            if (nullFields.contains("saleName")) {
                update.setNull(qdo.saleName);
            }
            // 销方税号:销方税号，通用
            if (nullFields.contains("saleTaxNo")) {
                update.setNull(qdo.saleTaxNo);
            }
            // 销方地址电话:销方地址电话，增值税通用，卷票无
            if (nullFields.contains("saleAddressPhone")) {
                update.setNull(qdo.saleAddressPhone);
            }
            // 销方开户行账户:销方开户行账户，增值税通用，卷票无
            if (nullFields.contains("saleBank")) {
                update.setNull(qdo.saleBank);
            }
            // 不含税金额:不含税金额：1.增值税专/普/电：不含税总金额合计；2.机动车：不含税价
            if (nullFields.contains("totalAmount")) {
                update.setNull(qdo.totalAmount);
            }
            // 合计税额:合计税额：1.增值税专/普/电：税额合计；2.机动车：增值税税额；3.行程单：燃油附加费
            if (nullFields.contains("totalTax")) {
                update.setNull(qdo.totalTax);
            }
            // 价税合计:价税合计：1.增值税专/普/电/机动车：价税合计；2.二手车：车价合计；3.卷票：合计金额；4.火车票/出租车/定额/客运发票/其他发票：金额；5.行程单：合计（票价+民航发展基金+燃油附加+其他税费）
            if (nullFields.contains("amountTax")) {
                update.setNull(qdo.amountTax);
            }
            // 价税合计_中文:价税合计_中文，增值税通用
            if (nullFields.contains("amountTaxCn")) {
                update.setNull(qdo.amountTaxCn);
            }
            // 其他税费:其他税费（行程单专有）
            if (nullFields.contains("otherTax")) {
                update.setNull(qdo.otherTax);
            }
            // 民航发展基金:民航发展基金（行程单专有）
            if (nullFields.contains("civilAviationFund")) {
                update.setNull(qdo.civilAviationFund);
            }
            // 校验码:校验码，增值税普/电/卷票、行程单含有
            if (nullFields.contains("checkCode")) {
                update.setNull(qdo.checkCode);
            }
            // 税控盘号、机器码:税控盘号、机器码，增值税通用
            if (nullFields.contains("machineCode")) {
                update.setNull(qdo.machineCode);
            }
            // 发票密文:增值税专/普/电-密码区，机动车/二手车-税控码
            if (nullFields.contains("ciphertext")) {
                update.setNull(qdo.ciphertext);
            }
            // 备注:发票备注，增值税通用
            if (nullFields.contains("baiwangRemark")) {
                update.setNull(qdo.baiwangRemark);
            }
            // 乘车人:乘车/机人姓名-火车票/行程单/客运发票
            if (nullFields.contains("drawer")) {
                update.setNull(qdo.drawer);
            }
            // 出发地:出发地：火车票、客运发票
            if (nullFields.contains("leaveCity")) {
                update.setNull(qdo.leaveCity);
            }
            // 到站地:到达地：火车票、客运发票
            if (nullFields.contains("arriveCity")) {
                update.setNull(qdo.arriveCity);
            }
            // 出发时间:出发时间：yyyy-MM-dd HH:mm，火车票的乘车时间，出租车票的上车时间
            if (nullFields.contains("leaveTime")) {
                update.setNull(qdo.leaveTime);
            }
            // 到站时间:到达时间：yyyy-MM-dd HH:mm，出租车票的下车时间
            if (nullFields.contains("arriveTime")) {
                update.setNull(qdo.arriveTime);
            }
            // 车次:火车票车次
            if (nullFields.contains("trainNumber")) {
                update.setNull(qdo.trainNumber);
            }
            // 座位等级:火车票座位类型
            if (nullFields.contains("trainSeat")) {
                update.setNull(qdo.trainSeat);
            }
            // 身份证号:身份证号（行程单）
            if (nullFields.contains("idNum")) {
                update.setNull(qdo.idNum);
            }
            // 里程:里程（出租车）
            if (nullFields.contains("mileage")) {
                update.setNull(qdo.mileage);
            }
            // 税率(明细最大最小税率，格式：6%~13%、13%)
            if (nullFields.contains("taxRate")) {
                update.setNull(qdo.taxRate);
            }
            // 买方单位代码,个人身份证号
            if (nullFields.contains("buyerIdentification")) {
                update.setNull(qdo.buyerIdentification);
            }
            // 二手车市场税号
            if (nullFields.contains("marketTaxNo")) {
                update.setNull(qdo.marketTaxNo);
            }
            // 卖方单位代码,个人身份证号
            if (nullFields.contains("sellerId")) {
                update.setNull(qdo.sellerId);
            }
            // 发票在影像中的旋转角度（顺时针）
            if (nullFields.contains("orientation")) {
                update.setNull(qdo.orientation);
            }
            //  是否有公司印章，0：无；1：有
            if (nullFields.contains("hasSeal")) {
                update.setNull(qdo.hasSeal);
            }
            // 车牌号(出租车)
            if (nullFields.contains("carNo")) {
                update.setNull(qdo.carNo);
            }
            // 车架号/车辆识别代码(机动车、二手车)
            if (nullFields.contains("carCode")) {
                update.setNull(qdo.carCode);
            }
            // 发动机号码(机动车)
            if (nullFields.contains("carEngineCode")) {
                update.setNull(qdo.carEngineCode);
            }
            // 机打号码
            if (nullFields.contains("machineInvoiceNo")) {
                update.setNull(qdo.machineInvoiceNo);
            }
            // 机打代码
            if (nullFields.contains("machineInvoiceCode")) {
                update.setNull(qdo.machineInvoiceCode);
            }
            // 异步查询码
            if (nullFields.contains("asyncCode")) {
                update.setNull(qdo.asyncCode);
            }
            // 发票坐标对象 该发票在文件中的位置
            if (nullFields.contains("invoicePositionX1")) {
                update.setNull(qdo.invoicePositionX1);
            }
            // 发票坐标对象 该发票在文件中的位置
            if (nullFields.contains("invoicePositionY1")) {
                update.setNull(qdo.invoicePositionY1);
            }
            // 发票坐标对象 该发票在文件中的位置
            if (nullFields.contains("invoicePositionX2")) {
                update.setNull(qdo.invoicePositionX2);
            }
            // 发票坐标对象 该发票在文件中的位置
            if (nullFields.contains("invoicePositionY2")) {
                update.setNull(qdo.invoicePositionY2);
            }
            // 发票联次
            if (nullFields.contains("invoiceTemplateType")) {
                update.setNull(qdo.invoiceTemplateType);
            }
            // 发票联次名称
            if (nullFields.contains("invoiceTemplateName")) {
                update.setNull(qdo.invoiceTemplateName);
            }
            // 发票密文
            if (nullFields.contains("invoiceCiphertext")) {
                update.setNull(qdo.invoiceCiphertext);
            }
            // 主管税务机关代码
            if (nullFields.contains("taxAuthoritiesCode")) {
                update.setNull(qdo.taxAuthoritiesCode);
            }
            // 主管税务机关名称
            if (nullFields.contains("taxAuthoritiesName")) {
                update.setNull(qdo.taxAuthoritiesName);
            }
            // 厂牌型号
            if (nullFields.contains("carModel")) {
                update.setNull(qdo.carModel);
            }
            // 合格证号
            if (nullFields.contains("certificateNo")) {
                update.setNull(qdo.certificateNo);
            }
            // 二手车市场
            if (nullFields.contains("marketName")) {
                update.setNull(qdo.marketName);
            }
            // 登记证号
            if (nullFields.contains("registrationNo")) {
                update.setNull(qdo.registrationNo);
            }
            // 序列号
            if (nullFields.contains("serialNum")) {
                update.setNull(qdo.serialNum);
            }
            // 保险费
            if (nullFields.contains("premium")) {
                update.setNull(qdo.premium);
            }
            // 印刷序号
            if (nullFields.contains("printNumber")) {
                update.setNull(qdo.printNumber);
            }
            // 开票时间
            if (nullFields.contains("invoiceTime")) {
                update.setNull(qdo.invoiceTime);
            }
            // 入口
            if (nullFields.contains("entrance")) {
                update.setNull(qdo.entrance);
            }
            // 出口
            if (nullFields.contains("roadExit")) {
                update.setNull(qdo.roadExit);
            }
            // 高速标识
            if (nullFields.contains("isHighway")) {
                update.setNull(qdo.isHighway);
            }
            // 发票明细表ID
            if (nullFields.contains("reimDId")) {
                update.setNull(qdo.reimDId);
            }
            // 已核销金额
            if (nullFields.contains("writtenOffAmt")) {
                update.setNull(qdo.writtenOffAmt);
            }
            // 发票照片
            if (nullFields.contains("imgContent")) {
                update.setNull(qdo.imgContent);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 更新发票报销状态 或 删除报销单与发票关联，并将发票报销状态置为新建
     *
     * @param reimId
     * @param reimStatus 报销状态
     */
    public long updateReimStatusByReimId(Long reimId, String reimStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo);
        update.set(qdo.invReimStatus, reimStatus);
        //报销单删除，取消关联，并将发票报销状态置为新建
        if (InvoiceReimStatusEnum.NEW.getCode().equals(reimStatus)) {
            update.setNull(qdo.reimId);
            update.setNull(qdo.reimDId);
        }
        update.where(qdo.reimId.eq(reimId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据发票号查询抵扣税额/价税合计
     *
     * @param invoiceNos 发票号
     * @return key 发票号，value 抵扣税额/价税合计
     */
    public List<Tuple> findDeductTaxDeviceAmountTaxByNoIn(List<String> invoiceNos) {
        return jpaQueryFactory.select(
                        qdo.invoiceNo,
                        qdo.invoiceCode,
                        Expressions.cases()
                                .when(qdo.amountTax.ne(BigDecimal.ZERO))
                                .then(qdo.deductTax.divide(qdo.amountTax))
                                .otherwise(BigDecimal.ZERO))
                .from(qdo)
                .where(qdo.isDel.eq(0).and(qdo.deleteFlag.eq(0)))
                .where(qdo.invoiceNo.in(invoiceNos))
                .fetch();
    }

    /**
     * 根据发票号-发票代码新增已核销金额
     *
     * @param invoiceNoAndCode 发票号-发票代码
     * @param addWrittenOffAmt 新增的已核销金额
     */
    public void addWrittenOffAmtByInvoiceNo(String invoiceNoAndCode, BigDecimal addWrittenOffAmt) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.writtenOffAmt, qdo.writtenOffAmt.coalesce(BigDecimal.ZERO).add(addWrittenOffAmt))
                .where(qdo.isDel.eq(0).and(qdo.deleteFlag.eq(0)))
                .where(qdo.invoiceNo.concat("-").concat(qdo.invoiceCode.coalesce("")).eq(invoiceNoAndCode));
        update.execute();
    }

    /**
     * 根据发票号-发票代码减去已核销金额
     *
     * @param invoiceNoAndCode 发票号-发票代码
     * @param subWrittenOffAmt 减去的已核销金额
     */
    public void subWrittenOffAmtByInvoiceNo(String invoiceNoAndCode, BigDecimal subWrittenOffAmt) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.writtenOffAmt, qdo.writtenOffAmt.coalesce(BigDecimal.ZERO).subtract(subWrittenOffAmt))
                .where(qdo.isDel.eq(0).and(qdo.deleteFlag.eq(0)))
                .where(qdo.invoiceNo.concat("-").concat(qdo.invoiceCode.coalesce("")).eq(invoiceNoAndCode));
        update.execute();
    }

    /**
     * 根据主键查询
     *
     * @param invoiceNo 主键
     * @return 结果
     */
    public List<InvInvoiceVO> queryInvoiceNoByKey(String invoiceNo) {
        JPAQuery<InvInvoiceVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.invoiceNo.eq(invoiceNo));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

}

