package com.elitesland.tw.tw5.server.prd.partner.common.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.partner.common.payload.BookInvoicePayload;
import com.elitesland.tw.tw5.api.prd.partner.common.query.BookInvoiceQuery;
import com.elitesland.tw.tw5.api.prd.partner.common.vo.BookInvoiceVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.BookInvoiceDO;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBookInvoiceDO;
import com.elitesland.tw.tw5.server.prd.partner.common.repo.BookInvoiceRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 业务伙伴-发票信息
 *
 * @author wangly
 * @date 2023-05-16
 */
@Repository
@RequiredArgsConstructor
public class BookInvoiceDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BookInvoiceRepo repo;
    private final QBookInvoiceDO qdo = QBookInvoiceDO.bookInvoiceDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BookInvoiceVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BookInvoiceVO.class,
                qdo.id,
                qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 地址薄id
                qdo.bookId,
                // 发票信息
                qdo.invoiceMessage,
                // 发票抬头
                qdo.invoiceTitle,
                // 税率 udc[prd:ab:tax_rate]
                qdo.taxRete,
                // 税号
                qdo.taxNo,
                // 开票类型 udc[prd:ab:inv_type]
                qdo.invoiceType,
                // 开票地址
                qdo.invoiceAddress,
                // 电话
                qdo.invoicePhone,
                // 开户行
                qdo.depositBank,
                // 账户
                qdo.invoiceAccount,
                // 币种 udc[SYSTEM_BASIC:CURRENCY]
                qdo.currency,
                // 是否为默认开票信息
                qdo.isDefault
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BookInvoiceVO> getJpaQueryWhere(BookInvoiceQuery query) {
        JPAQuery<BookInvoiceVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
         //   jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BookInvoiceQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BookInvoiceQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 地址薄id 精确 */
        if (!ObjectUtils.isEmpty(query.getBookId())) {
            list.add(qdo.bookId.eq(query.getBookId()));
        }
        /** 发票信息 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceMessage())) {
            list.add(qdo.invoiceMessage.eq(query.getInvoiceMessage()));
        }
        /** 发票抬头 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceTitle())) {
            list.add(qdo.invoiceTitle.eq(query.getInvoiceTitle()));
        }
        /** 税率 udc[prd:ab:tax_rate] 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxRete())) {
            list.add(qdo.taxRete.eq(query.getTaxRete()));
        }
        /** 税号 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxNo())) {
            list.add(qdo.taxNo.eq(query.getTaxNo()));
        }
        /** 开票类型 udc[prd:ab:inv_type] 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceType())) {
            list.add(qdo.invoiceType.eq(query.getInvoiceType()));
        }
        /** 开票地址 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceAddress())) {
            list.add(qdo.invoiceAddress.eq(query.getInvoiceAddress()));
        }
        /** 电话 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoicePhone())) {
            list.add(qdo.invoicePhone.eq(query.getInvoicePhone()));
        }
        /** 开户行 精确 */
        if (!ObjectUtils.isEmpty(query.getDepositBank())) {
            list.add(qdo.depositBank.eq(query.getDepositBank()));
        }
        /** 账户 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceAccount())) {
            list.add(qdo.invoiceAccount.eq(query.getInvoiceAccount()));
        }
        /** 币种 udc[SYSTEM_BASIC:CURRENCY] 精确 */
        if (!ObjectUtils.isEmpty(query.getCurrency())) {
            list.add(qdo.currency.eq(query.getCurrency()));
        }
        /** 是否为默认开票信息 精确 */
        if (!ObjectUtils.isEmpty(query.getIsDefault())) {
            list.add(qdo.isDefault.eq(query.getIsDefault()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BookInvoiceVO queryByKey(Long id) {
        JPAQuery<BookInvoiceVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据bookId查询默认
     *
     * @param bookId bookId
     * @return {@link BookInvoiceVO}
     */
    public BookInvoiceVO queryDefaultByBookId(Long bookId) {
        JPAQuery<BookInvoiceVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.bookId.eq(bookId));
        jpaQuery.where(qdo.isDefault.eq(Boolean.TRUE));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BookInvoiceVO> queryListDynamic(BookInvoiceQuery query) {
        JPAQuery<BookInvoiceVO> jpaQuery = getJpaQueryWhere(query);
        OrderItem sortNoOrderItem = OrderItem.desc("isDefault");
        OrderItem createTimeOrderItem = OrderItem.desc("createTime");
        List<OrderItem> OrderItemList = new ArrayList<>();
        OrderItemList.add(sortNoOrderItem);
        OrderItemList.add(createTimeOrderItem);
        query.setOrders(OrderItemList);
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BookInvoiceVO> queryPaging(BookInvoiceQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<BookInvoiceVO> jpaQuery = getJpaQueryWhere(query);
        List<BookInvoiceVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BookInvoiceVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BookInvoiceDO save(BookInvoiceDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BookInvoiceDO> saveAll(List<BookInvoiceDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BookInvoicePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 地址薄id
        if (payload.getBookId() != null) {
            update.set(qdo.bookId, payload.getBookId());
        }
        // 发票信息
        if (payload.getInvoiceMessage() != null) {
            update.set(qdo.invoiceMessage, payload.getInvoiceMessage());
        }
        // 发票抬头
        if (payload.getInvoiceTitle() != null) {
            update.set(qdo.invoiceTitle, payload.getInvoiceTitle());
        }
        // 税率 udc[prd:ab:tax_rate]
        if (payload.getTaxRete() != null) {
            update.set(qdo.taxRete, payload.getTaxRete());
        }
        // 税号
        if (payload.getTaxNo() != null) {
            update.set(qdo.taxNo, payload.getTaxNo());
        }
        // 开票类型 udc[prd:ab:inv_type]
        if (payload.getInvoiceType() != null) {
            update.set(qdo.invoiceType, payload.getInvoiceType());
        }
        // 开票地址
        if (payload.getInvoiceAddress() != null) {
            update.set(qdo.invoiceAddress, payload.getInvoiceAddress());
        }
        // 电话
        if (payload.getInvoicePhone() != null) {
            update.set(qdo.invoicePhone, payload.getInvoicePhone());
        }
        // 开户行
        if (payload.getDepositBank() != null) {
            update.set(qdo.depositBank, payload.getDepositBank());
        }
        // 账户
        if (payload.getInvoiceAccount() != null) {
            update.set(qdo.invoiceAccount, payload.getInvoiceAccount());
        }
        // 币种 udc[SYSTEM_BASIC:CURRENCY]
        if (payload.getCurrency() != null) {
            update.set(qdo.currency, payload.getCurrency());
        }
        // 是否为默认开票信息
        if (payload.getIsDefault() != null) {
            update.set(qdo.isDefault, payload.getIsDefault());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 地址薄id
            if (nullFields.contains("bookId")) {
                update.setNull(qdo.bookId);
            }
            // 发票信息
            if (nullFields.contains("invoiceMessage")) {
                update.setNull(qdo.invoiceMessage);
            }
            // 发票抬头
            if (nullFields.contains("invoiceTitle")) {
                update.setNull(qdo.invoiceTitle);
            }
            // 税率 udc[prd:ab:tax_rate]
            if (nullFields.contains("taxRete")) {
                update.setNull(qdo.taxRete);
            }
            // 税号
            if (nullFields.contains("taxNo")) {
                update.setNull(qdo.taxNo);
            }
            // 开票类型 udc[prd:ab:inv_type]
            if (nullFields.contains("invoiceType")) {
                update.setNull(qdo.invoiceType);
            }
            // 开票地址
            if (nullFields.contains("invoiceAddress")) {
                update.setNull(qdo.invoiceAddress);
            }
            // 电话
            if (nullFields.contains("invoicePhone")) {
                update.setNull(qdo.invoicePhone);
            }
            // 开户行
            if (nullFields.contains("depositBank")) {
                update.setNull(qdo.depositBank);
            }
            // 账户
            if (nullFields.contains("invoiceAccount")) {
                update.setNull(qdo.invoiceAccount);
            }
            // 币种 udc[SYSTEM_BASIC:CURRENCY]
            if (nullFields.contains("currency")) {
                update.setNull(qdo.currency);
            }
            // 是否为默认开票信息
            if (nullFields.contains("isDefault")) {
                update.setNull(qdo.isDefault);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
    /**
     * 根据bookId查询列表
     *
     * @param bookId bookId
     * @return {@link BookInvoiceDO}
     */
    public List<BookInvoiceDO> queryByBookId(Long bookId) {
        JPAQuery<BookInvoiceDO> jpaQuery = jpaQueryFactory.select(qdo).from(qdo);
        jpaQuery.where(qdo.bookId.eq(bookId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 逻辑删除-通过地址薄id
     *
     * @param bookId 地址薄id
     * @return 删除的行数
     */
    public long deleteByBookId(Long bookId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.bookId.eq(bookId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
//    /**
//     * 4.0发票信息同步查询
//     *
//     * @param bookId 地址薄id
//     * @return 删除的行数
//     */
//    public List<PrdAbInvoicePayload> queryTwBookInvoiceListByBookId(List<Long> bookIdList){
//        return jpaQueryFactory.select(Projections.bean(PrdAbInvoicePayload.class,
//                qdo.id,
//                qdo.remark,
//                //qdo.createUserId,
//                //qdo.creator,
//                //qdo.createTime,
//                //qdo.modifyUserId,
//                //qdo.updater,
//                //qdo.modifyTime,
//                // 地址薄id
//                qdo.bookId,
//                // 发票信息
//                qdo.invoiceMessage,
//                // 发票抬头
//                qdo.invoiceTitle,
//                // 税率 udc[prd:ab:tax_rate]
//                qdo.taxRete,
//                // 税号
//                qdo.taxNo,
//                // 开票类型 udc[prd:ab:inv_type]
//                qdo.invoiceType,
//                // 开票地址
//                qdo.invoiceAddress,
//                // 电话
//                qdo.invoicePhone,
//                // 开户行
//                qdo.depositBank,
//                // 账户
//                qdo.invoiceAccount,
//                // 币种 udc[SYSTEM_BASIC:CURRENCY]
//                qdo.currency,
//                // 是否为默认开票信息
//                qdo.isDefault.as("defaultFlag")
//        )).from(qdo).where(qdo.deleteFlag.eq(0)).where(qdo.bookId.in(bookIdList)).fetch();
//
//    }
}

