package com.elitesland.tw.tw5.server.prd.pay.dao;

import com.elitesland.tw.tw5.api.prd.pay.payload.TDocHistPayload;
import com.elitesland.tw.tw5.api.prd.pay.query.TDocHistQuery;
import com.elitesland.tw.tw5.api.prd.pay.vo.TDocHistVO;
import com.elitesland.tw.tw5.server.prd.pay.entity.TDocHistDO;
import com.elitesland.tw.tw5.server.prd.pay.entity.QTDocHistDO;
import com.elitesland.tw.tw5.server.prd.pay.repo.TDocHistRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 支付管理-支付履历
 *
 * @author wangly
 * @date 2023-12-12
 */
@Repository
@RequiredArgsConstructor
public class TDocHistDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final TDocHistRepo repo;
    private final QTDocHistDO qdo = QTDocHistDO.tDocHistDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<TDocHistVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(TDocHistVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                //qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 履历类型
                qdo.histType,
                // 履历详细
                qdo.histDtl,
                // 履历日期时间
                qdo.histTime,
                // 单据id
                qdo.docId,
                // 单据编号
                qdo.docNo,
                // 单据类型
                qdo.docType,
                //用户id
                qdo.ownerId
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<TDocHistVO> getJpaQueryWhere(TDocHistQuery query) {
        JPAQuery<TDocHistVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(TDocHistQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(TDocHistQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 主键id 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 履历类型 精确 */
        if (!ObjectUtils.isEmpty(query.getHistType())) {
            list.add(qdo.histType.eq(query.getHistType()));
        }
        /** 履历详细 精确 */
        if (!ObjectUtils.isEmpty(query.getHistDtl())) {
            list.add(qdo.histDtl.eq(query.getHistDtl()));
        }
        /** 履历日期时间 精确 */
        if (!ObjectUtils.isEmpty(query.getHistTime())) {
            list.add(qdo.histTime.eq(query.getHistTime()));
        }
        /** 单据id 精确 */
        if (!ObjectUtils.isEmpty(query.getDocId())) {
            list.add(qdo.docId.eq(query.getDocId()));
        }
        /** 单据编号 精确 */
        if (!ObjectUtils.isEmpty(query.getDocNo())) {
            list.add(qdo.docNo.eq(query.getDocNo()));
        }
        /** 单据类型 精确 */
        if (!ObjectUtils.isEmpty(query.getDocType())) {
            list.add(qdo.docType.eq(query.getDocType()));
        }
        /** 用户id 精确 */
        if (!ObjectUtils.isEmpty(query.getOwnerId())) {
            list.add(qdo.ownerId.eq(query.getOwnerId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public TDocHistVO queryByKey(Long id) {
        JPAQuery<TDocHistVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TDocHistVO> queryListDynamic(TDocHistQuery query) {
        JPAQuery<TDocHistVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<TDocHistVO> queryPaging(TDocHistQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<TDocHistVO> jpaQuery = getJpaQueryWhere(query);
        List<TDocHistVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<TDocHistVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public TDocHistDO save(TDocHistDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<TDocHistDO> saveAll(List<TDocHistDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(TDocHistPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 主键id
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 履历类型
        if (payload.getHistType() != null) {
            update.set(qdo.histType, payload.getHistType());
        }
        // 履历详细
        if (payload.getHistDtl() != null) {
            update.set(qdo.histDtl, payload.getHistDtl());
        }
        // 履历日期时间
        if (payload.getHistTime() != null) {
            update.set(qdo.histTime, payload.getHistTime());
        }
        // 单据id
        if (payload.getDocId() != null) {
            update.set(qdo.docId, payload.getDocId());
        }
        // 单据编号
        if (payload.getDocNo() != null) {
            update.set(qdo.docNo, payload.getDocNo());
        }
        // 单据类型
        if (payload.getDocType() != null) {
            update.set(qdo.docType, payload.getDocType());
        }
        // 用户id
        if (payload.getOwnerId() != null) {
            update.set(qdo.ownerId, payload.getOwnerId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键id
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 履历类型
            if (nullFields.contains("histType")) {
                update.setNull(qdo.histType);
            }
            // 履历详细
            if (nullFields.contains("histDtl")) {
                update.setNull(qdo.histDtl);
            }
            // 履历日期时间
            if (nullFields.contains("histTime")) {
                update.setNull(qdo.histTime);
            }
            // 单据id
            if (nullFields.contains("docId")) {
                update.setNull(qdo.docId);
            }
            // 单据编号
            if (nullFields.contains("docNo")) {
                update.setNull(qdo.docNo);
            }
            // 单据类型
            if (nullFields.contains("docType")) {
                update.setNull(qdo.docType);
            }
            // 单据类型
            if (nullFields.contains("ownerId")) {
                update.setNull(qdo.ownerId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

