package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PaymentSlipPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PaymentSlipQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PaymentSlipVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccFinancialSubjectDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.PaymentSlipDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPaymentSlipDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchasePaymentDO;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchasePaymentEnum;
import com.elitesland.tw.tw5.server.prd.purchase.repo.PaymentSlipRepo;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

/**
 * 付款单记录
 *
 * @author likunpeng
 * @date 2023-11-23
 */
@Repository
@RequiredArgsConstructor
public class PaymentSlipDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PaymentSlipRepo repo;
    private final QPaymentSlipDO qdo = QPaymentSlipDO.paymentSlipDO;
    private final QPurchasePaymentDO qPurchasePaymentDO = QPurchasePaymentDO.purchasePaymentDO;
    private final QAccFinancialSubjectDO qAccFinancialSubjectDO = QAccFinancialSubjectDO.accFinancialSubjectDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PaymentSlipVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PaymentSlipVO.class,
                        qdo.id,
                        qdo.remark,
                        //qdo.createUserId,
                        //qdo.creator,
                        //qdo.createTime,
                        //qdo.modifyUserId,
                        //qdo.updater,
                        //qdo.modifyTime,
                        // 付款流水号
                        qdo.paySerialNum,
                        // 付款申请单ID
                        qdo.paymentApplyId,
                        // 付款申请单编号
                        qPurchasePaymentDO.paymentNo.as("paymentApplyNo"),
                        // 付款申请单名称
                        qPurchasePaymentDO.purchaseName.as("paymentApplyName"),
                        // 付款事由
                        qPurchasePaymentDO.reasonDesc,
                        // 付款申请人
                        qPurchasePaymentDO.purchaseInchargeResId,
                        // 申请单类型
                        qPurchasePaymentDO.paymentApplicationType,
                        // 申请日期
                        qPurchasePaymentDO.applicationDate,
                        // 付款金额
                        qdo.paymentAmt,
                        // 货币
                        qPurchasePaymentDO.currCode,
                        // 付款日期
                        qdo.paymentDate,
                        // 应付会计备注
                        qdo.apAccountantRemark,
                        // 财务经理备注
                        qdo.financeRemark,
                        //cfo备注
                        qdo.cfoRemark,
                        // 出纳备注
                        qdo.cashierRemark,
                        // 状态
                        qdo.state,
                        // 收款公司
                        qdo.receivingCompanyBookId,
                        // 收款银行
                        qdo.receivingBank,
                        // 收款账号
                        qdo.receivingAccount,
                        // 付款银行
                        qdo.paymentBank,
                        // 付款账号
                        qdo.paymentAccount,
                        // 付款公司
                        qdo.payCompanyBookId,
                        // 科目说明
                        qdo.accMasId,
                        //科目说明名称
                        qAccFinancialSubjectDO.accName.as("accMasName"),
                        // 付款方式
                        qdo.paymentMethod,
                        // 支付方式：1.网银支付 2.手动支付
                        qdo.payMode,
                        // 支付状态：S 成功、F 失败、C撤销、D 过期、R 否决 G退票
                        qdo.payStatus,
                        // 支付时间（状态时间）
                        qdo.payTime,
                        // 实际付款日期
                        qdo.actualDate,
                        // 应付会计更新时间
                        qdo.apAccountantTime,
                        // 财务经理更新时间
                        qdo.financeTime,
                        //cfo更新时间
                        qdo.cfoTime,
                        // 出纳更新时间
                        qdo.cashierTime,
                        // 网银提交标识 1.已提交 2.未提交
                        qdo.bankFlag,
                        // 网银提交时间
                        qdo.bankTime,
                        // 支付标识：1.已支付 ；2.待支付
                        qPurchasePaymentDO.payFlag,
                        // 支付用途
                        qdo.payPurpose,
                        // 流程实例ID
                        qdo.procInstId,
                        // 流程审批状态
                        qdo.procInstStatus,
                        // 提交时间
                        qdo.submitTime,
                        // 审批时间
                        qdo.approvedTime,
                        qdo.payApprovedTime,
                        qdo.payProofDigest,
                        qdo.payProofNo,
                        qdo.payProofStatus,
                        qdo.payProofFailReason,
                        qdo.jdeCompany,
                        qdo.jdePaymentItem,
                        qdo.jdeDocumentNo,
                        qdo.jdePayNo,
                        qdo.jdeVoucher,
                        qdo.jdeDocumentType,
                        // 付款单备注
                        qPurchasePaymentDO.accountingNote
                )).from(qdo)
                .leftJoin(qPurchasePaymentDO).on(qdo.paymentApplyId.eq(qPurchasePaymentDO.id)
                        .and(qPurchasePaymentDO.deleteFlag.eq(0))).
                leftJoin(qAccFinancialSubjectDO).on(qdo.accMasId.eq(qAccFinancialSubjectDO.id).and(qAccFinancialSubjectDO.deleteFlag.eq(0)));
    }

    /**
     * 通过id查询列表
     *
     * @param idList id集合
     * @return jpaQuery对象
     */
    public List<PaymentSlipVO> queryPaymentSlipVOListByIdlist(List<Long> idList) {
        JPAQuery<PaymentSlipVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.where(qdo.id.in(idList));
        return jpaQuery.fetch();
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PaymentSlipVO> getJpaQueryWhere(PaymentSlipQuery query) {
        JPAQuery<PaymentSlipVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PaymentSlipQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .leftJoin(qPurchasePaymentDO).on(qdo.paymentApplyId.eq(qPurchasePaymentDO.id)
                        .and(qPurchasePaymentDO.deleteFlag.eq(0)));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PaymentSlipQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 付款申请单ID 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentApplyId())) {
            list.add(qdo.paymentApplyId.eq(query.getPaymentApplyId()));
        }
        /** 付款申请单ID 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentApplyIdList())) {
            list.add(qdo.paymentApplyId.in(query.getPaymentApplyIdList()));
        }
        /** 付款流水号 精确 */
        if (!ObjectUtils.isEmpty(query.getPaySerialNum())) {
            list.add(qdo.paySerialNum.like(SqlUtil.toSqlLikeString(query.getPaySerialNum())));
        }
        /** 付款申请单编号 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentApplyNo())) {
            list.add(qPurchasePaymentDO.paymentNo.like(SqlUtil.toSqlLikeString(query.getPaymentApplyNo())));
        }
        /** 付款申请人 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseInchargeResId())) {
            list.add(qPurchasePaymentDO.purchaseInchargeResId.eq(query.getPurchaseInchargeResId()));
        }
        /** 付款公司 精确 */
        if (!ObjectUtils.isEmpty(query.getPayCompanyBookId())) {
            list.add(qdo.payCompanyBookId.eq(query.getPayCompanyBookId()));
        }
        /** 供应商 精确 */
        if (!ObjectUtils.isEmpty(query.getSupplierBookId())) {
            list.add(qPurchasePaymentDO.supplierLegalBookId.eq(query.getSupplierBookId()));
        }

        /** 付款事由 精确 */
        if (!ObjectUtils.isEmpty(query.getReasonDesc())) {
            list.add(qPurchasePaymentDO.reasonDesc.like(SqlUtil.toSqlLikeString(query.getReasonDesc())));
        }
        /** 付款单备注 模糊 */
        if (!ObjectUtils.isEmpty(query.getAccountingNote())) {
            list.add(qPurchasePaymentDO.accountingNote.like("%" + query.getAccountingNote() + "%"));
        }
        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getState())) {
            list.add(qdo.state.eq(query.getState()));
        }
        /** 状态集合 精确 */
        if (!ObjectUtils.isEmpty(query.getStateList())) {
            list.add(qdo.state.in(query.getStateList()));
        }
        /** 付款日期 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentDateStart())) {
            list.add(qdo.paymentDate.goe(query.getPaymentDateStart()));
        }
        /** 付款日期 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentDateEnd())) {
            list.add(qdo.paymentDate.loe(query.getPaymentDateEnd()));
        }
        /** 支付方式：1.网银支付 2.手动支付 精确 */
        if (!ObjectUtils.isEmpty(query.getPayMode())) {
            list.add(qdo.payMode.eq(query.getPayMode()));
        }
        /** 支付状态：S 成功、F 失败、C撤销、D 过期、R 否决 G退票 精确 */
        if (!ObjectUtils.isEmpty(query.getPayStatus())) {
            list.add(qdo.payStatus.eq(query.getPayStatus()));
        }
        /** 付款银行 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentBank())) {
            list.add(qdo.paymentBank.eq(query.getPaymentBank()));
        }
        /** 付款账号 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentAccount())) {
            list.add(qdo.paymentAccount.eq(query.getPaymentAccount()));
        }
        /** 付款金额 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentAmt())) {
            list.add(qdo.paymentAmt.eq(query.getPaymentAmt()));
        }

        /** 科目说明 精确 */
        if (!ObjectUtils.isEmpty(query.getAccMasId())) {
            list.add(qdo.accMasId.eq(query.getAccMasId()));
        }
        /** 付款方式 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentMethod())) {
            list.add(qdo.paymentMethod.eq(query.getPaymentMethod()));
        }

        /** 收款公司 精确 */
        if (!ObjectUtils.isEmpty(query.getReceivingCompanyBookId())) {
            list.add(qdo.receivingCompanyBookId.eq(query.getReceivingCompanyBookId()));
        }
        /** 收款银行 精确 */
        if (!ObjectUtils.isEmpty(query.getReceivingBank())) {
            list.add(qdo.receivingBank.eq(query.getReceivingBank()));
        }
        /** 收款账号 精确 */
        if (!ObjectUtils.isEmpty(query.getReceivingAccount())) {
            list.add(qdo.receivingAccount.eq(query.getReceivingAccount()));
        }

        /** 出纳更新时间 精确 */
        if (!ObjectUtils.isEmpty(query.getCashierTime())) {
            list.add(qdo.cashierTime.eq(query.getCashierTime()));
        }
        /** 出纳备注 精确 */
        if (!ObjectUtils.isEmpty(query.getCashierRemark())) {
            list.add(qdo.cashierRemark.eq(query.getCashierRemark()));
        }
        /** 应付会计更新时间 精确 */
        if (!ObjectUtils.isEmpty(query.getApAccountantTime())) {
            list.add(qdo.apAccountantTime.eq(query.getApAccountantTime()));
        }
        /** 应付会计备注 精确 */
        if (!ObjectUtils.isEmpty(query.getApAccountantRemark())) {
            list.add(qdo.apAccountantRemark.eq(query.getApAccountantRemark()));
        }
        /** 财务经理更新时间 精确 */
        if (!ObjectUtils.isEmpty(query.getFinanceTime())) {
            list.add(qdo.financeTime.eq(query.getFinanceTime()));
        }
        /** 财务经理备注 精确 */
        if (!ObjectUtils.isEmpty(query.getFinanceRemark())) {
            list.add(qdo.financeRemark.eq(query.getFinanceRemark()));
        }
        /** CFO更新时间 精确 */
        if (!ObjectUtils.isEmpty(query.getCfoTime())) {
            list.add(qdo.cfoTime.eq(query.getCfoTime()));
        }
        /** CFO备注 精确 */
        if (!ObjectUtils.isEmpty(query.getCfoRemark())) {
            list.add(qdo.cfoRemark.eq(query.getCfoRemark()));
        }
        /** 实际付款日期 精确 */
        if (!ObjectUtils.isEmpty(query.getActualDate())) {
            list.add(qdo.actualDate.eq(query.getActualDate()));
        }
        /** 支付时间 精确 */
        if (!ObjectUtils.isEmpty(query.getPayTime())) {
            list.add(qdo.payTime.eq(query.getPayTime()));
        }

        /** 网银提交标识 1.已提交 2.未提交 精确 */
        if (!ObjectUtils.isEmpty(query.getBankFlag())) {
            list.add(qdo.bankFlag.eq(query.getBankFlag()));
        }
        /** 网银提交时间 精确 */
        if (!ObjectUtils.isEmpty(query.getBankTime())) {
            list.add(qdo.bankTime.eq(query.getBankTime()));
        }
        /** 支付标识：1.已支付 ；2.待支付 精确 */
        if (!ObjectUtils.isEmpty(query.getPayFlag())) {
            list.add(qdo.payFlag.eq(query.getPayFlag()));
        }
        /** 支付用途 精确 */
        if (!ObjectUtils.isEmpty(query.getPayPurpose())) {
            list.add(qdo.payPurpose.eq(query.getPayPurpose()));
        }
        /** 财务记账日期 精确 */
        if (!ObjectUtils.isEmpty(query.getPayApprovedTimeStart())) {
            list.add(qdo.payApprovedTime.goe(query.getPayApprovedTimeStart()));
        }
        /** 财务记账日期 精确 */
        if (!ObjectUtils.isEmpty(query.getPayApprovedTimeEnd())) {
            list.add(qdo.payApprovedTime.loe(query.getPayApprovedTimeEnd()));
        }
        /** 付款凭证状态 精确 */
        if (!ObjectUtils.isEmpty(query.getPayProofStatus())) {
            list.add(qdo.payProofStatus.eq(query.getPayProofStatus()));
        }
        /** 付款凭证号 精确 */
        if (!ObjectUtils.isEmpty(query.getPayProofNo())) {
            list.add(qdo.payProofNo.eq(query.getPayProofNo()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PaymentSlipVO queryByKey(Long id) {
        JPAQuery<PaymentSlipVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PaymentSlipVO> queryListDynamic(PaymentSlipQuery query) {
        JPAQuery<PaymentSlipVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 根据主键集合查询
     *
     * @param idList 查询参数
     * @return 结果集合
     */
    public List<PaymentSlipVO> queryListByIds(List<Long> idList) {
        JPAQuery<PaymentSlipVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(idList));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PaymentSlipVO> queryPaging(PaymentSlipQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PaymentSlipVO> jpaQuery = getJpaQueryWhere(query);
        List<PaymentSlipVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PaymentSlipVO>builder().records(result).total(total).build();
    }


    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PaymentSlipVO> queryAccountPaging(PaymentSlipQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PaymentSlipVO> jpaQuery = getJpaQueryWhere(query);
        List<PaymentSlipVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PaymentSlipVO>builder().records(result).total(total).build();
    }

    //付款批准分组查询
    public PagingVO<PaymentSlipVO> paymentApproveGroupPaging(PaymentSlipQuery query) {
        JPAQuery<PaymentSlipVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PaymentSlipVO.class,
                        qdo.paySerialNum,
                        qdo.paymentAmt.sum().as("totalAmount"))).
                from(qdo).leftJoin(qPurchasePaymentDO).on(qdo.paymentApplyId.eq(qPurchasePaymentDO.id)
                        .and(qPurchasePaymentDO.deleteFlag.eq(0)));
        ;
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        jpaQuery.groupBy(qdo.paySerialNum);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        List<PaymentSlipVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();

        if (CollectionUtils.isEmpty(result)) {
            return PagingVO.empty();
        }
        return PagingVO.<PaymentSlipVO>builder().records(result).total(result.size()).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PaymentSlipDO save(PaymentSlipDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PaymentSlipDO> saveAll(List<PaymentSlipDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PaymentSlipPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 付款申请单ID
        if (payload.getPaymentApplyId() != null) {
            update.set(qdo.paymentApplyId, payload.getPaymentApplyId());
        }
        // 付款流水号
        if (payload.getPaySerialNum() != null) {
            update.set(qdo.paySerialNum, payload.getPaySerialNum());
        }
        // 付款银行
        if (payload.getPaymentBank() != null) {
            update.set(qdo.paymentBank, payload.getPaymentBank());
        }
        // 付款账号
        if (payload.getPaymentAccount() != null) {
            update.set(qdo.paymentAccount, payload.getPaymentAccount());
        }
        // 付款金额
        if (payload.getPaymentAmt() != null) {
            update.set(qdo.paymentAmt, payload.getPaymentAmt());
        }
        // 付款公司
        if (payload.getPayCompanyBookId() != null) {
            update.set(qdo.payCompanyBookId, payload.getPayCompanyBookId());
        }
        // 科目说明
        if (payload.getAccMasId() != null) {
            update.set(qdo.accMasId, payload.getAccMasId());
        }
        // 付款方式
        if (payload.getPaymentMethod() != null) {
            update.set(qdo.paymentMethod, payload.getPaymentMethod());
        }
        // 付款日期
        if (payload.getPaymentDate() != null) {
            update.set(qdo.paymentDate, payload.getPaymentDate());
        }
        // 收款公司
        if (payload.getReceivingCompanyBookId() != null) {
            update.set(qdo.receivingCompanyBookId, payload.getReceivingCompanyBookId());
        }
        // 收款银行
        if (payload.getReceivingBank() != null) {
            update.set(qdo.receivingBank, payload.getReceivingBank());
        }
        // 收款账号
        if (payload.getReceivingAccount() != null) {
            update.set(qdo.receivingAccount, payload.getReceivingAccount());
        }
        // 状态
        if (payload.getState() != null) {
            update.set(qdo.state, payload.getState());
        }
        // 出纳更新时间
        if (payload.getCashierTime() != null) {
            update.set(qdo.cashierTime, payload.getCashierTime());
        }
        // 出纳备注
        if (payload.getCashierRemark() != null) {
            update.set(qdo.cashierRemark, payload.getCashierRemark());
        }
        // 应付会计更新时间
        if (payload.getApAccountantTime() != null) {
            update.set(qdo.apAccountantTime, payload.getApAccountantTime());
        }
        // 应付会计备注
        if (payload.getApAccountantRemark() != null) {
            update.set(qdo.apAccountantRemark, payload.getApAccountantRemark());
        }
        // 财务经理更新时间
        if (payload.getFinanceTime() != null) {
            update.set(qdo.financeTime, payload.getFinanceTime());
        }
        // 财务经理备注
        if (payload.getFinanceRemark() != null) {
            update.set(qdo.financeRemark, payload.getFinanceRemark());
        }
        // 财务经理更新时间
        if (payload.getCfoTime() != null) {
            update.set(qdo.cfoTime, payload.getCfoTime());
        }
        // 财务经理备注
        if (payload.getCfoRemark() != null) {
            update.set(qdo.cfoRemark, payload.getCfoRemark());
        }
        // 实际付款日期
        if (payload.getActualDate() != null) {
            update.set(qdo.actualDate, payload.getActualDate());
        }
        // 支付时间
        if (payload.getPayTime() != null) {
            update.set(qdo.payTime, payload.getPayTime());
        }
        // 支付状态：S 成功、F 失败、C撤销、D 过期、R 否决 G退票
        if (payload.getPayStatus() != null) {
            update.set(qdo.payStatus, payload.getPayStatus());
        }
        // 支付方式：1.网银支付 2.手动支付
        if (payload.getPayMode() != null) {
            update.set(qdo.payMode, payload.getPayMode());
        }
        // 网银提交标识 1.已提交 2.未提交
        if (payload.getBankFlag() != null) {
            update.set(qdo.bankFlag, payload.getBankFlag());
        }
        // 网银提交时间
        if (payload.getBankTime() != null) {
            update.set(qdo.bankTime, payload.getBankTime());
        }
        // 支付标识：1.已支付 ；2.待支付
        if (payload.getPayFlag() != null) {
            update.set(qdo.payFlag, payload.getPayFlag());
        }
        // 支付用途
        if (payload.getPayPurpose() != null) {
            update.set(qdo.payPurpose, payload.getPayPurpose());
        }
        // 流程实例ID
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 流程审批状态
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        // 提交时间
        if (payload.getSubmitTime() != null) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        // 审批时间
        if (payload.getApprovedTime() != null) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 付款申请单ID
            if (nullFields.contains("paymentApplyId")) {
                update.setNull(qdo.paymentApplyId);
            }
            // 付款流水号
            if (nullFields.contains("paySerialNum")) {
                update.setNull(qdo.paySerialNum);
            }
            // 付款银行
            if (nullFields.contains("paymentBank")) {
                update.setNull(qdo.paymentBank);
            }
            // 付款账号
            if (nullFields.contains("paymentAccount")) {
                update.setNull(qdo.paymentAccount);
            }
            // 付款金额
            if (nullFields.contains("paymentAmt")) {
                update.setNull(qdo.paymentAmt);
            }
            // 付款公司
            if (nullFields.contains("payCompanyBookId")) {
                update.setNull(qdo.payCompanyBookId);
            }
            // 科目说明
            if (nullFields.contains("accMasId")) {
                update.setNull(qdo.accMasId);
            }
            // 付款方式
            if (nullFields.contains("paymentMethod")) {
                update.setNull(qdo.paymentMethod);
            }
            // 付款日期
            if (nullFields.contains("paymentDate")) {
                update.setNull(qdo.paymentDate);
            }
            // 收款公司
            if (nullFields.contains("receivingCompanyBookId")) {
                update.setNull(qdo.receivingCompanyBookId);
            }
            // 收款银行
            if (nullFields.contains("receivingBank")) {
                update.setNull(qdo.receivingBank);
            }
            // 收款账号
            if (nullFields.contains("receivingAccount")) {
                update.setNull(qdo.receivingAccount);
            }
            // 状态
            if (nullFields.contains("state")) {
                update.setNull(qdo.state);
            }
            // 出纳更新时间
            if (nullFields.contains("cashierTime")) {
                update.setNull(qdo.cashierTime);
            }
            // 出纳备注
            if (nullFields.contains("cashierRemark")) {
                update.setNull(qdo.cashierRemark);
            }
            // 应付会计更新时间
            if (nullFields.contains("apAccountantTime")) {
                update.setNull(qdo.apAccountantTime);
            }
            // 应付会计备注
            if (nullFields.contains("apAccountantRemark")) {
                update.setNull(qdo.apAccountantRemark);
            }
            // 财务经理更新时间
            if (nullFields.contains("financeTime")) {
                update.setNull(qdo.financeTime);
            }
            // 财务经理备注
            if (nullFields.contains("financeRemark")) {
                update.setNull(qdo.financeRemark);
            }
            // 实际付款日期
            if (nullFields.contains("actualDate")) {
                update.setNull(qdo.actualDate);
            }
            // 支付时间
            if (nullFields.contains("payTime")) {
                update.setNull(qdo.payTime);
            }
            // 支付状态：S 成功、F 失败、C撤销、D 过期、R 否决 G退票
            if (nullFields.contains("payStatus")) {
                update.setNull(qdo.payStatus);
            }
            // 支付方式：1.网银支付 2.手动支付
            if (nullFields.contains("payMode")) {
                update.setNull(qdo.payMode);
            }
            // 网银提交标识 1.已提交 2.未提交
            if (nullFields.contains("bankFlag")) {
                update.setNull(qdo.bankFlag);
            }
            // 网银提交时间
            if (nullFields.contains("bankTime")) {
                update.setNull(qdo.bankTime);
            }
            // 支付标识：1.已支付 ；2.待支付
            if (nullFields.contains("payFlag")) {
                update.setNull(qdo.payFlag);
            }
            // 支付用途
            if (nullFields.contains("payPurpose")) {
                update.setNull(qdo.payPurpose);
            }
            // 流程实例ID
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 流程审批状态
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            // 提交时间
            if (nullFields.contains("submitTime")) {
                update.setNull(qdo.submitTime);
            }
            // 审批时间
            if (nullFields.contains("approvedTime")) {
                update.setNull(qdo.approvedTime);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    //银企直联回调更新
    public long updateBankCallBack(PaymentSlipPayload payload) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qdo)
                .set(qdo.payStatus, payload.getPayStatus())
                .set(qdo.payTime, payload.getPayTime());
        if (StringUtils.hasText(payload.getState())) {
            jpaUpdateClause.set(qdo.state, payload.getState());
        }
        long result = jpaUpdateClause.where(qdo.id.eq(payload.getId())).execute();
        return result;
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据付款申请单ID查询列表
     *
     * @param paymentApplyId 付款申请单ID
     * @return {@link List}<{@link PaymentSlipVO}>
     */
    public List<PaymentSlipVO> queryListByPaymentApplyId(Long paymentApplyId) {
        JPAQuery<PaymentSlipVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQuerySelect.where(qdo.paymentApplyId.eq(paymentApplyId));
        return jpaQuerySelect.fetch();
    }

    /**
     * 根据付款申请单IDList查询列表
     *
     * @param paymentApplyIdList 付款申请单ID集合
     * @param state              状态
     * @return {@link List}<{@link PaymentSlipVO}>
     */
    public List<PaymentSlipVO> queryByPaymentApplyIdListAndState(List<Long> paymentApplyIdList, String state) {
        JPAQuery<PaymentSlipVO> jpaQuerySelect = jpaQueryFactory.select(Projections.bean(PaymentSlipVO.class,
                qdo.id,
                qdo.paymentAmt,
                qdo.paymentApplyId)).from(qdo);
        jpaQuerySelect.where(qdo.paymentApplyId.in(paymentApplyIdList));
        jpaQuerySelect.where(qdo.state.eq(state));
        jpaQuerySelect.where(qdo.deleteFlag.eq(0));
        return jpaQuerySelect.fetch();
    }
    /**
     * 根据申请单Id删除付款申请单记录
     *
     * @param paymentApplyId 申请单Id
     */
    public void deleteByPaymentApplyId(Long paymentApplyId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.paymentApplyId.eq(paymentApplyId));
        update.execute();
    }
    /**
     * 根据申请单Id修改付款申请单记录状态
     *
     * @param paymentApplyId 申请单Id
     * @param state          状态
     */
    public void updateStatusByPaymentApplyId(Long paymentApplyId, String state) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.state, state)
                .where(qdo.paymentApplyId.eq(paymentApplyId));
        update.execute();
    }

    /**
     * 修改网银用途
     *
     * @param id         主键
     * @param payPurpose 网银用途
     */
    public void updatePayPurpose(Long id, String payPurpose) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.payPurpose, payPurpose)
                .where(qdo.id.eq(id));
        update.execute();
    }

    /**
     * 修改收款账户
     *
     * @param id               主键
     * @param receivingBank    银行名称
     * @param receivingAccount 银行账户
     */
    public void updateReceiveAccount(Long id, String receivingBank, String receivingAccount) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.receivingBank, receivingBank)
                .set(qdo.receivingAccount, receivingAccount)
                .set(qdo.cashierTime, LocalDateTime.now())
                .where(qdo.id.eq(id));
        update.execute();


    }

//    /**
//     * 通过驳回
//     *
//     * @param keys  主键集合
//     * @param state 状态
//     */
//    public void passOrReject(List<Long> keys, String state) {
//        JPAUpdateClause update = jpaQueryFactory.update(qdo)
//                .set(qdo.state, state)
//                .where(qdo.id.in(keys));
//        update.execute();
//    }
//
//    /**
//     * 付款单记录-提交
//     *
//     * @param paySerialNum 流水号
//     * @param state        状态
//     */
//    public void paymentCommit(String paySerialNum, String state) {
//        JPAUpdateClause update = jpaQueryFactory.update(qdo)
//                .set(qdo.state, state)
//                .where(qdo.paySerialNum.in(paySerialNum));
//        update.execute();
//    }

    /**
     * 更新流水号
     *
     * @param generateSeqNum 流水号
     * @param keys           主键集合
     */
    public void updatePaySerialNumByIds(String generateSeqNum, List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.paySerialNum, generateSeqNum)
                .where(qdo.id.in(keys));
        update.execute();
    }

    /**
     * 查询流水号
     */
    public List<String> queryPaySerialNum() {
        List<String> list = Arrays.asList(PurchasePaymentEnum.PaymentSlipStatus.READY.getCode(), PurchasePaymentEnum.PaymentSlipStatus.FINANCE_REJECTED.getCode(), PurchasePaymentEnum.PaymentSlipStatus.CASHIER_TO_SUBMIT.getCode());
        return jpaQueryFactory.selectDistinct(qdo.paySerialNum).
                from(qdo).
                where(qdo.state.in(list)).
                where(qdo.deleteFlag.eq(0)).fetch();
    }

    //更新付款记录的状态
    public void updateStatusByIdList(List<Long> idList, String status, Integer payMode, String payStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.state, status)
                .set(qdo.payMode, payMode)
                .set(qdo.payStatus, payStatus)
                .set(qdo.payTime, LocalDateTime.now())
                .where(qdo.id.in(idList));
        update.execute();
    }

    /**
     * 查询付款单记录是否有除了已付款外的其他状态
     *
     * @param paymentApplyId 付款申请单Id
     * @return 是/否
     */
    public Boolean isOtherStatusByPaymentApplyId(Long paymentApplyId) {
        JPAQuery<Boolean> jpaQuery = jpaQueryFactory.select(
                        qdo.count().gt(0)
                ).from(qdo)
                .where(qdo.deleteFlag.eq(0)
                        .and(qdo.paymentApplyId.eq(paymentApplyId))
                        .and(qdo.state.ne(PurchasePaymentEnum.PaymentSlipStatus.PAID.getCode())));
        return jpaQuery.fetchOne();
    }

    /**
     * 查询付款单记录未支付的金额
     *
     * @param paymentApplyIdSet 付款申请单Id
     * @return result
     */
    public List<Tuple> findNoPayAmtByPaymentApplyIdIn(Set<Long> paymentApplyIdSet) {
        JPAQuery<Tuple> jpaQuery = jpaQueryFactory.select(
                        qdo.paymentApplyId,
                        qdo.paymentAmt.sum()
                ).from(qdo)
                .where(qdo.deleteFlag.eq(0)
                        .and(qdo.paymentApplyId.in(paymentApplyIdSet))
                        .and(qdo.state.ne(PurchasePaymentEnum.PaymentSlipStatus.PAID.getCode())))
                .groupBy(qdo.paymentApplyId);
        return jpaQuery.fetch();
    }

    //更新付款记录单的付款凭证的状态
    public void updatePayJdeById(Long paymentSlipId, String jdeVoucher, String jdePayNo, String payProofStatus, String payProofDigest, LocalDateTime payApprovedTime, String state) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.jdeVoucher, jdeVoucher)
                .set(qdo.jdePayNo, jdePayNo)
                .set(qdo.payProofStatus, payProofStatus)
                //.set(qdo.payApprovedTime,payApprovedTime)
                .set(qdo.payApprovedTime, payApprovedTime)
                .set(qdo.payProofDigest, payProofDigest)
                .set(qdo.state, state)
                .where(qdo.id.eq(paymentSlipId));
        update.execute();
    }

    //更新付款记录单的付款凭证的状态
    public void updatePayVerJdeById(Long paymentSlipId, String payProofNo, String payProofStatus, String state) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.payProofNo, payProofNo)
                .set(qdo.payProofStatus, payProofStatus)
                .set(qdo.state, state)
                .where(qdo.id.eq(paymentSlipId));
        update.execute();
    }
}

