package com.elitesland.tw.tw5.server.prd.shorturl.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.shorturl.payload.ShortUrlRefPayload;
import com.elitesland.tw.tw5.api.prd.shorturl.query.ShortUrlRefQuery;
import com.elitesland.tw.tw5.api.prd.shorturl.vo.ShortUrlRefVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.shorturl.entity.QShortUrlRefDO;
import com.elitesland.tw.tw5.server.prd.shorturl.entity.ShortUrlRefDO;
import com.elitesland.tw.tw5.server.prd.shorturl.repo.ShortUrlRefRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 短链服务关系表
 *
 * @author duwh
 * @date 2023-05-19
 */
@Repository
@RequiredArgsConstructor
public class ShortUrlRefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ShortUrlRefRepo repo;
    private final QShortUrlRefDO qdo = QShortUrlRefDO.shortUrlRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ShortUrlRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ShortUrlRefVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 短链地址
                qdo.shortUrl,
                // 长链地址
                qdo.longUrl,
                // 访问数量
                qdo.visitNum,
                // 拓展1
                qdo.ext1,
                // 拓展2
                qdo.ext2,
                // 拓展3
                qdo.ext3,
                // 拓展4
                qdo.ext4,
                // 拓展5
                qdo.ext5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ShortUrlRefVO> getJpaQueryWhere(ShortUrlRefQuery query) {
        JPAQuery<ShortUrlRefVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ShortUrlRefQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ShortUrlRefQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 短链地址 精确 */
        if (!ObjectUtils.isEmpty(query.getShortUrl())) {
            list.add(qdo.shortUrl.eq(query.getShortUrl()));
        }
        /** 短链地址识别码 精确 */
        if (!ObjectUtils.isEmpty(query.getCode())) {
            list.add(qdo.code.eq(query.getCode()));
        }
        /** 长链地址 精确 */
        if (!ObjectUtils.isEmpty(query.getLongUrl())) {
            list.add(qdo.longUrl.eq(query.getLongUrl()));
        }
        /** 访问数量 精确 */
        if (!ObjectUtils.isEmpty(query.getVisitNum())) {
            list.add(qdo.visitNum.eq(query.getVisitNum()));
        }
        /** 拓展1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 拓展4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 拓展5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ShortUrlRefVO queryByKey(Long id) {
        JPAQuery<ShortUrlRefVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据短链地址识别码查询
     *
     * @param code 短链地址识别码
     * @return 结果
     */
    public ShortUrlRefVO queryByCode(String code) {
        JPAQuery<ShortUrlRefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.code.eq(code));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ShortUrlRefVO> queryListDynamic(ShortUrlRefQuery query) {
        JPAQuery<ShortUrlRefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ShortUrlRefVO> queryPaging(ShortUrlRefQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ShortUrlRefVO> jpaQuery = getJpaQueryWhere(query);
        List<ShortUrlRefVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ShortUrlRefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ShortUrlRefDO save(ShortUrlRefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ShortUrlRefDO> saveAll(List<ShortUrlRefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ShortUrlRefPayload payload) {
        JPAUpdateClause update = getJpaUpdateClause(payload);
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    @Transactional
    public long updateByKeyDynamicNotLogin(ShortUrlRefPayload payload) {
        JPAUpdateClause update = getJpaUpdateClause(payload);
        // 执行修改
        return update.execute();
    }

    private JPAUpdateClause getJpaUpdateClause(ShortUrlRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 短链地址
        if (payload.getShortUrl() != null) {
            update.set(qdo.shortUrl, payload.getShortUrl());
        }
        // 短链地址识别码
        if (payload.getCode() != null) {
            update.set(qdo.code, payload.getCode());
        }
        // 长链地址
        if (payload.getLongUrl() != null) {
            update.set(qdo.longUrl, payload.getLongUrl());
        }
        // 访问数量
        if (payload.getVisitNum() != null) {
            update.set(qdo.visitNum, payload.getVisitNum());
        }
        // 拓展1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 拓展4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 拓展5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 短链地址
            if (nullFields.contains("shortUrl")) {
                update.setNull(qdo.shortUrl);
            }
            // 长链地址
            if (nullFields.contains("longUrl")) {
                update.setNull(qdo.longUrl);
            }
            // 访问数量
            if (nullFields.contains("visitNum")) {
                update.setNull(qdo.visitNum);
            }
            // 拓展1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 拓展4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 拓展5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
        }
        return update;
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

