package com.elitesland.tw.tw5.server.prd.acc.service;


import cn.hutool.extra.spring.SpringUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimSettingCheckPayload;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimSettingPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccBusinessItemQuery;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimSettingQuery;
import com.elitesland.tw.tw5.api.prd.acc.service.*;
import com.elitesland.tw.tw5.api.prd.acc.vo.*;
import com.elitesland.tw.tw5.api.prd.inv.query.InvInvoiceQuery;
import com.elitesland.tw.tw5.api.prd.inv.service.InvInvoiceService;
import com.elitesland.tw.tw5.api.prd.inv.vo.InvInvoiceVO;
import com.elitesland.tw.tw5.api.prd.my.service.VacationService;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.acc.common.functionEnum.AccReimSettingTypeEnum;
import com.elitesland.tw.tw5.server.prd.acc.convert.AccReimSettingConvert;
import com.elitesland.tw.tw5.server.prd.acc.dao.AccReimSettingDAO;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccReimSettingDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimSettingRepo;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.MessageNoticeScopeEnum;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 报销配置主表
 *
 * @author zoey
 * @date 2024-04-08
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class AccReimSettingServiceImpl extends BaseServiceImpl implements AccReimSettingService {

    private final AccReimSettingRepo accReimSettingRepo;
    private final AccReimSettingDAO accReimSettingDAO;
    private final CacheUtil cacheUtil;
    private final AccBusinessItemService accBusinessItemService;
    private final InvInvoiceService invInvoiceService;
    private final VacationService vacationService;

    @Override
    public PagingVO<AccReimSettingVO> queryPaging(AccReimSettingQuery query) {
        PagingVO<AccReimSettingVO> accReimSettingVOPagingVO = accReimSettingDAO.queryPaging(query);
        List<AccReimSettingVO> accReimSettingVOS = accReimSettingVOPagingVO.getRecords();
        if (!CollectionUtils.isEmpty(accReimSettingVOS)) {
            translateList(accReimSettingVOS);
        }
        return accReimSettingVOPagingVO;
    }

    @Override
    public List<AccReimSettingVO> queryListDynamic(AccReimSettingQuery query) {
        List<AccReimSettingVO> accReimSettingVOS = accReimSettingDAO.queryListDynamic(query);
        if (!CollectionUtils.isEmpty(accReimSettingVOS)) {
            translateList(accReimSettingVOS);
        }
        return accReimSettingVOS;
    }

    @Override
    public AccReimSettingVO queryByKey(Long key) {
        AccReimSettingDO entity = accReimSettingRepo.findById(key).orElseGet(AccReimSettingDO::new);
        Assert.notNull(entity.getId(), "不存在");
        AccReimSettingVO vo = AccReimSettingConvert.INSTANCE.toVo(entity);
        translateList(Collections.singletonList(vo));
        return vo;
    }


    private void translateList(List<AccReimSettingVO> voList) {
        for (AccReimSettingVO vo : voList) {
            String name = getApplicationSourceName(vo.getId(), vo.getApplicationScope(), vo.getApplicationSource());
            vo.setApplicationSourceName(name);

            if (StringUtils.hasText(vo.getExpenseCompany())) {
                // 翻译公司
                List<String> expenseCompanyNames = new ArrayList<>();
                for (String expenseCompany : vo.getExpenseCompany().split(",")) {
                    String expenseCompanyName = cacheUtil.getCompanyNameByBookId(Long.parseLong(expenseCompany));
                    expenseCompanyNames.add(expenseCompanyName);
                }
                String expenseCompanyName = String.join(",", expenseCompanyNames);
                vo.setExpenseCompanyName(expenseCompanyName);
            }

            if (StringUtils.hasText(vo.getBusAccItemId())) {
                // 翻译核算科目
                List<String> busAccItemNames = new ArrayList<>();
                AccBusinessItemQuery accBusinessItemQuery = new AccBusinessItemQuery();
                List<Long> idList = Arrays.stream(vo.getBusAccItemId().split(",")).map(Long::parseLong).collect(Collectors.toList());
                accBusinessItemQuery.setIdList(idList);
                List<AccBusinessItemVO> accBusinessItemVOS = accBusinessItemService.queryListDynamic(accBusinessItemQuery);
                if (!CollectionUtils.isEmpty(accBusinessItemVOS)) {
                    String accItemCodeStr = accBusinessItemVOS.stream().map(AccBusinessItemVO::getBusinessCode).collect(Collectors.joining(","));
                    String accItemNameStr = accBusinessItemVOS.stream().map(AccBusinessItemVO::getBusinessCode).collect(Collectors.joining(","));
                    vo.setBusAccItemCode(accItemCodeStr);
                    vo.setBusAccItemName(accItemNameStr);
                }
            }

            if (StringUtils.hasText(vo.getWhiteList())) {
                // 翻译白名单
                List<String> userNames = new ArrayList<>();
                for (String sourceId : vo.getWhiteList().split(",")) {
                    String userName = cacheUtil.getUserName(Long.parseLong(sourceId));
                    userNames.add(userName);
                }
                String userName = String.join(",", userNames);
                vo.setWhiteListName(userName);
            }
            if (StringUtils.hasText(vo.getReimDocType())) {
                // 报销单据类型
                List<String> reimDocTypes = new ArrayList<>();
                for (String reimDocType : vo.getReimDocType().split(",")) {
                    String reimDocTypeName = cacheUtil.transferSystemSelection("ACC:REIM_PROC_KEY", reimDocType);
                    reimDocTypes.add(reimDocTypeName);
                }
                String reimDocType = String.join(",", reimDocTypes);
                vo.setReimDocTypeName(reimDocType);
            }
        }
    }

    public String getApplicationSourceName(Long settingId, String applicationScope, String applicationSource) {
        String name = applicationSource;
        if (StringUtils.hasText(applicationSource)) {
            if (applicationScope.equals(MessageNoticeScopeEnum.APPOINT_ORG.getCode())) {
                name = accReimSettingDAO.getOrgNames(settingId);
            }
            if (applicationScope.equals(MessageNoticeScopeEnum.APPIONT_PEOPLE.getCode())) {
                List<String> userNames = new ArrayList<>();
                for (String sourceId : applicationSource.split(",")) {
                    String userName = cacheUtil.getUserName(Long.parseLong(sourceId));
                    userNames.add(userName);
                }
                name = String.join(",", userNames);
            }
            if (applicationScope.equals(MessageNoticeScopeEnum.APPOINT_ROLE.getCode())) {
                name = accReimSettingDAO.getRoleNames(settingId);
            }
        }
        return name;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public AccReimSettingVO insert(AccReimSettingPayload payload) {
//        checkRepeat(payload);
        AccReimSettingDO entityDo = AccReimSettingConvert.INSTANCE.toDo(payload);
        return AccReimSettingConvert.INSTANCE.toVo(accReimSettingRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public AccReimSettingVO update(AccReimSettingPayload payload) {
//        checkRepeat(payload);
        AccReimSettingDO entity = accReimSettingRepo.findById(payload.getId()).orElseGet(AccReimSettingDO::new);
        Assert.notNull(entity.getId(), "不存在");
        AccReimSettingDO entityDo = AccReimSettingConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return AccReimSettingConvert.INSTANCE.toVo(accReimSettingRepo.save(entity));
    }

    /**
     * 同一个配置类型，同一个报销单据类型，同一家公司,同一个适用范围不能出现重复的配置
     *
     * @param payload
     */
    @Override
    public void checkRepeat(AccReimSettingPayload payload) {
        // 重复校验
        Boolean repeatFlag = false;
        // 第一步先查类型公司一样的情况下，有没有匹配全员的配置，如果有，直接抛异常
        for (String reimDocType : payload.getReimDocType().split(",")) {
            if (payload.getId() == null) {
                for (String expenseCompany : payload.getExpenseCompany().split(",")) {
                    int count = accReimSettingRepo.countByReimSettingTypeAndReimDocTypeLikeAndExpenseCompanyLikeAndApplicationScopeAndDeleteFlag(payload.getReimSettingType(), "%" + reimDocType + "%", "%" + expenseCompany + "%", MessageNoticeScopeEnum.WHOLE.getCode(), 0);
                    if (count > 0) {
                        repeatFlag = true;
                        break;
                    }
                }
            } else {
                for (String expenseCompany : payload.getExpenseCompany().split(",")) {
                    int count = accReimSettingRepo.countByReimSettingTypeAndReimDocTypeLikeAndExpenseCompanyLikeAndApplicationScopeAndIdNotAndDeleteFlag(payload.getReimSettingType(), "%" + reimDocType + "%", "%" + expenseCompany + "%", MessageNoticeScopeEnum.WHOLE.getCode(), payload.getId(), 0);
                    if (count > 0) {
                        repeatFlag = true;
                        break;
                    }
                }
            }
        }
        if (repeatFlag) {
            throw TwException.error("", "配置重复，相同单据类型，相同适用公司已经存在适用于全员的报销配置!");
        } else {
            if (payload.getApplicationSource() != null) {
                for (String reimDocType : payload.getReimDocType().split(",")) {
                    for (String source : payload.getApplicationSource().split(",")) {
                        if (payload.getId() == null) {
                            for (String expenseCompany : payload.getExpenseCompany().split(",")) {
                                int count = accReimSettingRepo.countByReimSettingTypeAndReimDocTypeLikeAndExpenseCompanyLikeAndApplicationScopeAndApplicationSourceLikeAndDeleteFlag(payload.getReimSettingType(), "%" + reimDocType + "%", "%" + expenseCompany + "%", payload.getApplicationScope(), "%" + source + "%", 0);
                                if (count > 0) {
                                    repeatFlag = true;
                                    break;
                                }
                            }
                        } else {
                            for (String expenseCompany : payload.getExpenseCompany().split(",")) {
                                int count = accReimSettingRepo.countByReimSettingTypeAndReimDocTypeLikeAndExpenseCompanyLikeAndApplicationScopeAndApplicationSourceLikeAndIdNotAndDeleteFlag(payload.getReimSettingType(), "%" + reimDocType + "%", "%" + expenseCompany + "%", payload.getApplicationScope(), "%" + source + "%", payload.getId(), 0);
                                if (count > 0) {
                                    repeatFlag = true;
                                    break;
                                }
                            }
                        }
                    }
                    ;
                }
                if (repeatFlag) {
                    throw TwException.error("", "配置重复，相同单据类型，相同适用公司已经存在当前适用人员的报销配置!");
                }
            }
        }
    }

    /**
     * 同一个配置类型，同一个报销单据类型，同一家公司,同一个适用范围的配置
     * @param query
     */
//    @Override
//    public AccReimSettingVO queryBaseSetting(AccReimSettingQuery query){
//        String[] reimDocTypeArray = query.getReimDocType().split(",");
//        AccReimSettingVO accReimSettingVO = new AccReimSettingVO();
//        // 第一步先查类型公司一样的情况下，有没有匹配全员的配置，如果有，直接抛异常
//        for (String reimDocType : reimDocTypeArray) {
//            query.setReimDocType(reimDocType);
//            query.setApplicationScope(MessageNoticeScopeEnum.WHOLE.getCode());
//            List<AccReimSettingVO> accReimSettingVOS = queryListDynamic(query);
//            if(!CollectionUtils.isEmpty(accReimSettingVOS)){
//                accReimSettingVO = accReimSettingVOS.get(0);
//                break;
//            }
//        }
//        if(ObjectUtils.isEmpty(accReimSettingVO) || accReimSettingVO.getId()==null){
//            for (String reimDocType : reimDocTypeArray) {
//                // 查询部门级的配置
//                query.setReimDocType(reimDocType);
//                query.setApplicationScope(MessageNoticeScopeEnum.APPOINT_ORG.getCode());
//                // 查询报销申请人的部门
//                Long applicationUser = query.getApplicationUser();
//                Long applicationUserOrgId = cacheUtil.getDefaultOrgIdByUserId(applicationUser);
//                if(applicationUserOrgId!=null){
//                    query.setApplicationSource(applicationUserOrgId.toString());
//                    List<AccReimSettingVO> accReimSettingVOS = queryListDynamic(query);
//                    if(!CollectionUtils.isEmpty(accReimSettingVOS)){
//                        accReimSettingVO = accReimSettingVOS.get(0);
//                        break;
//                    }
//                }
//            }
//
//            for (String reimDocType : reimDocTypeArray) {
//                // 指定人级别配置
//                query.setReimDocType(reimDocType);
//                query.setApplicationScope(MessageNoticeScopeEnum.APPIONT_PEOPLE.getCode());
//                Long applicationUser = query.getApplicationUser();
//                query.setApplicationSource(applicationUser.toString());
//                List<AccReimSettingVO> accReimSettingVOS = queryListDynamic(query);
//                if(!CollectionUtils.isEmpty(accReimSettingVOS)){
//                    accReimSettingVO = accReimSettingVOS.get(0);
//                    break;
//                }
//            }
//        }
//        return accReimSettingVO;
//    }

    /**
     * 同一个配置类型，同一个报销单据类型，同一家公司,同一个适用范围的配置
     *
     * @param query
     */
    @Override
    public AccReimSettingVO queryBaseSetting(AccReimSettingQuery query) {
        AccReimSettingVO accReimSettingVO = null;
        // 第一步先查类型公司一样的情况下，有没有匹配全员的配置
        query.setApplicationScope(MessageNoticeScopeEnum.WHOLE.getCode());
        query.setApplicationSource(null);
        List<AccReimSettingVO> accReimSettingVOS = queryListDynamic(query);
        if (!CollectionUtils.isEmpty(accReimSettingVOS)) {
            accReimSettingVO = accReimSettingVOS.get(0);
        }
//        if(ObjectUtils.isEmpty(accReimSettingVO) || accReimSettingVO.getId()==null) {
        // 查询部门级的配置
        query.setApplicationScope(MessageNoticeScopeEnum.APPOINT_ORG.getCode());
        // 查询报销申请人的部门
        Long applicationUser = query.getApplicationUser();
        Long applicationUserOrgId = cacheUtil.getDefaultOrgIdByUserId(applicationUser);
        if (applicationUserOrgId != null) {
            query.setApplicationSource(applicationUserOrgId.toString());
            accReimSettingVOS = queryListDynamic(query);
            if (!CollectionUtils.isEmpty(accReimSettingVOS)) {
                accReimSettingVO = accReimSettingVOS.get(0);
            }
        }
        // 指定人级别配置
        query.setApplicationScope(MessageNoticeScopeEnum.APPIONT_PEOPLE.getCode());
        query.setApplicationSource(applicationUser.toString());
        accReimSettingVOS = queryListDynamic(query);
        if (!CollectionUtils.isEmpty(accReimSettingVOS)) {
            accReimSettingVO = accReimSettingVOS.get(0);
        }
//        }
        return accReimSettingVO;
    }

    @Override
    public Map<String, AccReimSettingVO> queryBaseSettings(AccReimSettingQuery query) {
        Map<String, AccReimSettingVO> accReimSettingVOMap = new HashMap<>();
        // 遍历枚举类
        for (AccReimSettingTypeEnum settingType : AccReimSettingTypeEnum.values()) {
            query.setReimSettingType(settingType.getCode());
            AccReimSettingVO accReimSettingVO = queryBaseSetting(query);
            accReimSettingVOMap.put(settingType.getCode(), accReimSettingVO);
        }
        return accReimSettingVOMap;
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(AccReimSettingPayload payload) {
        AccReimSettingDO entity = accReimSettingRepo.findById(payload.getId()).orElseGet(AccReimSettingDO::new);
        Assert.notNull(entity.getId(), "不存在");
        long result = accReimSettingDAO.updateByKeyDynamic(payload);
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            accReimSettingDAO.deleteSoft(keys);
        }
    }

    @Override
    public Map<String, Object> ruleCheck(AccReimSettingCheckPayload reimSettingCheckPayload) {
        Map<String, Object> checkResult = new HashMap<>();
        Map<String, Long> keys = reimSettingCheckPayload.getKeys();
        String invoiceNo = reimSettingCheckPayload.getInvoiceNo();
        Long reimUserId = reimSettingCheckPayload.getReimUserId();
        String expenseDate = reimSettingCheckPayload.getExpenseDate();
        BigDecimal reimAmt = reimSettingCheckPayload.getReimAmt();
        String expensePlaceGrade = reimSettingCheckPayload.getExpensePlaceGrade();
        Long expenseCompanyId = reimSettingCheckPayload.getExpenseCompanyId();
        List<InvInvoiceVO> invInvoiceVOS = new ArrayList<>();
        if (StringUtils.hasText(invoiceNo)) {
            List<String> invNoList = Arrays.stream(invoiceNo.split(",")).collect(Collectors.toList());
            InvInvoiceQuery invInvoiceQuery = new InvInvoiceQuery();
            invInvoiceQuery.setInvoiceNoIn(invNoList);
            invInvoiceVOS = invInvoiceService.queryListDynamic(invInvoiceQuery);
        }
        for (String settingType : keys.keySet()) {
            Long key = keys.get(settingType);
            if (reimUserId == null) {
                throw TwException.error("", "报销人不能为空!");
            }
            switch (AccReimSettingTypeEnum.valueOf(settingType)) {
                case SERIATE://连号
                    AccReimSettingContinuousService continuousService = SpringUtil.getBean(AccReimSettingContinuousService.class);
                    if (invoiceNo == null) {
                        throw TwException.error("", "查询连号配置必须有发票！");
                    }
                    log.debug("连号校验开始：：：：：：：：：：：：：：：" + LocalDateTime.now());
                    AccReimSettingContinuousVO accReimSettingContinuousVO = continuousService.ruleCheck(key, invInvoiceVOS, reimUserId);
                    log.debug("连号校验结束：：：：：：：：：：：：：：：：：：" + LocalDateTime.now());
                    checkResult.put(settingType, accReimSettingContinuousVO);
                    break;
                case OVERDUE://逾期打折
                    AccReimSettingOverdueService overdueService = SpringUtil.getBean(AccReimSettingOverdueService.class);
                    AccReimSettingOverdueVO accReimSettingOverdueVO = overdueService.ruleCheck(key, expenseDate, invoiceNo, reimUserId);
                    checkResult.put(settingType, accReimSettingOverdueVO);
                    break;
                case LODGE_OVERAGE://住宿额度配置
                    AccReimSettingLimitService limitService = SpringUtil.getBean(AccReimSettingLimitService.class);
                    if (reimAmt == null) {
                        throw TwException.error("", "查询住宿额度配置，报销金额不能为空!");
                    }
                    AccReimSettingLimitVO accReimSettingLimitVO = limitService.ruleCheck(key, reimUserId, reimAmt, expensePlaceGrade);
                    checkResult.put(settingType, accReimSettingLimitVO);
                    break;
                case INVOICE_TITLE://抬头配置
                    AccReimSettingTitleService titleService = SpringUtil.getBean(AccReimSettingTitleService.class);
                    if (invoiceNo == null) {
                        throw TwException.error("", "发票抬头配置，发票号码和费用承担公司都不能为空!");
                    }
                    log.debug("抬头校验开始：：：：：：：：：：：：：：：" + LocalDateTime.now());
                    AccReimSettingTitleVO accReimSettingTitleVO = titleService.ruleCheck(key, invInvoiceVOS, reimUserId, expenseCompanyId);
                    log.debug("抬头校验结束：：：：：：：：：：：：：：：" + LocalDateTime.now());
                    checkResult.put(settingType, accReimSettingTitleVO);
                    break;
                case HOLIDAY_CHECK://节假日校验
                    LocalDate compareDate = LocalDate.parse(expenseDate);
                    // 判断费用发生日期是否是节假日
                    List<LocalDate> vacationDay = vacationService.findVacationDay(compareDate, compareDate);
                    if (!CollectionUtils.isEmpty(vacationDay)) {
                        //当前日期是节假日
                        checkResult.put(settingType, true);
                    } else {
                        checkResult.put(settingType, false);
                    }
                default:
            }
        }
        return checkResult;
    }

}
