package com.elitesland.tw.tw5.server.prd.copartner.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.copartner.payload.PerformanceReadmeConfigPayload;
import com.elitesland.tw.tw5.api.prd.copartner.query.PerformanceReadmeConfigQuery;
import com.elitesland.tw.tw5.api.prd.copartner.vo.PerformanceReadmeConfigVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.copartner.entity.PerformanceReadmeConfigDO;
import com.elitesland.tw.tw5.server.prd.copartner.entity.QPerformanceReadmeConfigDO;
import com.elitesland.tw.tw5.server.prd.copartner.repo.PerformanceReadmeConfigRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 合伙人绩效自述配置
 *
 * @author carl
 * @date 2024-06-24
 */
@Repository
@RequiredArgsConstructor
public class PerformanceReadmeConfigDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PerformanceReadmeConfigRepo repo;
    private final QPerformanceReadmeConfigDO qdo = QPerformanceReadmeConfigDO.performanceReadmeConfigDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PerformanceReadmeConfigVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PerformanceReadmeConfigVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                //发起人
                qdo.applyUserId,
                //发起时间
                qdo.applyTime,
                //完成时间
                qdo.finishTime,
                // 名称
                qdo.configName,
                // 考核周期
                qdo.configCycle,
                // 开始日期
                qdo.startDate,
                // 结束日期
                qdo.endDate,
                // 状态
                qdo.configStatus
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PerformanceReadmeConfigVO> getJpaQueryWhere(PerformanceReadmeConfigQuery query) {
        JPAQuery<PerformanceReadmeConfigVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PerformanceReadmeConfigQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PerformanceReadmeConfigQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getConfigName())) {
            list.add(qdo.configName.like(SqlUtil.toSqlLikeString(query.getConfigName())));
        }
        /** 考核周期 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigCycle())) {
            list.add(qdo.configCycle.eq(query.getConfigCycle()));
        }
        /** 开始日期 大于等于 */
        if (!ObjectUtils.isEmpty(query.getStartDate())) {
            list.add(qdo.startDate.goe(query.getStartDate()));
        }
        /** 结束日期 小于等于 */
        if (!ObjectUtils.isEmpty(query.getEndDate())) {
            list.add(qdo.endDate.loe(query.getEndDate()));
        }
        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigStatus())) {
            list.add(qdo.configStatus.eq(query.getConfigStatus()));
        }
        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyUserId())) {
            list.add(qdo.applyUserId.eq(query.getApplyUserId()));
        }

        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PerformanceReadmeConfigVO queryByKey(Long id) {
        JPAQuery<PerformanceReadmeConfigVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param ids 主键
     * @return 结果
     */
    public List<PerformanceReadmeConfigVO> queryByKeys(List<Long> ids) {
        JPAQuery<PerformanceReadmeConfigVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PerformanceReadmeConfigVO> queryListDynamic(PerformanceReadmeConfigQuery query) {
        JPAQuery<PerformanceReadmeConfigVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PerformanceReadmeConfigVO> queryPaging(PerformanceReadmeConfigQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PerformanceReadmeConfigVO> jpaQuery = getJpaQueryWhere(query);
        List<PerformanceReadmeConfigVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PerformanceReadmeConfigVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PerformanceReadmeConfigDO save(PerformanceReadmeConfigDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PerformanceReadmeConfigDO> saveAll(List<PerformanceReadmeConfigDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PerformanceReadmeConfigPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 名称
        if (payload.getConfigName() != null) {
            update.set(qdo.configName, payload.getConfigName());
        }
        // 考核周期
        if (payload.getConfigCycle() != null) {
            update.set(qdo.configCycle, payload.getConfigCycle());
        }
        // 开始日期
        if (payload.getStartDate() != null) {
            update.set(qdo.startDate, payload.getStartDate());
        }
        // 结束日期
        if (payload.getEndDate() != null) {
            update.set(qdo.endDate, payload.getEndDate());
        }
        // 状态
        if (payload.getConfigStatus() != null) {
            update.set(qdo.configStatus, payload.getConfigStatus());
        }
        // 发起人
        if (payload.getApplyUserId() != null) {
            update.set(qdo.applyUserId, payload.getApplyUserId());
        }
        // 发起时间
        if (payload.getApplyTime() != null) {
            update.set(qdo.applyTime, payload.getApplyTime());
        }
        // 完成时间
        if (payload.getFinishTime() != null) {
            update.set(qdo.finishTime, payload.getFinishTime());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 名称
            if (nullFields.contains("configName")) {
                update.setNull(qdo.configName);
            }
            // 考核周期
            if (nullFields.contains("configCycle")) {
                update.setNull(qdo.configCycle);
            }
            // 开始日期
            if (nullFields.contains("startDate")) {
                update.setNull(qdo.startDate);
            }
            // 结束日期
            if (nullFields.contains("endDate")) {
                update.setNull(qdo.endDate);
            }
            // 状态
            if (nullFields.contains("configStatus")) {
                update.setNull(qdo.configStatus);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

