package com.elitesland.tw.tw5.server.prd.partner.common.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.ab.payload.PrdAbAccountPayload;
import com.elitesland.tw.tw5.api.prd.partner.common.payload.BookAccountPayload;
import com.elitesland.tw.tw5.api.prd.partner.common.query.BookAccountQuery;
import com.elitesland.tw.tw5.api.prd.partner.common.vo.BookAccountVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.BookAccountDO;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBookAccountDO;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBusinessPartnerDO;
import com.elitesland.tw.tw5.server.prd.partner.common.repo.BookAccountRepo;
import com.elitesland.tw.tw5.server.prd.partner.constants.BusinessInsideOrOutSideEnum;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 业务伙伴-账户信息
 *
 * @author wangly
 * @date 2023-05-16
 */
@Repository
@RequiredArgsConstructor
public class BookAccountDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BookAccountRepo repo;
    private final QBookAccountDO qdo = QBookAccountDO.bookAccountDO;
    private final QBusinessPartnerDO qBusinessPartnerDO = QBusinessPartnerDO.businessPartnerDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BookAccountVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BookAccountVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                //qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 地址薄id
                qdo.bookId,
                // 账户类型 udc[prd:ab:account_type]
                qdo.accountType,
                // 账户用途 udc[prd:ab:account_usage]
                qdo.accountUsage,
                // 银行 udc[prd:ab:bank_name]
                qdo.depositBank,
                // 开户地
                qdo.depositCity,
                // 开户网点
                qdo.depositBankOutlet,
                // 户名
                qdo.accountName,
                // 账户号
                qdo.accountNo,
                // 币种 udc[SYSTEM_BASIC:CURRENCY]
                qdo.currency,
                // 是否为默认账户
                qdo.isDefault
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BookAccountVO> getJpaQueryWhere(BookAccountQuery query) {
        JPAQuery<BookAccountVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
//            // 动态排序
//            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        //默认值置顶
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BookAccountQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BookAccountQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 地址薄id 精确 */
        if (!ObjectUtils.isEmpty(query.getBookId())) {
            list.add(qdo.bookId.eq(query.getBookId()));
        }
        /** 账户类型 udc[prd:ab:account_type] 精确 */
        if (!ObjectUtils.isEmpty(query.getAccountType())) {
            list.add(qdo.accountType.eq(query.getAccountType()));
        }
        /** 账户用途 udc[prd:ab:account_usage] */
        if (!ObjectUtils.isEmpty(query.getAccountUsage())) {
            list.add(qdo.accountUsage.eq(query.getAccountUsage()));
        }
        /** 银行 udc[prd:ab:bank_name] 精确 */
        if (!ObjectUtils.isEmpty(query.getDepositBank())) {
            list.add(qdo.depositBank.eq(query.getDepositBank()));
        }
        /** 开户地 精确 */
        if (!ObjectUtils.isEmpty(query.getDepositCity())) {
            list.add(qdo.depositCity.eq(query.getDepositCity()));
        }
        /** 开户网点 精确 */
        if (!ObjectUtils.isEmpty(query.getDepositBankOutlet())) {
            list.add(qdo.depositBankOutlet.eq(query.getDepositBankOutlet()));
        }
        /** 户名 精确 */
        if (!ObjectUtils.isEmpty(query.getAccountName())) {
            list.add(qdo.accountName.eq(query.getAccountName()));
        }
        /** 账户号 精确 */
        if (!ObjectUtils.isEmpty(query.getAccountNo())) {
            list.add(qdo.accountNo.eq(query.getAccountNo()));
        }
        /** 币种 udc[SYSTEM_BASIC:CURRENCY] 精确 */
        if (!ObjectUtils.isEmpty(query.getCurrency())) {
            list.add(qdo.currency.eq(query.getCurrency()));
        }
        /** 是否为默认账户  精确 */
        if (!ObjectUtils.isEmpty(query.getIsDefault())) {
            list.add(qdo.isDefault.eq(query.getIsDefault()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BookAccountVO queryByKey(Long id) {
        JPAQuery<BookAccountVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据bookId查询默认
     *
     * @param bookId bookId
     * @return {@link BookAccountVO}
     */
    public BookAccountVO queryDefaultByBookId(Long bookId) {
        JPAQuery<BookAccountVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.bookId.eq(bookId));
        jpaQuery.where(qdo.isDefault.eq(Boolean.TRUE));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据bookIdList查询
     *
     * @param bookIdList bookIdList
     * @return {@link PrdAbAccountPayload}
     */
    public List<PrdAbAccountPayload> queryTwBookAccountListByBookId(List<Long> bookIdList) {
        JPAQuery<PrdAbAccountPayload> jpaQuery = jpaQueryFactory.select(Projections.bean(PrdAbAccountPayload.class,
                qdo.id,
                qdo.bookId,
                qdo.remark,
                qdo.createUserId,
                //qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 地址薄id
                qdo.bookId,
                // 账户类型 udc[prd:ab:account_type]
                qdo.accountType.as("accountType"),
                // 账户用途 udc[prd:ab:account_usage]
                qdo.accountUsage,
                // 银行 udc[prd:ab:bank_name]
                qdo.depositBank.as("depositBank"),
                // 开户地
                qdo.depositCity.as("depositCity"),
                // 开户网点
                qdo.depositBankOutlet.as("depositBankOutlet"),
                // 户名
                qdo.accountName.as("accountName"),
                // 账户号
                qdo.accountNo.as("accountNo"),
                // 币种 udc[SYSTEM_BASIC:CURRENCY]
                qdo.currency.as("currency"),
                // 是否为默认账户
                qdo.isDefault.as("defaultFlag")
        )).from(qdo);
        jpaQuery.where(qdo.bookId.in(bookIdList));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BookAccountVO> queryListDynamic(BookAccountQuery query) {
        JPAQuery<BookAccountVO> jpaQuery = getJpaQueryWhere(query);
        OrderItem sortNoOrderItem = OrderItem.desc("isDefault");
        OrderItem createTimeOrderItem = OrderItem.desc("createTime");
        List<OrderItem> OrderItemList = new ArrayList<>();
        OrderItemList.add(sortNoOrderItem);
        OrderItemList.add(createTimeOrderItem);
        query.setOrders(OrderItemList);
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BookAccountVO> queryPaging(BookAccountQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<BookAccountVO> jpaQuery = getJpaQueryWhere(query);
        List<BookAccountVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BookAccountVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BookAccountDO save(BookAccountDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BookAccountDO> saveAll(List<BookAccountDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BookAccountPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 地址薄id
        if (payload.getBookId() != null) {
            update.set(qdo.bookId, payload.getBookId());
        }
        // 账户类型 udc[prd:ab:account_type]
        if (payload.getAccountType() != null) {
            update.set(qdo.accountType, payload.getAccountType());
        }
        // 银行 udc[prd:ab:bank_name]
        if (payload.getDepositBank() != null) {
            update.set(qdo.depositBank, payload.getDepositBank());
        }
        // 开户地
        if (payload.getDepositCity() != null) {
            update.set(qdo.depositCity, payload.getDepositCity());
        }
        // 开户网点
        if (payload.getDepositBankOutlet() != null) {
            update.set(qdo.depositBankOutlet, payload.getDepositBankOutlet());
        }
        // 户名
        if (payload.getAccountName() != null) {
            update.set(qdo.accountName, payload.getAccountName());
        }
        // 账户号
        if (payload.getAccountNo() != null) {
            update.set(qdo.accountNo, payload.getAccountNo());
        }
        // 币种 udc[SYSTEM_BASIC:CURRENCY]
        if (payload.getCurrency() != null) {
            update.set(qdo.currency, payload.getCurrency());
        }
        // 是否为默认账户
        if (payload.getIsDefault() != null) {
            update.set(qdo.isDefault, payload.getIsDefault());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 地址薄id
            if (nullFields.contains("bookId")) {
                update.setNull(qdo.bookId);
            }
            // 账户类型 udc[prd:ab:account_type]
            if (nullFields.contains("accountType")) {
                update.setNull(qdo.accountType);
            }
            // 银行 udc[prd:ab:bank_name]
            if (nullFields.contains("depositBank")) {
                update.setNull(qdo.depositBank);
            }
            // 开户地
            if (nullFields.contains("depositCity")) {
                update.setNull(qdo.depositCity);
            }
            // 开户网点
            if (nullFields.contains("depositBankOutlet")) {
                update.setNull(qdo.depositBankOutlet);
            }
            // 户名
            if (nullFields.contains("accountName")) {
                update.setNull(qdo.accountName);
            }
            // 账户号
            if (nullFields.contains("accountNo")) {
                update.setNull(qdo.accountNo);
            }
            // 币种 udc[SYSTEM_BASIC:CURRENCY]
            if (nullFields.contains("currency")) {
                update.setNull(qdo.currency);
            }
            // 是否为默认账户
            if (nullFields.contains("isDefault")) {
                update.setNull(qdo.isDefault);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据bookId查询列表
     *
     * @param bookId bookId
     * @return {@link BookAccountVO}
     */
    public List<BookAccountDO> queryByBookId(Long bookId) {
        JPAQuery<BookAccountDO> jpaQuery = jpaQueryFactory.select(qdo).from(qdo);
        jpaQuery.where(qdo.bookId.eq(bookId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 逻辑删除-通过地址薄id
     *
     * @param bookId 地址薄id
     * @return 删除的行数
     */
    public long deleteByBookId(Long bookId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.bookId.eq(bookId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据bookIdList查询银行账号
     *
     * @param bookIdList bookIdList
     * @return {@link List< BookAccountVO >}
     */
    public List<BookAccountVO> queryByBookIdList(List<Long> bookIdList) {
        return getJpaQuerySelect().where(qdo.bookId.in(bookIdList).and(qdo.deleteFlag.eq(0))).fetch();
    }

    /**
     * 根据idList查询银行账号
     *
     * @param idList idList
     * @return {@link List< BookAccountVO >}
     */
    public List<BookAccountVO> queryByIdList(List<Long> idList) {
        return getJpaQuerySelect().where(qdo.id.in(idList).and(qdo.deleteFlag.eq(0))).fetch();
    }

    //查询内部公司的银行下拉
    public List<BookAccountVO> queryInternalList(BookAccountQuery bookAccountQuery) {
        JPAQuery<BookAccountVO> jpaQuery = jpaQueryFactory.select(Projections.bean(BookAccountVO.class,
                        qdo.id,
                        qdo.remark,
                        qdo.createUserId,
                        //qdo.creator,
                        qdo.createTime,
                        //qdo.modifyUserId,
                        //qdo.updater,
                        //qdo.modifyTime,
                        // 地址薄id
                        qdo.bookId,
                        // 账户类型 udc[prd:ab:account_type]
                        qdo.accountType,
                        // 账户用途 udc[prd:ab:account_usage]
                        qdo.accountUsage,
                        // 银行 udc[prd:ab:bank_name]
                        qdo.depositBank,
                        // 开户地
                        qdo.depositCity,
                        // 开户网点
                        qdo.depositBankOutlet,
                        // 户名
                        qdo.accountName,
                        // 账户号
                        qdo.accountNo,
                        // 币种 udc[SYSTEM_BASIC:CURRENCY]
                        qdo.currency,
                        // 是否为默认账户
                        qdo.isDefault,
                        // 名称
                        qBusinessPartnerDO.partnerName.as("bookName")
                )).from(qdo).leftJoin(qBusinessPartnerDO).on(qBusinessPartnerDO.bookId.eq(qdo.bookId)).
                where(qdo.deleteFlag.eq(0)).
                where(qBusinessPartnerDO.deleteFlag.eq(0)).
                where(qBusinessPartnerDO.insideOrOutSide.eq(BusinessInsideOrOutSideEnum.INSIDE.getCode()));
        jpaQuery.where(where(bookAccountQuery));
        return jpaQuery.fetch();
    }
}

