package com.elitesland.tw.tw5.server.prd.partner.common.dao;


import cn.hutool.extra.pinyin.PinyinUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.partner.common.payload.BusinessPartnerPayload;
import com.elitesland.tw.tw5.api.prd.partner.common.query.BusinessPartnerQuery;
import com.elitesland.tw.tw5.api.prd.partner.common.vo.*;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.BusinessPartnerDO;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBookAddressDO;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBookContactsDO;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBusinessPartnerDO;
import com.elitesland.tw.tw5.server.prd.partner.common.repo.BusinessPartnerRepo;
import com.elitesland.tw.tw5.server.prd.partner.constants.BusinessInsideOrOutSideEnum;
import com.elitesland.tw.tw5.server.prd.partner.constants.BusinessPartnerIdentityEnum;
import com.elitesland.tw.tw5.server.prd.partner.constants.BusinessPartnerStageEnum;
import com.elitesland.tw.tw5.server.prd.partner.identity.entity.QBusinessCollaboratePartnerInfoDO;
import com.elitesland.tw.tw5.server.prd.partner.identity.entity.QBusinessCustomerInfoDO;
import com.elitesland.tw.tw5.server.prd.partner.identity.entity.QBusinessSupplierInfoDO;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 业务伙伴管理
 *
 * @author wangluyu
 * @date 2023-05-16
 */
@Repository
@RequiredArgsConstructor
public class BusinessPartnerDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BusinessPartnerRepo repo;
    private final QBusinessPartnerDO qdo = QBusinessPartnerDO.businessPartnerDO;
    private final QBookAddressDO qBookAddressDO = QBookAddressDO.bookAddressDO;
    private final QBookContactsDO qBookContactsDO = QBookContactsDO.bookContactsDO;
    private final QBusinessCustomerInfoDO qBusinessCustomerInfoDO = QBusinessCustomerInfoDO.businessCustomerInfoDO;
    private final QBusinessSupplierInfoDO qBusinessSupplierInfoDO = QBusinessSupplierInfoDO.businessSupplierInfoDO;
    private final QBusinessCollaboratePartnerInfoDO qBusinessCollaboratePartnerInfoDO = QBusinessCollaboratePartnerInfoDO.businessCollaboratePartnerInfoDO;


    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BusinessPartnerVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BusinessPartnerVO.class,
                        qdo.id,
                        qdo.remark,
                        qdo.createUserId,
                        qdo.creator,
                        qdo.createTime,
                        qdo.customerFlag,
                        qdo.collaborateFlag,
                        qdo.supplierFlag,
                        //qdo.modifyUserId,
                        //qdo.updater,
                        //qdo.modifyTime,
                        // 地址簿id 用来和银行账户 基本信息 地址信息等关联
                        qdo.bookId,
                        //tw4.0地址薄id
                        qdo.bookIdV4,
                        // 业务伙伴编号
                        qdo.businessPartnerNo,
                        // 类型 udc[CRM:BUSINESS_PARTNER:TYPE]
                        qdo.partnerType,
                        // 业务伙伴名称
                        qdo.partnerName,
                        // 业务伙伴身份 udc[CRM:BUSINESS_PARTNER:IDENTITY]
                        qdo.partnerIdentity,
                        // 行业 udc[crm:leads_customer_industry]
                        qdo.partnerIndustry,
                        // 性质 udc[crm:company_nature]
                        qdo.partnerNature,
                        // 组织规模 udc[crm:customer_scale]
                        qdo.organizationScale,
                        // 网址
                        qdo.partnerWebsite,
                        // 邮箱
                        qdo.partnerEmail,
                        // 传真
                        qdo.partnerFax,
                        // 电话
                        qdo.partnerPhone,
                        // 业务伙伴区域
                        qdo.partnerRegion,
                        // 主交易货币  udc[SYSTEM_BASIC:CURRENCY]
                        qdo.currency,
                        // 主要语言
                        qdo.language,
                        //英文名字
                        qdo.englishName,
                        //证件类型
                        qdo.certificateType,
                        //证件号
                        qdo.certificateNo,
                        // 性别 udc[org:employee:sex]
                        qdo.partnerSex,
                        // 生日
                        qdo.partnerBirthday,
                        // 国籍
                        qdo.nationality,
                        // 籍贯
                        qdo.nativePlace,
                        // 民族
                        qdo.nation,
                        // 婚姻状况
                        qdo.maritalStatus,
                        // 证件有效期起
                        qdo.certificateDateStart,
                        // 证件有效期止
                        qdo.certificateDateEnd,
                        //审批状态
                        qdo.applyStatus,
                        // 排序号
                        qdo.sortNo,
                        // 拓展字段1
                        qdo.ext1,
                        // 拓展字段2
                        qdo.ext2,
                        // 拓展字段3
                        qdo.ext3,
                        // 拓展字段4
                        qdo.ext4,
                        // 拓展字段5
                        qdo.ext5,
                        // 母公司ID
                        qdo.parentId,
                        //第三方系统编号
                        qdo.thirdSystemNo,
                        //国家
                        qBookAddressDO.country,
                        //省
                        qBookAddressDO.province,
                        //市
                        qBookAddressDO.city,
                        //区
                        qBookAddressDO.district,
                        qBookAddressDO.provinceName,
                        qBookAddressDO.cityName,
                        qBookAddressDO.districtName,
                        qBookAddressDO.detailAddress,
                        qBusinessCustomerInfoDO.customerStatus,
                        qBusinessCustomerInfoDO.confirmFlag,
                        qBusinessCustomerInfoDO.customerNo,
                        qBusinessCollaboratePartnerInfoDO.collaborateStatus,
                        qBusinessSupplierInfoDO.supplierStatus,
                        qBusinessSupplierInfoDO.supplierNo,
                        qBookContactsDO.contactsPhone,
                        qBookContactsDO.contactsEmail,
                        qdo.insideOrOutSide,
                        // 上级主管部门
                        qdo.parentManageBU
                )).from(qdo).
                leftJoin(qBookAddressDO).on(qdo.bookId.eq(qBookAddressDO.bookId).and(qBookAddressDO.isDefault.eq(true)).and(qBookAddressDO.deleteFlag.eq(0))).
                leftJoin(qBusinessCustomerInfoDO).on(qdo.id.eq(qBusinessCustomerInfoDO.partnerId)).
                leftJoin(qBusinessCollaboratePartnerInfoDO).on(qdo.id.eq(qBusinessCollaboratePartnerInfoDO.partnerId)).
                leftJoin(qBusinessSupplierInfoDO).on(qdo.id.eq(qBusinessSupplierInfoDO.partnerId)).
                leftJoin(qBookContactsDO).on(qdo.bookId.eq(qBookContactsDO.bookId).and(qBookContactsDO.isDefault.eq(true)).and(qBookContactsDO.deleteFlag.eq(0)))
                ;
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BusinessPartnerVO> getJpaQueryWhere(BusinessPartnerQuery query) {
        JPAQuery<BusinessPartnerVO> jpaQuery = getJpaQuerySelect();

        //获取当前登录用户
        final Long loginUserId = GlobalUtil.getLoginUserId();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        //如果地址有重复数据 随机取一条
        // jpaQuery.groupBy(qdo.id);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BusinessPartnerQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo).leftJoin(qBookAddressDO).on(qdo.bookId.eq(qBookAddressDO.bookId).and(qBookAddressDO.isDefault.eq(true)).and(qBookAddressDO.deleteFlag.eq(0))).
                        leftJoin(qBusinessCustomerInfoDO).on(qdo.id.eq(qBusinessCustomerInfoDO.partnerId)).
                        leftJoin(qBusinessCollaboratePartnerInfoDO).on(qdo.id.eq(qBusinessCollaboratePartnerInfoDO.partnerId)).
                        leftJoin(qBusinessSupplierInfoDO).on(qdo.id.eq(qBusinessSupplierInfoDO.partnerId)).
                        leftJoin(qBookContactsDO).on(qdo.bookId.eq(qBookContactsDO.bookId).and(qBookContactsDO.isDefault.eq(true)).and(qBookContactsDO.deleteFlag.eq(0)));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        //如果地址有重复数据 随机取一条
        //  jpaQuery.groupBy(qdo.id);
        //因为分组的原因不能使用fetchOne
        return jpaQuery.fetchOne();
    }

    /**
     * 销售线索统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long saleCount(BusinessPartnerQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        if (query.getUserIdList() != null && query.getUserIdList().size() > 0) {
            jpaQuery.where(qdo.createUserId.in(query.getUserIdList()));
        }
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        return jpaQuery.fetchOne();
    }


    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BusinessPartnerQuery query) {
        List<Predicate> list = new ArrayList<>();

        /** 业务伙伴名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getPartnerName())) {
            String pattern = "^[a-zA-Z]+$";
            Pattern p = Pattern.compile(pattern);
            Matcher m = p.matcher(query.getPartnerName());
            if (m.matches()) {
                list.add(qdo.partnerNamePinyin.like(SqlUtil.toSqlLikeString(query.getPartnerName())));
            } else {
                list.add(qdo.partnerName.like(SqlUtil.toSqlLikeString(query.getPartnerName())));
            }
        }
        /** 内部/外部 精确 */
        if (!ObjectUtils.isEmpty(query.getInsideOrOutSide())) {
            list.add(qdo.insideOrOutSide.eq(query.getInsideOrOutSide()));
        }

        /** 业务伙伴编号 精确 */
        if (!ObjectUtils.isEmpty(query.getBusinessPartnerNo())) {
            list.add(qdo.businessPartnerNo.eq(query.getBusinessPartnerNo()));
        }
        /** 业务伙伴类型 精确 */
        if (!ObjectUtils.isEmpty(query.getPartnerType())) {
            list.add(qdo.partnerType.eq(query.getPartnerType()));
        }

        /** 业务伙伴身份 udc[CRM:BUSINESS_PARTNER:IDENTITY] 精确 */
        if (!ObjectUtils.isEmpty(query.getPartnerIdentity())) {
            String[] identityArray = query.getPartnerIdentity().split(",");
            List identityList = Arrays.asList(identityArray);
            if (identityList.contains(BusinessPartnerIdentityEnum.CUSTOMER.getCode())) {
                list.add(qdo.customerFlag.eq(true));
            }
            if (identityList.contains(BusinessPartnerIdentityEnum.SUPPLIER.getCode())) {
                list.add(qdo.supplierFlag.eq(true));
            }
            if (identityList.contains(BusinessPartnerIdentityEnum.COLLABORATE.getCode())) {
                list.add(qdo.collaborateFlag.eq(true));
            }
            if (identityList.contains(BusinessPartnerIdentityEnum.DISTRIBUTOR.getCode())) {
                list.add(qdo.distributorFlag.eq(true));
            }
        }
        /** 创建人 精确*/
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            list.add(qdo.createUserId.eq(query.getCreateUserId()));
        }
        if (query.getUserIdList() != null && query.getUserIdList().size() > 0) {
            list.add(qdo.createUserId.in(query.getUserIdList()));
        }

        /** 行业 udc[crm:leads_customer_industry] 精确 */
        if (!ObjectUtils.isEmpty(query.getPartnerIndustry())) {
            List<String> collect = Arrays.stream(query.getPartnerIndustry().split(",")).collect(Collectors.toList());
            list.add(qdo.partnerIndustry.in(collect));
        }
        /** 省 精确 */
        if (!ObjectUtils.isEmpty(query.getProvince())) {
            list.add(qBookAddressDO.province.eq(query.getProvince()));
        }
        /** 市 精确*/
        if (!ObjectUtils.isEmpty(query.getCity())) {
            list.add(qBookAddressDO.city.eq(query.getCity()));
        }
        /** 区 精确*/
        if (!ObjectUtils.isEmpty(query.getDistrict())) {
            list.add(qBookAddressDO.district.eq(query.getDistrict()));
        }

        /** 省 精确 */
        if (!ObjectUtils.isEmpty(query.getProvinceName())) {
            list.add(qBookAddressDO.provinceName.eq(query.getProvinceName()));
        }
        /** 市 精确*/
        if (!ObjectUtils.isEmpty(query.getCityName())) {
            list.add(qBookAddressDO.cityName.eq(query.getCityName()));
        }
        /** 区 精确*/
        if (!ObjectUtils.isEmpty(query.getDistrictName())) {
            list.add(qBookAddressDO.districtName.eq(query.getDistrictName()));
        }
        /** 客户状态 精确*/
        if (!ObjectUtils.isEmpty(query.getCustomerStatus())) {
            list.add(qBusinessCustomerInfoDO.customerStatus.eq(query.getCustomerStatus()));
        }
        /** 供应商状态 精确*/
        if (!ObjectUtils.isEmpty(query.getSupplierStatus())) {
            list.add(qBusinessSupplierInfoDO.supplierStatus.eq(query.getSupplierStatus()));
        }
        /** 合作伙伴状态 精确*/
        if (!ObjectUtils.isEmpty(query.getCollaborateStatus())) {
            list.add(qBusinessCollaboratePartnerInfoDO.collaborateStatus.eq(query.getCollaborateStatus()));
        }
        /** 权限处理 精确*/
        if (query.getPermissionFlag()) {
            list.add(qdo.createUserId.eq(query.getLoginUserId()));
        }

        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BusinessPartnerVO queryByKey(Long id) {
        JPAQuery<BusinessPartnerVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BusinessPartnerDO queryById(Long id) {
        return jpaQueryFactory.select(qdo).
                from(qdo).where(qdo.id.eq(id)).
                where(qdo.deleteFlag.eq(0)).
                fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BusinessPartnerVO> queryListDynamic(BusinessPartnerQuery query) {
        JPAQuery<BusinessPartnerVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 母公司下拉查询
     *
     * @return 结果集合
     */
    public List<BusinessPartnerSimpleVO> queryListSimple() {
        return jpaQueryFactory.select(Projections.bean(BusinessPartnerSimpleVO.class, qdo.id, qdo.partnerName)).from(qdo).fetch();
    }

    /**
     * 业务伙伴名字重复性校验
     *
     * @return Long 个数
     */
    public Long judgePartnerName(String partnerName) {
        return jpaQueryFactory
                .select(qdo.count())
                .from(qdo).where(qdo.partnerName.eq(partnerName)).where(qdo.deleteFlag.eq(0)).fetchOne();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BusinessPartnerVO> queryPaging(BusinessPartnerQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<BusinessPartnerVO> jpaQuery = getJpaQueryWhere(query);
        List<BusinessPartnerVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BusinessPartnerVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BusinessPartnerDO save(BusinessPartnerDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BusinessPartnerDO> saveAll(List<BusinessPartnerDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BusinessPartnerPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 地址簿id 用来和银行账户 基本信息 地址信息等关联
        if (payload.getBookId() != null) {
            update.set(qdo.bookId, payload.getBookId());
        }
        // 业务伙伴编号
        if (payload.getBusinessPartnerNo() != null) {
            update.set(qdo.businessPartnerNo, payload.getBusinessPartnerNo());
        }
        // 类型 udc[CRM:BUSINESS_PARTNER:TYPE]
        if (payload.getPartnerType() != null) {
            update.set(qdo.partnerType, payload.getPartnerType());
        }
        // 业务伙伴名称
        if (payload.getPartnerName() != null) {
            update.set(qdo.partnerName, payload.getPartnerName());
            String pinyin = PinyinUtil.getPinyin(payload.getPartnerName(), "");
            update.set(qdo.partnerNamePinyin, pinyin);
        }
        // 业务伙伴身份 udc[CRM:BUSINESS_PARTNER:IDENTITY]
        if (payload.getPartnerIdentity() != null) {
            update.set(qdo.partnerIdentity, payload.getPartnerIdentity());
        }
        // 行业 udc[crm:leads_customer_industry]
        if (payload.getPartnerIndustry() != null) {
            update.set(qdo.partnerIndustry, payload.getPartnerIndustry());
        }
        // 性质 udc[crm:company_nature]
        if (payload.getPartnerNature() != null) {
            update.set(qdo.partnerNature, payload.getPartnerNature());
        }
        // 组织规模 udc[crm:customer_scale]
        if (payload.getOrganizationScale() != null) {
            update.set(qdo.organizationScale, payload.getOrganizationScale());
        }
        // 是否有客户身份
        if (payload.getCustomerFlag() != null) {
            update.set(qdo.customerFlag, payload.getCustomerFlag());
        }
        // 网址
        if (payload.getPartnerWebsite() != null) {
            update.set(qdo.partnerWebsite, payload.getPartnerWebsite());
        }
        // 邮箱
        if (payload.getPartnerEmail() != null) {
            update.set(qdo.partnerEmail, payload.getPartnerEmail());
        }
        // 传真
        if (payload.getPartnerFax() != null) {
            update.set(qdo.partnerFax, payload.getPartnerFax());
        }
        // 电话
        if (payload.getPartnerPhone() != null) {
            update.set(qdo.partnerPhone, payload.getPartnerPhone());
        }
        // 区域
        if (payload.getPartnerRegion() != null) {
            update.set(qdo.partnerRegion, payload.getPartnerRegion());
        }
        // 主交易货币  udc[SYSTEM_BASIC:CURRENCY]
        if (payload.getCurrency() != null) {
            update.set(qdo.currency, payload.getCurrency());
        }
        // 主要语言
        if (payload.getLanguage() != null) {
            update.set(qdo.language, payload.getLanguage());
        }
        // 性别 udc[org:employee:sex]
        if (payload.getPartnerSex() != null) {
            update.set(qdo.partnerSex, payload.getPartnerSex());
        }
        // 生日
        if (payload.getPartnerBirthday() != null) {
            update.set(qdo.partnerBirthday, payload.getPartnerBirthday());
        }
//        // 微信
//        if (payload.getPartnerWeChat() != null) {
//            update.set(qdo.partnerWeChat, payload.getPartnerWeChat());
//        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 拓展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 拓展字段4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 拓展字段5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 母公司ID
        if (payload.getParentId() != null) {
            update.set(qdo.parentId, payload.getParentId());
        }
        // 内部/外部
        if (payload.getInsideOrOutSide() != null) {
            update.set(qdo.insideOrOutSide, payload.getInsideOrOutSide());
        }
        // 状态
        if (!ObjectUtils.isEmpty(payload.getApplyStatus())) {
            update.set(qdo.applyStatus, payload.getApplyStatus());
        }
        //上级主管部门
        if (!ObjectUtils.isEmpty(payload.getParentManageBU())) {
            update.set(qdo.parentManageBU, payload.getParentManageBU());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 地址簿id 用来和银行账户 基本信息 地址信息等关联
            if (nullFields.contains("bookId")) {
                update.setNull(qdo.bookId);
            }
            // 业务伙伴编号
            if (nullFields.contains("businessPartnerNo")) {
                update.setNull(qdo.businessPartnerNo);
            }
            // 类型 udc[CRM:BUSINESS_PARTNER:TYPE]
            if (nullFields.contains("partnerType")) {
                update.setNull(qdo.partnerType);
            }
            // 业务伙伴名称
            if (nullFields.contains("partnerName")) {
                update.setNull(qdo.partnerName);
                update.setNull(qdo.partnerNamePinyin);
            }
            // 业务伙伴身份 udc[CRM:BUSINESS_PARTNER:IDENTITY]
            if (nullFields.contains("partnerIdentity")) {
                update.setNull(qdo.partnerIdentity);
            }
            // 行业 udc[crm:leads_customer_industry]
            if (nullFields.contains("partnerIndustry")) {
                update.setNull(qdo.partnerIndustry);
            }
            // 性质 udc[crm:company_nature]
            if (nullFields.contains("partnerNature")) {
                update.setNull(qdo.partnerNature);
            }
            // 组织规模 udc[crm:customer_scale]
            if (nullFields.contains("organizationScale")) {
                update.setNull(qdo.organizationScale);
            }
            // 网址
            if (nullFields.contains("partnerWebsite")) {
                update.setNull(qdo.partnerWebsite);
            }
            // 邮箱
            if (nullFields.contains("partnerEmail")) {
                update.setNull(qdo.partnerEmail);
            }
            // 传真
            if (nullFields.contains("partnerFax")) {
                update.setNull(qdo.partnerFax);
            }
            // 电话
            if (nullFields.contains("partnerPhone")) {
                update.setNull(qdo.partnerPhone);
            }

            // 区域
            if (nullFields.contains("partnerRegion")) {
                update.setNull(qdo.partnerRegion);
            }
            // 主交易货币  udc[SYSTEM_BASIC:CURRENCY]
            if (nullFields.contains("currency")) {
                update.setNull(qdo.currency);
            }
            // 主要语言
            if (nullFields.contains("language")) {
                update.setNull(qdo.language);
            }
            // 性别 udc[org:employee:sex]
            if (nullFields.contains("partnerSex")) {
                update.setNull(qdo.partnerSex);
            }
            // 生日
            if (nullFields.contains("partnerBirthday")) {
                update.setNull(qdo.partnerBirthday);
            }
            // 微信
            if (nullFields.contains("partnerWeChat")) {
                update.setNull(qdo.partnerWeChat);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 拓展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 拓展字段4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 拓展字段5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
            // 母公司ID
            if (nullFields.contains("parentId")) {
                update.setNull(qdo.parentId);
            }
            // 审批状态
            if (nullFields.contains("applyStatus")) {
                update.setNull(qdo.applyStatus);
            }
            // 内部/外部
            if (nullFields.contains("insideOrOutSide")) {
                update.setNull(qdo.insideOrOutSide);
            }
            // 上级主管部门
            if (nullFields.contains("parentManageBU")) {
                update.setNull(qdo.parentManageBU);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public List<BusinessPartnerVO> findByIds(List<Long> ids) {
        JPAQuery<BusinessPartnerVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.groupBy(qdo.id);
        List<BusinessPartnerVO> businessPartnerVOList = jpaQuery.fetch();
        return businessPartnerVOList;
    }

    public BusinessPartnerSimpleVO queryByPartnerName(String partnerName) {
        return jpaQueryFactory.select(Projections.bean(BusinessPartnerSimpleVO.class, qdo.id, qdo.customerFlag, qdo.partnerIdentity, qdo.partnerName, qdo.parentId)).from(qdo)
                .where(qdo.partnerName.eq(partnerName))
                .where(qdo.deleteFlag.eq(0))
                .fetchFirst();
    }

    public List<BusinessPartnerDO> findByPartnerName(String partnerName) {
        return jpaQueryFactory.select(qdo).from(qdo)
                .where(qdo.partnerName.eq(partnerName))
                .where(qdo.deleteFlag.eq(0))
                .fetch();
    }

    public List<BusinessPartnerVO> listForDm(List<String> partnerNameList) {
        List<BusinessPartnerVO> partnerDoList = jpaQueryFactory.select(Projections.bean(BusinessPartnerVO.class,
                qdo.id,
                qdo.partnerName
        )).from(qdo)
                .where(qdo.partnerName.in(partnerNameList))
                .fetch();
        return partnerDoList;
    }

    public List<BusinessPartnerVO> queryCustPartnerListSimple() {
        return jpaQueryFactory.select(Projections.bean(BusinessPartnerVO.class, qdo.id, qdo.bookId, qdo.partnerName)).
                from(qdo).
                where(qdo.customerFlag.eq(true)).
                where(qdo.deleteFlag.eq(0)).
                where(qdo.insideOrOutSide.eq(BusinessInsideOrOutSideEnum.OUTSIDE.getCode())).fetch();
    }


    // 查询业务伙伴详情
    public List<BusinessPartnerInfoVO> queryBusinessPartnerInfo(BusinessPartnerQuery query) {
        JPAQuery<BusinessPartnerInfoVO> jpaQuery = jpaQueryFactory.select(Projections.bean(BusinessPartnerInfoVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 地址簿id 用来和银行账户 基本信息 地址信息等关联
                qdo.bookId,
                // 业务伙伴编号
                qdo.businessPartnerNo,
                //第三方系统编号
                qdo.thirdSystemNo,
                // 类型 udc[CRM:BUSINESS_PARTNER:TYPE]
                qdo.partnerType,
                // 业务伙伴名称
                qdo.partnerName,
                // 业务伙伴身份 udc[CRM:BUSINESS_PARTNER:IDENTITY]
                qdo.partnerIdentity,
                // 母公司ID
                qdo.parentId,
                // 行业 udc[crm:leads_customer_industry]
                qdo.partnerIndustry,
                // 性质 udc[crm:company_nature]
                qdo.partnerNature,
                // 组织规模 udc[crm:customer_scale]
                qdo.organizationScale,
                // 业务伙伴区域
                qdo.partnerRegion,
                // 网址
                qdo.partnerWebsite,
                // 邮箱
                qdo.partnerEmail,
                // 传真
                qdo.partnerFax,
                // 电话
                qdo.partnerPhone,
                // 主交易货币  udc[SYSTEM_BASIC:CURRENCY]
                qdo.currency,
                // 主要语言
                qdo.language,
                //国家
                qBookAddressDO.country,
                //省
                qBookAddressDO.province,
                //市
                qBookAddressDO.city,
                //区
                qBookAddressDO.district,
                qBookAddressDO.provinceName,
                qBookAddressDO.cityName,
                qBookAddressDO.districtName,
                qBookAddressDO.detailAddress,
                //联系人名字
                qBookContactsDO.contactsName,
                //联系人手机
                qBookContactsDO.contactsPhone,
                //英文名字
                qdo.englishName,
                //证件类型
                qdo.certificateType,
                //证件号
                qdo.certificateNo,
                // 性别 udc[org:employee:sex]
                qdo.partnerSex,
                // 生日
                qdo.partnerBirthday,
                // 国籍
                qdo.nationality,
                // 籍贯
                qdo.nativePlace,
                // 民族
                qdo.nation,
                // 婚姻状况
                qdo.maritalStatus,
                // 证件有效期起
                qdo.certificateDateStart,
                // 证件有效期止
                qdo.certificateDateEnd,
                //内部/外部
                qdo.insideOrOutSide
        )).from(qdo).
                leftJoin(qBookAddressDO).on(qdo.bookId.eq(qBookAddressDO.bookId).and(qBookAddressDO.isDefault.eq(true)).and(qBookAddressDO.deleteFlag.eq(0))).
                leftJoin(qBookContactsDO).on(qdo.bookId.eq(qBookContactsDO.bookId).and(qBookContactsDO.isDefault.eq(true)).and(qBookContactsDO.deleteFlag.eq(0)));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        //如果地址有重复数据 随机取一条
        // jpaQuery.groupBy(qdo.id);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery.fetch();
    }

    public List<BusinessPartnerVO> queryByBookId(Long bookId) {
        return jpaQueryFactory.select(
                Projections.bean(
                        BusinessPartnerVO.class,
                        qdo.partnerIndustry,
                        qdo.id,qdo.partnerIndustry,
                        qdo.bookId, qdo.partnerName,
                        qdo.idenNo,qdo.customerFlag,
                        qdo.jdeCompanyNo,
                        qdo.supplierFlag,
                        qdo.applyStatus,
                        qdo.collaborateFlag,
                        qdo.insideOrOutSide)).
                from(qdo).
                where(qdo.deleteFlag.eq(0)).
                where(qdo.bookId.eq(bookId)).
                fetch();
    }

    public BusinessPartnerVO findById(Long id) {
        return jpaQueryFactory.select(
                Projections.bean(BusinessPartnerVO.class,
                        qdo.id,
                        qdo.bookId,
                        qdo.partnerName,
                        qdo.customerFlag,
                        qdo.supplierFlag,
                        qdo.collaborateFlag,
                        qdo.insideOrOutSide)).
                from(qdo).
                where(qdo.deleteFlag.eq(0)).
                where(qdo.id.eq(id)).
                fetchFirst();
    }
    // 通过bookId查询客户信息
    public BusinessPartnerVO queryCustomerByBookId(Long bookId) {
        return jpaQueryFactory.select(
                        Projections.bean(BusinessPartnerVO.class,
                                qdo.id,
                                qdo.partnerIndustry,
                                qdo.partnerRegion,
                                qdo.businessPartnerNo,
                                qdo.bookId,
                                qdo.partnerName,
                                qdo.customerFlag,
                                qBusinessCustomerInfoDO.syncJdeFlag,
                                qBusinessCustomerInfoDO.customerGradle,
                                qBusinessCustomerInfoDO.customerNo,
                                qBusinessCustomerInfoDO.customerStatus,
                                qBusinessCustomerInfoDO.id.as("businessCustomerInfoId"))).
                from(qdo).leftJoin(qBusinessCustomerInfoDO).on(qdo.id.eq(qBusinessCustomerInfoDO.partnerId)).
                where(qBusinessCustomerInfoDO.deleteFlag.eq(0)).
                where(qdo.deleteFlag.eq(0)).
                where(qdo.bookId.eq(bookId)).
                fetchFirst();
    }
    // 通过bookId查询供应商信息
    public BusinessPartnerVO querySupplierByBookId(Long bookId) {
        return jpaQueryFactory.select(Projections.bean(BusinessPartnerVO.class,qdo.id,qdo.partnerIndustry,qdo.partnerRegion,qdo.businessPartnerNo, qdo.bookId, qdo.partnerName, qdo.supplierFlag,qBusinessSupplierInfoDO.syncJdeFlag, qBusinessSupplierInfoDO.supplierNo, qBusinessSupplierInfoDO.id.as("businessSupplierInfoId"))).
                from(qdo).leftJoin(qBusinessSupplierInfoDO).on(qdo.id.eq(qBusinessSupplierInfoDO.partnerId)).
                where(qBusinessSupplierInfoDO.deleteFlag.eq(0)).
                where(qdo.deleteFlag.eq(0)).
                where(qdo.bookId.eq(bookId)).
                fetchFirst();
    }
    /**
     * 变更工作流相关数据
     *
     * @param payload
     * @return
     */

    public long updateWorkFlow(BusinessPartnerPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 流程实例id
        if (!ObjectUtils.isEmpty(payload.getProcInstId())) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 状态
        if (!ObjectUtils.isEmpty(payload.getApplyStatus())) {
            update.set(qdo.applyStatus, payload.getApplyStatus());
        }

        if (!ObjectUtils.isEmpty(payload.getDeleteFlag())) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null) {
            // 流程实例id
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 流程实例id
            if (nullFields.contains("applyStatus")) {
                update.setNull(qdo.applyStatus);
            }
        }

        // 执行修改
        return update.execute();
    }

    //待确认潜在客户查询
    public PagingVO<BusinessConfirmedCustomerVO> queryConfirmedCustomer(BusinessPartnerQuery query) {
        Long total = countConfirmedCustomer();
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<BusinessConfirmedCustomerVO> jpaQuery = jpaQueryFactory.select(Projections.bean(BusinessConfirmedCustomerVO.class,
                qdo.id.as("partnerId"),
                qdo.bookId,
                qBusinessCustomerInfoDO.id,
                qdo.businessPartnerNo,
                qdo.partnerName,
                qBusinessCustomerInfoDO.customerGradle,
                qBusinessCustomerInfoDO.customerStatus,
                qdo.partnerIndustry,
                qBusinessCustomerInfoDO.customerSource,
                qdo.partnerEmail,
                qdo.partnerPhone,
                qdo.partnerFax,
                //省
                qBookAddressDO.province,
                //市
                qBookAddressDO.city,
                //区
                qBookAddressDO.district,
                qBookAddressDO.provinceName,
                qBookAddressDO.cityName,
                qBookAddressDO.districtName
        )).from(qdo).leftJoin(qBusinessCustomerInfoDO).on(qdo.id.eq(qBusinessCustomerInfoDO.partnerId)).
                leftJoin(qBookAddressDO).on(qdo.bookId.eq(qBookAddressDO.bookId).and(qBookAddressDO.isDefault.eq(true)).and(qBookAddressDO.deleteFlag.eq(0))).
                where(qdo.deleteFlag.eq(0)).
                where(qBusinessCustomerInfoDO.deleteFlag.eq(0)).
                where(qBusinessCustomerInfoDO.customerStage.eq(BusinessPartnerStageEnum.POTENTIAL.getCode())).
                where(qBusinessCustomerInfoDO.confirmFlag.eq(false));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        //如果地址有重复数据 随机取一条
        // jpaQuery.groupBy(qdo.id);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        List<BusinessConfirmedCustomerVO> result = jpaQuery.offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BusinessConfirmedCustomerVO>builder().records(result).total(total).build();


    }

    //待确认潜在客户统计
    public Long countConfirmedCustomer() {
        Long count = jpaQueryFactory.select(qdo.id.count()).
                from(qdo).leftJoin(qBusinessCustomerInfoDO).on(qdo.id.eq(qBusinessCustomerInfoDO.partnerId)).
                leftJoin(qBookAddressDO).on(qdo.bookId.eq(qBookAddressDO.bookId).and(qBookAddressDO.isDefault.eq(true)).and(qBookAddressDO.deleteFlag.eq(0))).
                where(qdo.deleteFlag.eq(0)).
                where(qBusinessCustomerInfoDO.deleteFlag.eq(0)).
                where(qBusinessCustomerInfoDO.customerStage.eq(BusinessPartnerStageEnum.POTENTIAL.getCode())).
                where(qBusinessCustomerInfoDO.confirmFlag.eq(false)).
                fetchOne();
        if (count == null) {
            return 0L;
        }
        return count;
    }

    //待确认潜在客户更新
    public void updateConfirmedCustomer(List<Long> keys) {
        jpaQueryFactory.update(qBusinessCustomerInfoDO).
                set(qBusinessCustomerInfoDO.confirmFlag, true).
                where(qBusinessCustomerInfoDO.partnerId.in(keys)).
                execute();
    }

    //查询附近客户
    public List<BusinessPartnerSimpleVO> queryNearCustomerList() {
        return jpaQueryFactory.select(Projections.bean(BusinessPartnerSimpleVO.class, qdo.id, qdo.partnerName)).
                from(qdo).
                where(qdo.customerFlag.eq(true)).
                where(qdo.deleteFlag.eq(0)).
                fetch();
    }

    //查询需要更新的客户主档数据
    public List<BusinessPartnerMainSyncVO> queryBusinessPartSync(String businessPartnerNo) {
        JPAQuery<BusinessPartnerMainSyncVO> jpaQuery = jpaQueryFactory.select(Projections.bean(BusinessPartnerMainSyncVO.class,
                qdo.id, qdo.bookId.as("bookId"),
                qdo.id, qdo.bookIdV4.as("bookIdV4"),
                qdo.businessPartnerNo.as("bookNo"),
                qdo.partnerName.as("customerName"),
                qBusinessCustomerInfoDO.customerGradle.as("customerGrade"),
                qBusinessCustomerInfoDO.customerStatus.as("customerStatus"),
                qdo.partnerIndustry.as("companyIndustry"),
                qBusinessCustomerInfoDO.customerSource.as("customerSource"),
                qdo.partnerEmail.as("companyEmail"),
                qdo.partnerPhone.as("companyPhone"),
                qdo.partnerFax.as("companyFax"),
                //省
                qBookAddressDO.province.as("province"),
                //市
                qBookAddressDO.city.as("city"),
                //区
                qBookAddressDO.district.as("district"),
                qBookAddressDO.provinceName.as("provinceName"),
                qBookAddressDO.cityName.as("cityName"),
                qBookAddressDO.districtName.as("districtName"),
                qBookAddressDO.detailAddress.as("companyAddress"),
                qdo.partnerNature.as("companyNature"),
                // 组织规模 udc[crm:customer_scale]
                qdo.organizationScale.as("companyScale"),
                // 网址
                qdo.partnerWebsite.as("companyWebsite"),
                // 主交易货币  udc[SYSTEM_BASIC:CURRENCY]
                qdo.currency.as("currency"),
                // 主要语言
                qdo.language.as("language"),
                //公司还是个人
                qdo.partnerType.as("abType"),
                //身份
                qdo.partnerIdentity.as("relateType"),
                //内部外部
                qdo.insideOrOutSide.as("innerType")
        )).from(qdo).leftJoin(qBusinessCustomerInfoDO).on(qdo.id.eq(qBusinessCustomerInfoDO.partnerId).and(qBusinessCustomerInfoDO.deleteFlag.eq(0))).
                leftJoin(qBookAddressDO).on(qdo.bookId.eq(qBookAddressDO.bookId).and(qBookAddressDO.isDefault.eq(true)).and(qBookAddressDO.deleteFlag.eq(0))).
                where(qdo.deleteFlag.eq(0)).
                where(qdo.insideOrOutSide.eq(BusinessInsideOrOutSideEnum.OUTSIDE.getCode())).
                where(qdo.syncTime.lt(qdo.modifyTime).or(qdo.syncTime.isNull()));
        if (StringUtils.hasText(businessPartnerNo)) {
            jpaQuery.where(qdo.businessPartnerNo.eq(businessPartnerNo));
        }
        return jpaQuery.fetch();

    }

    //业务伙伴主档同步之后更新同步时间和bookIdV4
    public void updateSyncTimeAndBookIdV4(Long id, Long bookIdV4, LocalDateTime sybcTime) {
        jpaQueryFactory.update(qdo).
                set(qdo.syncTime, sybcTime).
                set(qdo.bookIdV4, bookIdV4).
                where(qdo.id.eq(id)).
                execute();
    }

    //更新业务伙伴的更新时间
    public void updateBusinessParnerModifyTime(List<Long> bookIdList) {
        jpaQueryFactory.update(qdo).
                set(qdo.modifyTime, LocalDateTime.now()).
                where(qdo.bookId.in(bookIdList)).
                execute();
    }

    /**
     * 根据bookIds返回列表
     *
     * @param bookIdList 地址薄Id
     * @return 结果
     */
    public List<Tuple> findNameByBookIds(List<Long> bookIdList) {
        JPAQuery<Tuple> jpaQuery = jpaQueryFactory.select(
                qdo.bookId,
                qdo.partnerName
        ).from(qdo)
                .where(qdo.bookId.in(bookIdList))
                .where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据bookId集合和类型查询
     *
     * @param bookIds
     * @param insideOrOutside
     * @return
     */
    public List<BusinessPartnerVO> queryByBookIdsAndType(List<Long> bookIds, String insideOrOutside) {
        return jpaQueryFactory.select(Projections.bean(BusinessPartnerVO.class,
                qdo.id, qdo.bookId, qdo.partnerName, qdo.jdeCompanyNo, qdo.customerFlag, qdo.insideOrOutSide))
                .from(qdo)
                .where(qdo.deleteFlag.eq(0))
                .where(qdo.bookId.in(bookIds))
                .where(qdo.insideOrOutSide.eq(insideOrOutside))
                .fetch();
    }
}

