package com.elitesland.tw.tw5.server.prd.salecon.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.salecon.payload.ConAchieveDPayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConAchieveDQuery;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConAchieveDVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.org.entity.QPrdOrgEmployeeRefDO;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBusinessPartnerDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.ConAchieveDDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConAchieveDDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConAchieveDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QSaleConContractDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConAchieveDRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 业绩统计明细
 *
 * @author likunpeng
 * @date 2023-04-24
 */
@Repository
@RequiredArgsConstructor
public class ConAchieveDDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ConAchieveDRepo repo;
    private final QConAchieveDDO qdo = QConAchieveDDO.conAchieveDDO;
    private final QConAchieveDO qdoMain = QConAchieveDO.conAchieveDO;
    private final QSaleConContractDO qdoConSub = QSaleConContractDO.saleConContractDO;
    private final QSaleConContractDO qdoConMain = new QSaleConContractDO("qdoConMain");
    // private final QCrmCustomerDO qdoCrmCust = QCrmCustomerDO.crmCustomerDO;
    private final QBusinessPartnerDO businessPartnerDO = QBusinessPartnerDO.businessPartnerDO;
    private final QPrdOrgEmployeeRefDO qdoRef = QPrdOrgEmployeeRefDO.prdOrgEmployeeRefDO;


    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ConAchieveDVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ConAchieveDVO.class,
                        qdo.id,
                        qdo.remark,
                        qdo.createUserId,
                        qdo.creator,
                        qdo.createTime,
                        qdo.modifyUserId,
                        qdo.updater,
                        qdo.modifyTime,
                        // 业绩统计ID
                        qdo.achieveId,
                        // 部门类别
                        qdo.buType,
                        // 部门ID
                        qdo.buId,
                        // 业绩类别
                        qdo.achieveType,
                        // 价值角色
                        qdo.valueRole,
                        // 部门归属人员ID
                        qdo.chargeResId,
                        qdo.achievedIdV4,
                        // 比例
                        qdo.ratio,
                        // 金额
                        qdo.amt,
                        // 主合同编号
                        qdoConMain.id.as("mainConId"),
                        // 主合同编号
                        qdoConMain.code.as("mainConCode"),
                        // 主合同名称
                        qdoConMain.name.as("mainConName"),
                        // 子合同编号
                        qdoConSub.id.as("subConId"),
                        // 子合同编号
                        qdoConSub.code.as("subConCode"),
                        // 子合同名称
                        qdoConSub.name.as("subConName"),
                        // 子合同状态
                        qdoConSub.status.as("subConStatus"),
                        // 客户名称
                        qdoConMain.custName,
                        // 签单日期
                        qdoConSub.signDate,
                        // 子合同有效金额
                        qdoConSub.effectiveAmt,
                        // 子合同签单金额
                        qdoConSub.amt.as("signAmt"),
                        // 销售区域BU
                        qdoConSub.regionBuId,
                        // 销售负责人
                        qdoConSub.saleManUserId,
                        // 交付BU
                        qdoConSub.deliBuId,
                        // 交付负责人
                        qdoConSub.deliUserId,
                        // PMO
                        qdoConSub.pmoUserId,
                        // 签单BU
                        qdoConSub.signBuId,
                        // 客户名称
                        businessPartnerDO.partnerName.as("customerName"),
                        // 客户ID
                        qdoConSub.custId
                )).from(qdo)
                .leftJoin(qdoMain).on(qdoMain.id.longValue().eq(qdo.achieveId.longValue()))
                .leftJoin(qdoConSub).on(qdoMain.contractId.longValue().eq(qdoConSub.id.longValue()).and(qdoConSub.deleteFlag.eq(0)))
                .leftJoin(qdoConMain).on(qdoConSub.parentId.longValue().eq(qdoConMain.id.longValue()).and(qdoConMain.deleteFlag.eq(0)))
                .leftJoin(businessPartnerDO).on(qdoConSub.custId.longValue().eq(businessPartnerDO.bookId.longValue()))
                .leftJoin(qdoRef).on(qdoConSub.saleManUserId.longValue().eq(qdoRef.userId.longValue()).and(qdoRef.isDefault.eq(0)).and(qdoRef.isCopy.eq(0)).and(qdoRef.deleteFlag.eq(0)));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ConAchieveDVO> getJpaQueryWhere(ConAchieveDQuery query) {
        JPAQuery<ConAchieveDVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        jpaQuery.groupBy(qdo.id);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ConAchieveDQuery query) {
//        long total = jpaQueryFactory
//            .select(qdo.count())
//            .from(qdo)
//            .where(where(query))
//            .fetchOne();
        long total = getJpaQueryWhere(query).fetchCount();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ConAchieveDQuery query) {
        List<Predicate> list = new ArrayList<>();
        list.add(qdo.deleteFlag.eq(0));
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 业绩统计ID 精确 */
        if (!ObjectUtils.isEmpty(query.getAchieveId())) {
            list.add(qdo.achieveId.eq(query.getAchieveId()));
        }
        /** 部门类别 精确 */
        if (!ObjectUtils.isEmpty(query.getBuType())) {
            list.add(qdo.buType.eq(query.getBuType()));
        }
        /** 部门ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBuId())) {
            list.add(qdo.buId.eq(query.getBuId()));
        }
        /** 业绩类别 精确 */
        if (!ObjectUtils.isEmpty(query.getAchieveType())) {
            list.add(qdo.achieveType.eq(query.getAchieveType()));
        }
        /** 业绩类别 精确 */
        if (!ObjectUtils.isEmpty(query.getAchieveTypes())) {
            list.add(qdo.achieveType.in(query.getAchieveTypes()));
        }
        /** 价值角色 精确 */
        if (!ObjectUtils.isEmpty(query.getValueRole())) {
            list.add(qdo.valueRole.eq(query.getValueRole()));
        }
        /** 部门归属人员ID 精确 */
        if (!ObjectUtils.isEmpty(query.getChargeResId())) {
            list.add(qdo.chargeResId.eq(query.getChargeResId()));
        }
        /** 比例 精确 */
        if (!ObjectUtils.isEmpty(query.getRatio())) {
            list.add(qdo.ratio.eq(query.getRatio()));
        }
        /** 金额 精确 */
        if (!ObjectUtils.isEmpty(query.getAmt())) {
            list.add(qdo.amt.eq(query.getAmt()));
        }
        /** 金额 精确 */
        if (!ObjectUtils.isEmpty(query.getAmt())) {
            list.add(qdo.amt.eq(query.getAmt()));
        }
        /** 主合同编号/名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getMainConCodeOrName())) {
            list.add(qdoConMain.code.like(SqlUtil.toSqlLikeString(query.getMainConCodeOrName()))
                    .or(qdoConMain.name.like(SqlUtil.toSqlLikeString(query.getMainConCodeOrName()))));
        }
        /** 子合同编号/名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getSubConCodeOrName())) {
            list.add(qdoConSub.code.like(SqlUtil.toSqlLikeString(query.getSubConCodeOrName()))
                    .or(qdoConSub.name.like(SqlUtil.toSqlLikeString(query.getSubConCodeOrName()))));
        }
        /** 签订年度 精确 */
        if (!ObjectUtils.isEmpty(query.getSignYear())) {
            list.add(qdoConSub.signDate.year().eq(query.getSignYear().intValue()));
        }
        /** 签单日期范围 区间*/
        if (!ObjectUtils.isEmpty(query.getSignDate())) {
            list.add(qdoConSub.signDate.between(query.getSignDate().get(0), query.getSignDate().get(1)));
        }
        /** 合同状态 精确 */
        if (!CollectionUtils.isEmpty(query.getStatusList()) && query.getStatusList().size() == 2) {
            if ("0".equals(query.getStatusList().get(0))) {
                list.add(qdoConSub.status.eq(query.getStatusList().get(1)));
            } else if ("1".equals(query.getStatusList().get(0))) {
                list.add(qdoConSub.status.notEqualsIgnoreCase(query.getStatusList().get(1)));
            }
        }
        /** 客户id 精确 */
        if (!ObjectUtils.isEmpty(query.getCustId())) {
            list.add(qdoConSub.custId.longValue().eq(query.getCustId()));
        }
        /**
         * 关于列表权限
         */
        if ((!CollectionUtils.isEmpty(query.getOrgIdsByPermission()) || (!CollectionUtils.isEmpty(query.getUserIdsByPermission())))) {
            list.add(qdo.chargeResId.in(query.getUserIdsByPermission()).or(qdo.buId.in(query.getOrgIdsByPermission())));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ConAchieveDVO queryByKey(Long id) {
        JPAQuery<ConAchieveDVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ConAchieveDVO> queryListDynamic(ConAchieveDQuery query) {
        JPAQuery<ConAchieveDVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ConAchieveDVO> queryPaging(ConAchieveDQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<ConAchieveDVO> jpaQuery = getJpaQueryWhere(query);
        List<ConAchieveDVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ConAchieveDVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ConAchieveDDO save(ConAchieveDDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ConAchieveDDO> saveAll(List<ConAchieveDDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ConAchieveDPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 业绩统计ID
        if (payload.getAchieveId() != null) {
            update.set(qdo.achieveId, payload.getAchieveId());
        }
        // 部门类别
        if (payload.getBuType() != null) {
            update.set(qdo.buType, payload.getBuType());
        }
        // 部门ID
        if (payload.getBuId() != null) {
            update.set(qdo.buId, payload.getBuId());
        }
        // 业绩类别
        if (payload.getAchieveType() != null) {
            update.set(qdo.achieveType, payload.getAchieveType());
        }
        // 价值角色
        if (payload.getValueRole() != null) {
            update.set(qdo.valueRole, payload.getValueRole());
        }
        // 部门归属人员ID
        if (payload.getChargeResId() != null) {
            update.set(qdo.chargeResId, payload.getChargeResId());
        }
        // 比例
        if (payload.getRatio() != null) {
            update.set(qdo.ratio, payload.getRatio());
        }
        // 金额
        if (payload.getAmt() != null) {
            update.set(qdo.amt, payload.getAmt());
        }
        // 金额
        if (payload.getAchievedIdV4() != null) {
            update.set(qdo.achievedIdV4, payload.getAchievedIdV4());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 业绩统计ID
            if (nullFields.contains("achieveId")) {
                update.setNull(qdo.achieveId);
            }
            // 部门类别
            if (nullFields.contains("buType")) {
                update.setNull(qdo.buType);
            }
            // 部门ID
            if (nullFields.contains("buId")) {
                update.setNull(qdo.buId);
            }
            // 业绩类别
            if (nullFields.contains("achieveType")) {
                update.setNull(qdo.achieveType);
            }
            // 价值角色
            if (nullFields.contains("valueRole")) {
                update.setNull(qdo.valueRole);
            }
            // 部门归属人员ID
            if (nullFields.contains("chargeResId")) {
                update.setNull(qdo.chargeResId);
            }
            // 比例
            if (nullFields.contains("ratio")) {
                update.setNull(qdo.ratio);
            }
            // 金额
            if (nullFields.contains("amt")) {
                update.setNull(qdo.amt);
            }
            // 金额
            if (nullFields.contains("achievedIdV4")) {
                update.setNull(qdo.achievedIdV4);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据业绩明细ID查询明细
     *
     * @param id
     * @return List<ConAchieveDVO>
     */
    public List<ConAchieveDVO> queryByAchieveId(Long id) {
        JPAQuery<ConAchieveDVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.achieveId.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.groupBy(qdo.id);
        return jpaQuery.fetch();
    }
}

