package com.elitesland.tw.tw5.server.prd.acc.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimSourceConfigPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimSourceConfigQuery;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimSourceConfigVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccReimSourceConfigDO;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccReimSourceConfigDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimSourceConfigRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 报销金额来源配置管理
 *
 * @author sunxw
 * @date 2023-12-04
 */
@Repository
@RequiredArgsConstructor
public class AccReimSourceConfigDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AccReimSourceConfigRepo repo;
    private final QAccReimSourceConfigDO qdo = QAccReimSourceConfigDO.accReimSourceConfigDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimSourceConfigVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AccReimSourceConfigVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 核算项目(对多个生效时，可以,分割)
                qdo.busAccItemCodes,
                // 配置名称
                qdo.configName,
                // 是否启用 0-否 1-是
                qdo.enableFlag,
                // 金额是否必须由计算器得出 0-否 1-是
                qdo.calcFlag,
                // 配置说明(富文本字段)
                qdo.configRemark,
                // 配置类型 INNER-内置制度 HALF-半配置化 ALL-全配置化
                qdo.configType
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimSourceConfigVO> getJpaQueryWhere(AccReimSourceConfigQuery query) {
        JPAQuery<AccReimSourceConfigVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AccReimSourceConfigQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AccReimSourceConfigQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 核算项目(对多个生效时，可以,分割) 精确 */
        if (!ObjectUtils.isEmpty(query.getBusAccItemCodes())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getBusAccItemCodes());
            list.add(qdo.busAccItemCodes.like(likeStr));
        }
        /** 配置名称 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigName())) {
            list.add(qdo.configName.eq(query.getConfigName()));
        }
        /** 是否启用 0-否 1-是 精确 */
        if (!ObjectUtils.isEmpty(query.getEnableFlag())) {
            list.add(qdo.enableFlag.eq(query.getEnableFlag()));
        }
        /** 金额是否必须由计算器得出 0-否 1-是 精确 */
        if (!ObjectUtils.isEmpty(query.getCalcFlag())) {
            list.add(qdo.calcFlag.eq(query.getCalcFlag()));
        }
        /** 配置说明(富文本字段) 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigRemark())) {
            list.add(qdo.configRemark.eq(query.getConfigRemark()));
        }
        /** 配置类型 INNER-内置制度 HALF-半配置化 ALL-全配置化 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigType())) {
            list.add(qdo.configType.eq(query.getConfigType()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AccReimSourceConfigVO queryByKey(Long id) {
        JPAQuery<AccReimSourceConfigVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimSourceConfigVO> queryListDynamic(AccReimSourceConfigQuery query) {
        JPAQuery<AccReimSourceConfigVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimSourceConfigVO> queryPaging(AccReimSourceConfigQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AccReimSourceConfigVO> jpaQuery = getJpaQueryWhere(query);
        List<AccReimSourceConfigVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimSourceConfigVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AccReimSourceConfigDO save(AccReimSourceConfigDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AccReimSourceConfigDO> saveAll(List<AccReimSourceConfigDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AccReimSourceConfigPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 核算项目(对多个生效时，可以,分割)
        if (payload.getBusAccItemCodes() != null) {
            update.set(qdo.busAccItemCodes, payload.getBusAccItemCodes());
        }
        // 配置名称
        if (payload.getConfigName() != null) {
            update.set(qdo.configName, payload.getConfigName());
        }
        // 是否启用 0-否 1-是
        if (payload.getEnableFlag() != null) {
            update.set(qdo.enableFlag, payload.getEnableFlag());
        }
        // 金额是否必须由计算器得出 0-否 1-是
        if (payload.getCalcFlag() != null) {
            update.set(qdo.calcFlag, payload.getCalcFlag());
        }
        // 配置说明(富文本字段)
        if (payload.getConfigRemark() != null) {
            update.set(qdo.configRemark, payload.getConfigRemark());
        }
        // 配置类型 INNER-内置制度 HALF-半配置化 ALL-全配置化
        if (payload.getConfigType() != null) {
            update.set(qdo.configType, payload.getConfigType());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 核算项目(对多个生效时，可以,分割)
            if (nullFields.contains("busAccItemIds")) {
                update.setNull(qdo.busAccItemCodes);
            }
            // 配置名称
            if (nullFields.contains("configName")) {
                update.setNull(qdo.configName);
            }
            // 是否启用 0-否 1-是
            if (nullFields.contains("enableFlag")) {
                update.setNull(qdo.enableFlag);
            }
            // 金额是否必须由计算器得出 0-否 1-是
            if (nullFields.contains("calcFlag")) {
                update.setNull(qdo.calcFlag);
            }
            // 配置说明(富文本字段)
            if (nullFields.contains("configRemark")) {
                update.setNull(qdo.configRemark);
            }
            // 配置类型 INNER-内置制度 HALF-半配置化 ALL-全配置化
            if (nullFields.contains("configType")) {
                update.setNull(qdo.configType);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

