package com.elitesland.tw.tw5.server.prd.acc.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimSourceRecordPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimSourceRecordQuery;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimSourceRecordVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccReimSourceRecordDO;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccReimSourceRecordDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimSourceRecordRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 报销金额来源记录管理
 *
 * @author sunxw
 * @date 2023-12-04
 */
@Repository
@RequiredArgsConstructor
public class AccReimSourceRecordDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AccReimSourceRecordRepo repo;
    private final QAccReimSourceRecordDO qdo = QAccReimSourceRecordDO.accReimSourceRecordDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimSourceRecordVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AccReimSourceRecordVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 报销主表ID
                qdo.reimId,
                // 报销明细ID
                qdo.reimDtlId,
                // 报销规则ID
                qdo.ruleId,
                // 报销内容详情(JSON)
                qdo.content
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimSourceRecordVO> getJpaQueryWhere(AccReimSourceRecordQuery query) {
        JPAQuery<AccReimSourceRecordVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AccReimSourceRecordQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AccReimSourceRecordQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 报销主表ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReimId())) {
            list.add(qdo.reimId.eq(query.getReimId()));
        }
        /** 报销明细ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReimDtlId())) {
            list.add(qdo.reimDtlId.eq(query.getReimDtlId()));
        }
        /** 报销规则ID 精确 */
        if (!ObjectUtils.isEmpty(query.getRuleId())) {
            list.add(qdo.ruleId.eq(query.getRuleId()));
        }
        /** 报销内容详情(JSON) 精确 */
        if (!ObjectUtils.isEmpty(query.getContent())) {
            list.add(qdo.content.eq(query.getContent()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AccReimSourceRecordVO queryByKey(Long id) {
        JPAQuery<AccReimSourceRecordVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimSourceRecordVO> queryListDynamic(AccReimSourceRecordQuery query) {
        JPAQuery<AccReimSourceRecordVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimSourceRecordVO> queryPaging(AccReimSourceRecordQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AccReimSourceRecordVO> jpaQuery = getJpaQueryWhere(query);
        List<AccReimSourceRecordVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimSourceRecordVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AccReimSourceRecordDO save(AccReimSourceRecordDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AccReimSourceRecordDO> saveAll(List<AccReimSourceRecordDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AccReimSourceRecordPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 报销主表ID
        if (payload.getReimId() != null) {
            update.set(qdo.reimId, payload.getReimId());
        }
        // 报销明细ID
        if (payload.getReimDtlId() != null) {
            update.set(qdo.reimDtlId, payload.getReimDtlId());
        }
        // 报销规则ID
        if (payload.getRuleId() != null) {
            update.set(qdo.ruleId, payload.getRuleId());
        }
        // 报销内容详情(JSON)
        if (payload.getContent() != null) {
            update.set(qdo.content, payload.getContent());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 报销主表ID
            if (nullFields.contains("reimId")) {
                update.setNull(qdo.reimId);
            }
            // 报销明细ID
            if (nullFields.contains("reimDtlId")) {
                update.setNull(qdo.reimDtlId);
            }
            // 报销规则ID
            if (nullFields.contains("ruleId")) {
                update.setNull(qdo.ruleId);
            }
            // 报销内容详情(JSON)
            if (nullFields.contains("content")) {
                update.setNull(qdo.content);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据报销单ID逻辑删除
     *
     * @param reimId
     * @return 删除的行数
     */
    public long delByReimId(Long reimId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.reimId.eq(reimId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

