package com.elitesland.tw.tw5.server.prd.acc.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccFinancialSubjectPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccFinancialSubjectQuery;
import com.elitesland.tw.tw5.api.prd.acc.service.AccFinancialSubjectService;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccFinancialSubjectVO;
import com.elitesland.tw.tw5.server.common.ExcelUtil;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.prd.acc.convert.AccFinancialSubjectConvert;
import com.elitesland.tw.tw5.server.prd.acc.dao.AccFinancialSubjectDAO;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccFinancialSubjectDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccFinancialSubjectRepo;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.InputStream;
import java.time.LocalDate;
import java.util.*;

/**
 * 会计科目管理
 *
 * @author carl
 * @date 2023-09-25
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class AccFinancialSubjectServiceImpl extends BaseServiceImpl implements AccFinancialSubjectService {

    private final AccFinancialSubjectRepo accFinancialSubjectRepo;
    private final AccFinancialSubjectDAO accFinancialSubjectDAO;
    private final ExcelUtil excelUtil;
    private final UdcUtil udcUtil;
    private final TransactionUtilService transactionUtilService;

    @Override
    public PagingVO<AccFinancialSubjectVO> queryPaging(AccFinancialSubjectQuery query) {
        return accFinancialSubjectDAO.queryPaging(query);
    }

    @Override
    public List<AccFinancialSubjectVO> queryListDynamic(AccFinancialSubjectQuery query) {
        return accFinancialSubjectDAO.queryListDynamic(query);
    }

    @Override
    public AccFinancialSubjectVO queryByKey(Long key) {
        return accFinancialSubjectDAO.queryByKey(key);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public AccFinancialSubjectVO insert(AccFinancialSubjectPayload payload) {
        checkData(payload);
        List<Long> ids = accFinancialSubjectDAO.queryByCodeAndName(payload.getAccCode(), payload.getAccName());
        if (ObjectUtils.isEmpty(ids)) {
            if (!StringUtils.hasText(payload.getAccStatus())) {
                payload.setAccStatus("ACTIVE");
            }
            payload.setAccLevel(1);
            //如果是,要判断上级
            if (!ObjectUtils.isEmpty(payload.getParentId())) {
                AccFinancialSubjectVO accFinancialSubjectVO = accFinancialSubjectDAO.queryByKey(payload.getParentId());
                payload.setAccLevel(accFinancialSubjectVO.getAccLevel() + 1);
            }
            AccFinancialSubjectDO entityDo = AccFinancialSubjectConvert.INSTANCE.toDo(payload);
            return AccFinancialSubjectConvert.INSTANCE.toVo(accFinancialSubjectRepo.save(entityDo));
        } else {
            throw TwException.error("", "科目编号和名称不可重复");
        }
    }

    /**
     * 数据校验
     *
     * @param payload
     */
    void checkData(AccFinancialSubjectPayload payload) {
        if (!StringUtils.hasText(payload.getAccCode()) || !StringUtils.hasText(payload.getAccName())) {
            throw TwException.error("", "科目编号和名称不可为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(AccFinancialSubjectPayload payload) {
        List<Long> ids = accFinancialSubjectDAO.queryByCodeAndName(payload.getAccCode(), payload.getAccName());
        if (ObjectUtils.isEmpty(ids) || (ids.size() == 1 && ids.get(0).equals(payload.getId()))) {
            payload.setAccLevel(1);
            //如果是,要判断上级
            if (!ObjectUtils.isEmpty(payload.getParentId())) {
                AccFinancialSubjectVO accFinancialSubjectVO = accFinancialSubjectDAO.queryByKey(payload.getParentId());
                payload.setAccLevel(accFinancialSubjectVO.getAccLevel() + 1);
            }
            long result = accFinancialSubjectDAO.updateByKeyDynamic(payload);
            return result;
        } else {
            throw TwException.error("", "科目编号和名称不可重复");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            accFinancialSubjectDAO.deleteSoft(keys);
        }
    }

    @Override
    public void downloadPlus(HttpServletResponse response, AccFinancialSubjectQuery query) {
        ClassPathResource classPathResource = new ClassPathResource("template/accFinancialSubject.xlsx");
        try {
            InputStream inputStream = classPathResource.getInputStream();
            Workbook workbook = WorkbookFactory.create(inputStream);
            XSSFSheet batchProjectSheet = (XSSFSheet) workbook.getSheet("科目数据");
            excelUtil.generateRangeList(batchProjectSheet, 4, 1, "LOV", 2, "C"); // 科目状态 数据验证
            excelUtil.generateRangeList(batchProjectSheet, 10, 1, "LOV", 2, "A"); // 汇总 数据验证
            excelUtil.generateRangeList(batchProjectSheet, 3, 1, "科目数据", 2, "C"); // 上级会计科目 数据验证
            String fileName = "会计科目数据-" + LocalDate.now();
            Boolean isHave = false;
            if (!StringUtils.hasText(query.getDownloadType()) || query.getDownloadType().equals("data")) {
                //数据导出
                List<AccFinancialSubjectVO> accFinancialSubjectVOS = accFinancialSubjectDAO.queryListDynamic(query);
                if (!ObjectUtils.isEmpty(accFinancialSubjectVOS)) {
                    accFinancialSubjectVOS = udcUtil.translateList(accFinancialSubjectVOS);
                    isHave = true;
                    int nextRow = 1;
                    for (AccFinancialSubjectVO accFinancialSubjectVO : accFinancialSubjectVOS) {
                        Row row = batchProjectSheet.createRow(nextRow);

                        Optional<AccFinancialSubjectVO> first = accFinancialSubjectVOS.stream().filter(vo -> vo.getId().equals(accFinancialSubjectVO.getParentId())).findFirst();
                        if (first.isPresent()) {
                            excelUtil.setCellValue(row, 3, first.get().getAccName());// 上级名称
                        }
                        excelUtil.setCellValue(row, 0, nextRow); // 序号
                        excelUtil.setCellValue(row, 1, accFinancialSubjectVO.getAccCode());// 编号
                        excelUtil.setCellValue(row, 2, accFinancialSubjectVO.getAccName());// 名称
                        excelUtil.setCellValue(row, 4, accFinancialSubjectVO.getAccStatusDesc());// 科目状态
                        excelUtil.setCellValue(row, 5, accFinancialSubjectVO.getAccType1());// 大类
                        excelUtil.setCellValue(row, 6, accFinancialSubjectVO.getAccType2());// 明细类1
                        excelUtil.setCellValue(row, 7, accFinancialSubjectVO.getAccType3());// 明细类2
                        excelUtil.setCellValue(row, 8, accFinancialSubjectVO.getDtlAcc());// 明细账

                        String sumFlag = "是";
                        if (accFinancialSubjectVO.getSumFlag() == null || accFinancialSubjectVO.getSumFlag() == 0) {
                            sumFlag = "否";
                        }
                        excelUtil.setCellValue(row, 9, sumFlag);// 汇总
                        excelUtil.setCellValue(row, 10, accFinancialSubjectVO.getLedgerType());// 子账类型
                        excelUtil.setCellValue(row, 11, accFinancialSubjectVO.getRemark());// 备注
                        nextRow++;
                    }
                }
            } else {
                //下载模板
                fileName = "会计科目导入模板-" + LocalDate.now();
            }

            if (!isHave) {
                excelUtil.setColumnFormulas(batchProjectSheet, 12, "VLOOKUP(D" + ExcelUtil.formulasReplace + ",B:C,1,FALSE)");//公海列表
                excelUtil.setColumnFormulas(batchProjectSheet, 13, "VLOOKUP(E" + ExcelUtil.formulasReplace + ",LOV!C:D,2,FALSE)");//市场活动
                excelUtil.setColumnFormulas(batchProjectSheet, 14, "VLOOKUP(J" + ExcelUtil.formulasReplace + ",LOV!A:B,2,FALSE)");
            }
            batchProjectSheet.setColumnHidden(12, true);
            batchProjectSheet.setColumnHidden(13, true);
            batchProjectSheet.setColumnHidden(14, true);

            ExcelUtil.writeResponse(response, fileName, workbook);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean batchImport(MultipartFile file) {

        if (file == null) {
            throw TwException.error("", "上传文件异常");
        }
        Workbook workbook = null;
        try {
            workbook = WorkbookFactory.create(file.getInputStream());
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.getMessage());
            throw TwException.error("", "文件解析异常");
        }
        Sheet sheet = workbook.getSheet("科目数据");
        if (sheet == null) {
            throw TwException.error("", "表结构错误");
        }
        int dataStartRow = 1;

        List<String> codeNames = new ArrayList<>();
        Map<String, String> map = new HashMap<>();
        List<AccFinancialSubjectPayload> accFinancialSubjectPayloads = new ArrayList<>();
        for (int i = dataStartRow; i <= sheet.getLastRowNum(); i++) {
            Row row = sheet.getRow(i);
            String accCode = ExcelUtil.getCellFormatValue(row.getCell(1));
            String accName = ExcelUtil.getCellFormatValue(row.getCell(2));
            if (!StringUtils.hasText(accCode) || !StringUtils.hasText(accName)) {
                break;
            }
            if (!StringUtils.hasText(map.get(accName))) {
                map.put(accName, accCode);
            }

            String codeNameStr = "," + accCode + "-" + accName + ",";
            if (codeNames.contains(codeNameStr)) {
                throw TwException.error("", "科目编号或名称不可重复:【" + codeNameStr + "】");
            } else {
                codeNames.add(codeNameStr);
            }
            String upperName = ExcelUtil.getCellFormatValue(row.getCell(3));
            String accType1 = ExcelUtil.getCellFormatValue(row.getCell(5));
            String accType2 = ExcelUtil.getCellFormatValue(row.getCell(6));
            String accType3 = ExcelUtil.getCellFormatValue(row.getCell(7));
            String dtlAcc = ExcelUtil.getCellFormatValue(row.getCell(8));//明细账
            String ledgerType = ExcelUtil.getCellFormatValue(row.getCell(10));//子账类型
            String remark = ExcelUtil.getCellFormatValue(row.getCell(11));//备注
            String upperCode = ExcelUtil.getCellFormatValue(row.getCell(12));//上级科目编号
            String accStatus = ExcelUtil.getCellFormatValue(row.getCell(13));//状态
            if (!StringUtils.hasText(accStatus)) {
                accStatus = "ACTIVE";
            }

            String sumFlag = ExcelUtil.getCellFormatValue(row.getCell(14));
            if (!StringUtils.hasText(sumFlag)) {
                sumFlag = "0";
            }
            AccFinancialSubjectPayload accFinancialSubjectPayload = new AccFinancialSubjectPayload();
            accFinancialSubjectPayload.setAccName(accName);
            accFinancialSubjectPayload.setAccCode(accCode);
            accFinancialSubjectPayload.setAccStatus(accStatus);
            accFinancialSubjectPayload.setUpperCode(upperCode);
            accFinancialSubjectPayload.setUpperName(upperName);
            accFinancialSubjectPayload.setAccType1(accType1);
            accFinancialSubjectPayload.setAccType2(accType2);
            accFinancialSubjectPayload.setAccType3(accType3);
            accFinancialSubjectPayload.setLedgerType(ledgerType);
            accFinancialSubjectPayload.setSumFlag(Integer.valueOf(sumFlag));
            accFinancialSubjectPayload.setRemark(remark);
            accFinancialSubjectPayload.setDtlAcc(dtlAcc);
            if (!StringUtils.hasText(upperName) && !StringUtils.hasText(upperCode)) {
                accFinancialSubjectPayload.setAccLevel(1);
            }

            accFinancialSubjectPayloads.add(accFinancialSubjectPayload);
        }
        List<AccFinancialSubjectDO> accFinancialSubjectDOs = new ArrayList<>();
        accFinancialSubjectPayloads.forEach(accFinancialSubjectPayload -> {
            String upperName = accFinancialSubjectPayload.getUpperName();
            String upperCode = accFinancialSubjectPayload.getUpperCode();
            if (StringUtils.hasText(upperName)) {
                if (!StringUtils.hasText(upperCode)) {
                    //根据上级名称获取编号
                    String s = map.get(upperName);
                    //判断上级编号是否等于自己编号
                    if (!s.equals(accFinancialSubjectPayload.getAccCode())) {
                        accFinancialSubjectPayload.setUpperCode(s);
                        //设置科目级别
                        setLevel(accFinancialSubjectPayloads, accFinancialSubjectPayload);
                    }
                }
            }
            accFinancialSubjectDOs.add(AccFinancialSubjectConvert.INSTANCE.toDo(accFinancialSubjectPayload));
        });

        List<AccFinancialSubjectDO> accFinancialSubjectDOS = accFinancialSubjectDAO.saveAll(accFinancialSubjectDOs);
        List<AccFinancialSubjectDO> updataDOs = new ArrayList<>();
        accFinancialSubjectDOS.forEach(accFinancialSubjectDO -> {
            if (StringUtils.hasText(accFinancialSubjectDO.getUpperCode())) {
                Optional<AccFinancialSubjectDO> first = accFinancialSubjectDOS.stream().filter(masPayload -> masPayload.getAccCode().equals(accFinancialSubjectDO.getUpperCode())).findFirst();
                if (first.isPresent()) {
                    AccFinancialSubjectDO updataDO = first.get();
                    accFinancialSubjectDO.setParentId(updataDO.getId());
                } else {
                    accFinancialSubjectDO.setUpperCode(null);
                }
                updataDOs.add(accFinancialSubjectDO);
            }
        });
        //开启事务执行修改
        transactionUtilService.executeWithRunnable(() -> {
            accFinancialSubjectDAO.saveAll(updataDOs);
        });
        return true;
    }

    /**
     * 为科目设计级别
     *
     * @param accFinancialSubjectPayloads
     * @param accFinancialSubjectPayload
     */
    void setLevel(List<AccFinancialSubjectPayload> accFinancialSubjectPayloads, AccFinancialSubjectPayload accFinancialSubjectPayload) {
        Optional<AccFinancialSubjectPayload> optional = accFinancialSubjectPayloads.stream().filter(masPayload -> masPayload.getAccCode().equals(accFinancialSubjectPayload.getUpperCode())).findFirst();
        if (optional.isPresent()) {
            AccFinancialSubjectPayload payload = optional.get();
            if (payload.getAccLevel() == null) {
                setLevel(accFinancialSubjectPayloads, payload);
            }
            accFinancialSubjectPayload.setAccLevel(payload.getAccLevel() + 1);
        } else {
            accFinancialSubjectPayload.setAccLevel(1);
        }
    }
}
